package pangea.hiagent.llm;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.model.ChatModel;
import org.springframework.ai.openai.OpenAiChatModel;
import org.springframework.ai.openai.api.OpenAiApi;
import org.springframework.ai.openai.OpenAiChatOptions;
import org.springframework.stereotype.Component;
import pangea.hiagent.model.LlmConfig;

/**
 * Hisense模型适配器
 * 实现与Hisense模型服务的对接，完全遵循OpenAI API规范
 */
@Slf4j
@Component
public class HisenseModelAdapter implements ModelAdapter {

    // Hisense模型服务默认配置

    @Override
    public ChatModel createChatModel(LlmConfig config) {
        if (!validateConfig(config)) {
            throw new IllegalArgumentException("无效的Hisense配置");
        }

        try {
            // 获取配置信息，直接使用配置中的值
            String baseUrl = config.getBaseUrl();
            String apiKey = config.getApiKey();
            String modelName = config.getModelName();

            // 确保baseUrl格式正确，移除可能存在的/chat/completions路径
            if (baseUrl.endsWith("/chat/completions")) {
                baseUrl = baseUrl.substring(0, baseUrl.length() - "/chat/completions".length());
            }
            
            // 移除可能存在的/v1后缀，因为OpenAiApi会自动添加/v1/chat/completions
            if (baseUrl.endsWith("/v1")) {
                baseUrl = baseUrl.substring(0, baseUrl.length() - "/v1".length());
            }

            log.info("创建Hisense ChatModel, baseUrl: {}, modelName: {}", baseUrl, modelName);

            // 使用Builder模式创建OpenAiApi实例
            OpenAiApi openAiApi = OpenAiApi.builder()
                    .apiKey(apiKey)
                    .baseUrl(baseUrl)
                    .build();

            // 使用Builder模式创建OpenAiChatModel实例
            return OpenAiChatModel.builder()
                    .openAiApi(openAiApi)
                    .defaultOptions(OpenAiChatOptions.builder()
                            .model(modelName)
                            .temperature(config.getTemperature())
                            .maxTokens(config.getMaxTokens())
                            .build())
                    .build();

        } catch (Exception e) {
            log.error("创建Hisense ChatModel失败: {}", e.getMessage(), e);
            throw new RuntimeException("创建Hisense ChatModel失败: " + e.getMessage(), e);
        }
    }

    @Override
    public String getProviderName() {
        return "hisense";
    }

    @Override
    public boolean validateConfig(LlmConfig config) {
        // Hisense适配器至少需要启用配置
        // API密钥和基础URL可以使用默认值
        return config != null && config.getEnabled();
    }
}