package pangea.hiagent.utils;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;

import java.util.Map;

/**
 * JSON工具类
 * 提供JSON序列化和反序列化功能
 */
@Slf4j
public class JsonUtils {
    
    private static final ObjectMapper objectMapper = new ObjectMapper();
    
    /**
     * 将对象转换为JSON字符串
     * 
     * @param obj 待转换的对象
     * @return JSON字符串
     */
    public static String toJson(Object obj) {
        try {
            if (obj == null) {
                log.warn("尝试将null对象转换为JSON");
                return null;
            }
            String json = objectMapper.writeValueAsString(obj);
            log.debug("对象转JSON成功，对象类型: {}, JSON长度: {}", obj.getClass().getSimpleName(), json.length());
            return json;
        } catch (JsonProcessingException e) {
            log.error("对象转JSON失败", e);
            return null;
        }
    }
    
    /**
     * 将JSON字符串转换为指定类型的对象
     * 
     * @param json JSON字符串
     * @param clazz 目标类型
     * @param <T> 泛型
     * @return 转换后的对象
     */
    public static <T> T fromJson(String json, Class<T> clazz) {

        if (clazz == null) {
            log.warn("目标类型不能为null");
            return null;
        }
        
        try {
            if (json == null || json.isEmpty()) {
                log.warn("尝试将空JSON字符串转换为对象");
                return null;
            }
            
            T obj = objectMapper.readValue(json, clazz);
            log.debug("JSON转对象成功，目标类型: {}", clazz.getSimpleName());
            return obj;
        } catch (Exception e) {
            log.error("JSON转对象失败，JSON: {}, 目标类型: {}", json, clazz.getSimpleName(), e);
            return null;
        }
    }
    
    /**
     * 将JSON字符串转换为Map对象
     * 
     * @param json JSON字符串
     * @return Map对象
     */
    public static Map<String, Object> fromJsonToMap(String json) {
        try {
            return objectMapper.readValue(json, new TypeReference<Map<String, Object>>() {});
        } catch (Exception e) {
            log.error("JSON转Map失败", e);
            return null;
        }
    }
}