package pangea.hiagent.service;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import pangea.hiagent.model.TimerExecutionHistory;
import pangea.hiagent.repository.TimerExecutionHistoryRepository;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;

/**
 * 执行历史清理服务
 * 定期清理过期的执行历史记录
 */
@Slf4j
@Service
public class HistoryCleanupService {

    private final TimerExecutionHistoryRepository timerExecutionHistoryRepository;

    public HistoryCleanupService(TimerExecutionHistoryRepository timerExecutionHistoryRepository) {
        this.timerExecutionHistoryRepository = timerExecutionHistoryRepository;
    }

    /**
     * 清理过期的执行历史记录
     * 每天凌晨2点执行一次，清理30天前的记录
     */
    @Scheduled(cron = "0 0 2 * * ?")
    @Transactional
    public void cleanupOldHistory() {
        log.info("开始清理过期执行历史记录");
        
        // 计算30天前的时间
        LocalDateTime cutoffTime = LocalDateTime.now().minus(30, ChronoUnit.DAYS);
        
        // 构建查询条件：执行时间小于30天前
        LambdaQueryWrapper<TimerExecutionHistory> wrapper = new LambdaQueryWrapper<>();
        wrapper.lt(TimerExecutionHistory::getExecutionTime, cutoffTime);
        
        // 清理30天前的执行历史记录
        int deletedCount = timerExecutionHistoryRepository.delete(wrapper);
        
        log.info("清理完成，共删除 {} 条过期执行历史记录", deletedCount);
    }
}