package pangea.hiagent.web.service;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import pangea.hiagent.auth.AuthenticationStrategy;
import pangea.hiagent.model.AuthMode;
import pangea.hiagent.model.OAuth2Provider;
import pangea.hiagent.web.repository.OAuth2ProviderRepository;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 统一认证服务
 * 作为认证策略的门面，支持多种认证方式的协调调度
 */
@Slf4j
@Service
public class UnifiedAuthService {
    
    private final List<AuthenticationStrategy> authenticationStrategies;
    private final AuthConfigService authConfigService;
    private final OAuth2ProviderRepository oauth2ProviderRepository;
    
    public UnifiedAuthService(List<AuthenticationStrategy> authenticationStrategies,
                             AuthConfigService authConfigService,
                             OAuth2ProviderRepository oauth2ProviderRepository) {
        this.authenticationStrategies = authenticationStrategies;
        this.authConfigService = authConfigService;
        this.oauth2ProviderRepository = oauth2ProviderRepository;
    }
    
    /**
     * 执行统一认证
     * @param authMode 认证模式（如 "local"、"oauth2_auth_code" 等）
     * @param credentials 认证凭证，结构根据不同认证模式而异
     * @return JWT Token
     */
    public String authenticate(String authMode, Map<String, Object> credentials) {
        log.info("执行统一认证: authMode={}", authMode);
        
        // 查找支持该认证模式的策略
        AuthenticationStrategy strategy = findStrategy(authMode);
        
        if (strategy == null) {
            log.error("未找到支持的认证策略: authMode={}", authMode);
            throw new RuntimeException("不支持的认证模式: " + authMode);
        }
        
        try {
            // 验证认证模式是否启用
            if (!authConfigService.isAuthModeEnabled(authMode)) {
                log.warn("认证模式未启用: authMode={}", authMode);
                throw new RuntimeException("该认证模式未启用");
            }
            
            // 执行认证策略
            String token = strategy.authenticate(credentials);
            log.info("统一认证成功: authMode={}", authMode);
            return token;
            
        } catch (Exception e) {
            log.error("统一认证失败: authMode={}, 错误堆栈: ", authMode, e);
            throw e;
        }
    }
    
    /**
     * 执行本地用户名/密码认证
     * @param username 用户名
     * @param password 密码
     * @return JWT Token
     */
    public String loginWithLocal(String username, String password) {
        Map<String, Object> credentials = new HashMap<>();
        credentials.put("username", username);
        credentials.put("password", password);
        
        return authenticate(AuthMode.LOCAL.getCode(), credentials);
    }
    
    /**
     * 执行 OAuth2 授权码认证
     * @param authorizationCode 授权码
     * @param providerName OAuth2 提供者名称
     * @return JWT Token
     */
    public String loginWithOAuth2(String authorizationCode, String providerName) {
        Map<String, Object> credentials = new HashMap<>();
        credentials.put("authorizationCode", authorizationCode);
        credentials.put("providerName", providerName);
        
        return authenticate(AuthMode.OAUTH2_AUTHORIZATION_CODE.getCode(), credentials);
    }
    
    /**
     * 验证令牌
     */
    public boolean verifyToken(String token) {
        try {
            for (AuthenticationStrategy strategy : authenticationStrategies) {
                if (strategy.verify(token)) {
                    return true;
                }
            }
            return false;
        } catch (Exception e) {
            log.error("令牌验证过程中出错: 错误堆栈: ", e);
            return false;
        }
    }
    
    /**
     * 查找支持该认证模式的策略
     */
    private AuthenticationStrategy findStrategy(String authMode) {
        for (AuthenticationStrategy strategy : authenticationStrategies) {
            if (strategy.supports(authMode)) {
                log.debug("找到支持的认证策略: authMode={}, strategyName={}", authMode, strategy.getName());
                return strategy;
            }
        }
        return null;
    }
    
    /**
     * 获取所有可用的认证模式
     */
    public Map<String, Object> getAvailableAuthModes() {
        Map<String, Object> modes = new HashMap<>();
        for (AuthMode mode : AuthMode.values()) {
            if (authConfigService.isAuthModeEnabled(mode.getCode())) {
                Map<String, String> modeInfo = new HashMap<>();
                modeInfo.put("code", mode.getCode());
                modeInfo.put("description", mode.getDescription());
                modes.put(mode.getCode(), modeInfo);
            }
        }
        return modes;
    }
    
    /**
     * 构造 OAuth2 授权 URL
     * @param providerName OAuth2 提供者名称
     * @return 授权 URL
     */
    public String buildOAuth2AuthorizationUrl(String providerName) {
        log.debug("构造 OAuth2 授权 URL: providerName={}", providerName);
        
        LambdaQueryWrapper<OAuth2Provider> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(OAuth2Provider::getProviderName, providerName)
               .eq(OAuth2Provider::getEnabled, 1);
        OAuth2Provider provider = oauth2ProviderRepository.selectOne(wrapper);
        
        if (provider == null) {
            log.error("未找到配置的 OAuth2 提供者: {}", providerName);
            throw new RuntimeException("未找到配置的 OAuth2 提供者");
        }
        
        // 构造授权 URL
        StringBuilder authUrlBuilder = new StringBuilder(provider.getAuthorizeUrl());
        authUrlBuilder.append("?client_id=").append(provider.getClientId());
        authUrlBuilder.append("&redirect_uri=").append(provider.getRedirectUri());
        authUrlBuilder.append("&response_type=code");
        authUrlBuilder.append("&scope=").append(provider.getScope());
        authUrlBuilder.append("&state=").append(java.util.UUID.randomUUID().toString());
        
        String authUrl = authUrlBuilder.toString();
        log.debug("OAuth2 授权 URL 构造完成: {}", authUrl);
        
        return authUrl;
    }
}