/**
 * Token工具类
 * 提供JWT Token处理相关的实用函数
 */
export class TokenUtils {
  /**
   * 检查Token是否有效
   * @param token JWT Token
   * @returns 包含有效性、过期状态、过期时间和剩余分钟数的对象
   */
  static validateToken(token: string | null): {isValid: boolean, isExpired: boolean, expiresAt: Date | null, minutesLeft: number | null} {
    if (!token) {
      return {isValid: false, isExpired: true, expiresAt: null, minutesLeft: null};
    }
    
    try {
      const payload = JSON.parse(atob(token.split('.')[1]));
      const expTime = payload.exp * 1000;
      const expiresAt = new Date(expTime);
      const now = Date.now();
      const isExpired = expTime < now;
      const minutesLeft = isExpired ? 0 : (expTime - now) / 1000 / 60;
      
      return {
        isValid: !isExpired,
        isExpired,
        expiresAt,
        minutesLeft
      };
    } catch (e) {
      return {isValid: false, isExpired: true, expiresAt: null, minutesLeft: null};
    }
  }
  
  /**
   * 检查Token是否即将过期（默认30分钟内）
   * @param token JWT Token
   * @param thresholdMinutes 阈值分钟数
   * @returns 如果Token即将过期则返回true，否则返回false
   */
  static isTokenExpiringSoon(token: string | null, thresholdMinutes: number = 30): boolean {
    const validation = this.validateToken(token);
    if (!validation.isValid || validation.minutesLeft === null) {
      return true; // 如果Token无效或已过期，视为即将过期
    }
    
    return validation.minutesLeft < thresholdMinutes;
  }
  
  /**
   * 获取Token中的用户ID
   * @param token JWT Token
   * @returns 用户ID或null
   */
  static getUserIdFromToken(token: string | null): string | null {
    if (!token) {
      return null;
    }
    
    try {
      const payload = JSON.parse(atob(token.split('.')[1]));
      return payload.userId || null;
    } catch (e) {
      return null;
    }
  }
}