package pangea.hiagent.web.controller;

import lombok.extern.slf4j.Slf4j;
import pangea.hiagent.agent.service.ExceptionMonitoringService;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.Map;

/**
 * 异常监控控制器
 * 提供异常监控信息的查询接口
 */
@Slf4j
@RestController
@RequestMapping("/api/monitoring")
public class ExceptionMonitoringController {
    
    @Autowired
    private ExceptionMonitoringService exceptionMonitoringService;
    
    /**
     * 获取异常统计信息
     * 
     * @return 异常统计信息
     */
    @GetMapping("/exceptions/statistics")
    public ResponseEntity<Map<String, Long>> getExceptionStatistics() {
        try {
            Map<String, Long> statistics = exceptionMonitoringService.getExceptionStatistics();
            return ResponseEntity.ok(statistics);
        } catch (Exception e) {
            log.error("获取异常统计信息失败", e);
            return ResponseEntity.internalServerError().build();
        }
    }
    
    /**
     * 获取指定类型的异常计数
     * 
     * @param exceptionType 异常类型
     * @return 异常计数
     */
    @GetMapping("/exceptions/count/{exceptionType}")
    public ResponseEntity<Long> getExceptionCount(@PathVariable String exceptionType) {
        try {
            long count = exceptionMonitoringService.getExceptionCount(exceptionType);
            return ResponseEntity.ok(count);
        } catch (Exception e) {
            log.error("获取异常计数失败", e);
            return ResponseEntity.internalServerError().build();
        }
    }
    
    /**
     * 获取所有异常详细信息
     * 
     * @return 异常详细信息
     */
    @GetMapping("/exceptions/details")
    public ResponseEntity<Map<String, String>> getExceptionDetails() {
        try {
            Map<String, String> details = exceptionMonitoringService.getExceptionDetails();
            return ResponseEntity.ok(details);
        } catch (Exception e) {
            log.error("获取异常详细信息失败", e);
            return ResponseEntity.internalServerError().build();
        }
    }
    
    /**
     * 清空异常统计信息
     * 
     * @return 操作结果
     */
    @DeleteMapping("/exceptions/statistics")
    public ResponseEntity<Void> clearStatistics() {
        try {
            exceptionMonitoringService.clearStatistics();
            return ResponseEntity.ok().build();
        } catch (Exception e) {
            log.error("清空异常统计信息失败", e);
            return ResponseEntity.internalServerError().build();
        }
    }
    
    /**
     * 清空异常详细信息
     * 
     * @return 操作结果
     */
    @DeleteMapping("/exceptions/details")
    public ResponseEntity<Void> clearDetails() {
        try {
            exceptionMonitoringService.clearDetails();
            return ResponseEntity.ok().build();
        } catch (Exception e) {
            log.error("清空异常详细信息失败", e);
            return ResponseEntity.internalServerError().build();
        }
    }
    
    /**
     * 清空所有异常信息
     * 
     * @return 操作结果
     */
    @DeleteMapping("/exceptions/all")
    public ResponseEntity<Void> clearAll() {
        try {
            exceptionMonitoringService.clearAll();
            return ResponseEntity.ok().build();
        } catch (Exception e) {
            log.error("清空所有异常信息失败", e);
            return ResponseEntity.internalServerError().build();
        }
    }
}