package pangea.hiagent.common.utils;

import lombok.extern.slf4j.Slf4j;

/**
 * 日志工具类
 * 提供统一的日志记录方法，减少重复代码
 */
@Slf4j
public class LogUtils {
    
    /**
     * 记录跟踪日志
     * 
     * @param message 日志消息
     * @param params 参数
     */
    public static void trace(String message, Object... params) {
        if (log.isTraceEnabled()) {
            log.trace(message, params);
        }
    }
    
    /**
     * 记录调试日志
     * 
     * @param message 日志消息
     * @param params 参数
     */
    public static void debug(String message, Object... params) {
        if (log.isDebugEnabled()) {
            log.debug(message, params);
        }
    }
    
    /**
     * 记录信息日志
     * 
     * @param message 日志消息
     * @param params 参数
     */
    public static void info(String message, Object... params) {
        log.info(message, params);
    }
    
    /**
     * 记录警告日志
     * 
     * @param message 日志消息
     * @param params 参数
     */
    public static void warn(String message, Object... params) {
        log.warn(message, params);
    }
    
    /**
     * 记录错误日志
     * 
     * @param message 日志消息
     * @param throwable 异常对象
     */
    public static void error(String message, Throwable throwable) {
        log.error(message, throwable);
    }
    
    /**
     * 记录错误日志
     * 
     * @param message 日志消息
     * @param params 参数
     */
    public static void error(String message, Object... params) {
        log.error(message, params);
    }
    
    /**
     * 记录错误日志
     * 
     * @param message 日志消息
     * @param throwable 异常对象
     * @param params 参数
     */
    public static void error(String message, Throwable throwable, Object... params) {
        log.error(message, throwable, params);
    }
    
    /**
     * 记录带前缀的信息日志
     * 
     * @param prefix 前缀
     * @param message 日志消息
     * @param params 参数
     */
    public static void infoWithPrefix(String prefix, String message, Object... params) {
        log.info("[{}] {}", prefix, String.format(message, params));
    }
    
    /**
     * 记录带前缀的错误日志
     * 
     * @param prefix 前缀
     * @param message 日志消息
     * @param throwable 异常对象
     */
    public static void errorWithPrefix(String prefix, String message, Throwable throwable) {
        log.error("[{}] {}", prefix, message, throwable);
    }
    
    /**
     * 记录带前缀的警告日志
     * 
     * @param prefix 前缀
     * @param message 日志消息
     * @param params 参数
     */
    public static void warnWithPrefix(String prefix, String message, Object... params) {
        log.warn("[{}] {}", prefix, String.format(message, params));
    }
    
    /**
     * 记录方法进入日志
     * 
     * @param methodName 方法名
     * @param params 参数
     */
    public static void enterMethod(String methodName, Object... params) {
        if (log.isDebugEnabled()) {
            StringBuilder sb = new StringBuilder();
            sb.append("Entering method: ").append(methodName);
            if (params.length > 0) {
                sb.append(" with params: ");
                for (int i = 0; i < params.length; i++) {
                    if (i > 0) sb.append(", ");
                    sb.append(params[i]);
                }
            }
            log.debug(sb.toString());
        }
    }
    
    /**
     * 记录方法退出日志
     * 
     * @param methodName 方法名
     * @param result 返回结果
     */
    public static void exitMethod(String methodName, Object result) {
        if (log.isDebugEnabled()) {
            log.debug("Exiting method: {} with result: {}", methodName, result);
        }
    }
}