package pangea.hiagent.web.service;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import pangea.hiagent.model.AuthMode;

import java.util.HashMap;
import java.util.Map;

/**
 * 认证配置管理服务
 * 管理各种认证模式的启用/禁用状态和配置信息
 */
@Slf4j
@Service
public class AuthConfigService {
    
    // 本地缓存的认证模式配置
    private final Map<String, Boolean> authModeEnabledCache = new HashMap<>();
    private final Map<String, Map<String, Object>> authModeConfigCache = new HashMap<>();
    
    public AuthConfigService() {
        // 初始化默认配置
        initializeDefaultConfigs();
    }
    
    /**
     * 初始化默认配置
     */
    private void initializeDefaultConfigs() {
        // 本地认证模式默认启用
        authModeEnabledCache.put(AuthMode.LOCAL.getCode(), true);
        
        // OAuth2 授权码模式默认启用（但需要配置提供者）
        authModeEnabledCache.put(AuthMode.OAUTH2_AUTHORIZATION_CODE.getCode(), true);
        
        // 其他模式默认禁用
        authModeEnabledCache.put(AuthMode.OAUTH2_IMPLICIT.getCode(), false);
        authModeEnabledCache.put(AuthMode.OAUTH2_CLIENT_CREDENTIALS.getCode(), false);
        authModeEnabledCache.put(AuthMode.LDAP.getCode(), false);
        authModeEnabledCache.put(AuthMode.SAML.getCode(), false);
        
        log.info("认证配置初始化完成");
    }
    
    /**
     * 检查认证模式是否启用
     */
    public boolean isAuthModeEnabled(String authMode) {
        Boolean enabled = authModeEnabledCache.get(authMode);
        return enabled != null && enabled;
    }
    
    /**
     * 启用认证模式
     */
    public void enableAuthMode(String authMode) {
        authModeEnabledCache.put(authMode, true);
        log.info("启用认证模式: {}", authMode);
    }
    
    /**
     * 禁用认证模式
     */
    public void disableAuthMode(String authMode) {
        authModeEnabledCache.put(authMode, false);
        log.info("禁用认证模式: {}", authMode);
    }
    
    /**
     * 获取认证模式的配置
     */
    public Map<String, Object> getAuthModeConfig(String authMode) {
        return authModeConfigCache.getOrDefault(authMode, new HashMap<>());
    }
    
    /**
     * 设置认证模式的配置
     */
    public void setAuthModeConfig(String authMode, Map<String, Object> config) {
        authModeConfigCache.put(authMode, config);
        log.info("更新认证模式配置: {}", authMode);
    }
    
    /**
     * 获取所有启用的认证模式
     */
    public Map<String, Boolean> getAllAuthModes() {
        return new HashMap<>(authModeEnabledCache);
    }
}
