package pangea.hiagent.tool.impl;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.stereotype.Component;
import pangea.hiagent.rag.RagService;
import pangea.hiagent.web.service.AgentService;
import pangea.hiagent.model.Agent;

import java.util.List;

/**
 * 技术文档检索工具
 * 用于检索和查询技术文档内容
 */
@Slf4j
@Component
public class TechnicalDocumentationRetrievalTool {
    
    private final RagService ragService;
    private final AgentService agentService;
    
    public TechnicalDocumentationRetrievalTool(RagService ragService, AgentService agentService) {
        this.ragService = ragService;
        this.agentService = agentService;
    }
    
    /**
     * 检索技术文档
     * @param query 查询关键词
     * @param maxResults 最大返回结果数
     * @return 检索到的技术文档内容
     */
    @Tool(description = "根据关键词检索技术文档，返回最相关的内容片段")
    public String searchTechnicalDocumentation(String query, Integer maxResults) {
        log.debug("开始检索技术文档: {}, 最大结果数: {}", query, maxResults);
        
        try {
            if (query == null || query.trim().isEmpty()) {
                log.warn("查询关键词不能为空");
                return "错误：查询关键词不能为空";
            }
            
            // 设置默认最大结果数
            if (maxResults == null || maxResults <= 0) {
                maxResults = 5;
            }
            
            // 获取技术支持Agent
            Agent techSupportAgent = getTechSupportAgent();
            if (techSupportAgent == null) {
                log.error("未找到技术支持Agent");
                return "错误：未找到技术支持Agent配置";
            }
            
            // 使用RAG服务检索文档
            List<org.springframework.ai.document.Document> documents = 
                ragService.searchDocumentsByAgent(techSupportAgent, query);
            
            // 限制返回结果数量
            if (documents.size() > maxResults) {
                documents = documents.subList(0, maxResults);
            }
            
            // 格式化结果
            StringBuilder result = new StringBuilder();
            result.append("技术文档检索结果：\n");
            result.append("查询关键词: ").append(query).append("\n");
            result.append("找到 ").append(documents.size()).append(" 个相关文档片段\n\n");
            
            for (int i = 0; i < documents.size(); i++) {
                org.springframework.ai.document.Document doc = documents.get(i);
                result.append("文档 ").append(i + 1).append(":\n");
                result.append(doc.getText()).append("\n\n");
            }
            
            log.info("技术文档检索完成，找到 {} 个结果", documents.size());
            return result.toString();
        } catch (Exception e) {
            log.error("检索技术文档时发生错误: {}", e.getMessage(), e);
            return "检索技术文档时发生错误: " + e.getMessage();
        }
    }
    
    /**
     * 获取技术支持Agent
     * @return 技术支持Agent对象
     */
    private Agent getTechSupportAgent() {
        try {
            // 在实际应用中，这里应该通过某种方式获取技术支持Agent
            // 例如通过AgentService查询特定名称的Agent
            List<Agent> agents = agentService.listAgents();
            for (Agent agent : agents) {
                if ("技术支持".equals(agent.getName())) {
                    return agent;
                }
            }
        } catch (Exception e) {
            log.error("获取技术支持Agent时发生错误: {}", e.getMessage(), e);
        }
        return null;
    }
}