package pangea.hiagent.workpanel;

import pangea.hiagent.dto.WorkPanelEvent;
import java.util.List;
import java.util.function.Consumer;

/**
 * 工作面板数据收集器接口
 * 用于采集Agent执行过程中的各类数据（思考过程、工具调用等）
 */
public interface IWorkPanelDataCollector {
    
    /**
     * 记录思考过程
     * @param content 思考内容
     * @param thinkingType 思考类型（分析、规划、执行等）
     */
    void recordThinking(String content, String thinkingType);
    
    /**
     * 记录工具调用开始
     * @param toolName 工具名称
     * @param toolAction 工具执行的方法
     * @param input 工具输入参数
     */
    void recordToolCallStart(String toolName, String toolAction, Object input);
    
    /**
     * 记录工具调用完成
     * @param toolName 工具名称
     * @param output 工具输出结果
     * @param status 执行状态（success/failure）
     */
    void recordToolCallComplete(String toolName, Object output, String status);
    
    /**
     * 记录工具调用完成（带执行时间）
     * @param toolName 工具名称
     * @param output 工具输出结果
     * @param status 执行状态（success/failure）
     * @param executionTime 执行时间（毫秒）
     */
    void recordToolCallComplete(String toolName, Object output, String status, Long executionTime);
    
    /**
     * 记录工具调用失败
     * @param toolName 工具名称
     * @param errorMessage 错误信息
     */
    void recordToolCallError(String toolName, String errorMessage);
    
    /**
     * 记录日志
     * @param message 日志消息
     * @param level 日志级别（info/warn/error/debug）
     */
    void recordLog(String message, String level);
    
    /**
     * 记录embed嵌入事件
     * @param url 嵌入资源URL（可选）
     * @param type MIME类型
     * @param title 嵌入标题
     * @param htmlContent HTML内容（可选）
     */
    void recordEmbed(String url, String type, String title, String htmlContent);
    
    /**
     * 获取所有收集的事件
     */
    List<WorkPanelEvent> getEvents();
    
    /**
     * 订阅事件（用于实时推送）
     * @param consumer 事件处理函数
     */
    void subscribe(Consumer<WorkPanelEvent> consumer);
    
    /**
     * 清空所有事件
     */
    void clear();
    
    /**
     * 获取最后一个工具调用事件
     */
    WorkPanelEvent getLastToolCall();
}