package pangea.hiagent.tools;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.stereotype.Component;

/**
 * 创作风格参考工具
 * 用于提供各种写作风格的参考和指导
 */
@Slf4j
@Component
public class WritingStyleReferenceTool {
    
    public WritingStyleReferenceTool() {
        // 默认构造器
    }
    
    /**
     * 获取写作风格参考
     * @param styleType 风格类型（如正式、幽默、亲切等）
     * @param contentType 内容类型（如商务邮件、社交媒体、技术文档等）
     * @return 写作风格参考信息
     */
    @Tool(description = "根据风格类型和内容类型提供写作风格参考和指导")
    public String getWritingStyleReference(String styleType, String contentType) {
        log.debug("开始获取写作风格参考: 风格类型={}, 内容类型={}", styleType, contentType);
        
        try {
            if (styleType == null || styleType.trim().isEmpty()) {
                log.warn("风格类型不能为空");
                return "错误：风格类型不能为空";
            }
            
            if (contentType == null || contentType.trim().isEmpty()) {
                log.warn("内容类型不能为空");
                return "错误：内容类型不能为空";
            }
            
            // 生成写作风格参考
            String reference = generateWritingStyleReference(styleType, contentType);
            
            log.info("写作风格参考生成完成: 风格类型={}, 内容类型={}", styleType, contentType);
            return reference;
        } catch (Exception e) {
            log.error("获取写作风格参考时发生错误: {}", e.getMessage(), e);
            return "获取写作风格参考时发生错误: " + e.getMessage();
        }
    }
    
    /**
     * 生成写作风格参考
     * @param styleType 风格类型
     * @param contentType 内容类型
     * @return 写作风格参考信息
     */
    private String generateWritingStyleReference(String styleType, String contentType) {
        StringBuilder reference = new StringBuilder();
        reference.append("写作风格参考:\n\n");
        reference.append("风格类型: ").append(styleType).append("\n");
        reference.append("内容类型: ").append(contentType).append("\n\n");
        
        // 根据风格类型和内容类型提供参考
        switch (styleType.toLowerCase()) {
            case "正式":
                reference.append("正式风格特点:\n");
                reference.append("- 使用完整句子和标准语法\n");
                reference.append("- 避免缩写和俚语\n");
                reference.append("- 使用专业术语和正式词汇\n");
                reference.append("- 保持客观和中立的语调\n\n");
                break;
            case "幽默":
                reference.append("幽默风格特点:\n");
                reference.append("- 使用轻松愉快的语言\n");
                reference.append("- 适当使用比喻和双关语\n");
                reference.append("- 创造轻松的氛围\n");
                reference.append("- 注意幽默的适度性\n\n");
                break;
            case "亲切":
                reference.append("亲切风格特点:\n");
                reference.append("- 使用第二人称\"你\"\n");
                reference.append("- 采用对话式的表达\n");
                reference.append("- 表达关心和理解\n");
                reference.append("- 使用温暖友好的词汇\n\n");
                break;
            case "简洁":
                reference.append("简洁风格特点:\n");
                reference.append("- 使用短句和简单结构\n");
                reference.append("- 删除不必要的修饰词\n");
                reference.append("- 直接表达核心观点\n");
                reference.append("- 避免冗余信息\n\n");
                break;
            default:
                reference.append("通用风格特点:\n");
                reference.append("- 根据目标受众调整语言\n");
                reference.append("- 保持一致的语调\n");
                reference.append("- 注意段落结构和逻辑\n");
                reference.append("- 确保内容清晰易懂\n\n");
                break;
        }
        
        // 根据内容类型提供具体建议
        switch (contentType.toLowerCase()) {
            case "商务邮件":
                reference.append("商务邮件写作建议:\n");
                reference.append("- 明确的主题行\n");
                reference.append("- 专业的称呼和结尾\n");
                reference.append("- 清晰的行动呼吁\n");
                reference.append("- 适当的附件提醒\n\n");
                break;
            case "社交媒体":
                reference.append("社交媒体写作建议:\n");
                reference.append("- 使用吸引人的开头\n");
                reference.append("- 适当使用表情符号\n");
                reference.append("- 包含相关的标签\n");
                reference.append("- 鼓励互动和分享\n\n");
                break;
            case "技术文档":
                reference.append("技术文档写作建议:\n");
                reference.append("- 使用精确的技术术语\n");
                reference.append("- 提供清晰的步骤说明\n");
                reference.append("- 包含示例和截图\n");
                reference.append("- 注意版本和兼容性信息\n\n");
                break;
            case "营销文案":
                reference.append("营销文案写作建议:\n");
                reference.append("- 突出产品卖点\n");
                reference.append("- 使用感性的语言\n");
                reference.append("- 创造紧迫感\n");
                reference.append("- 包含明确的购买引导\n\n");
                break;
            default:
                reference.append("通用内容写作建议:\n");
                reference.append("- 明确写作目的\n");
                reference.append("- 了解目标受众\n");
                reference.append("- 保持内容结构清晰\n");
                reference.append("- 注意语言的准确性\n\n");
                break;
        }
        
        reference.append("写作技巧:\n");
        reference.append("- 开头吸引注意力\n");
        reference.append("- 中间内容条理清晰\n");
        reference.append("- 结尾有力且印象深刻\n");
        reference.append("- 多次审阅和修改\n");
        
        return reference.toString();
    }
}