package pangea.hiagent.memory;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.memory.ChatMemory;
import org.springframework.ai.chat.messages.Message;
import org.springframework.stereotype.Component;

import jakarta.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 基于Caffeine的ChatMemory实现
 * 提供内存级的对话历史存储功能
 */
@Slf4j
@Component
public class CaffeineChatMemory implements ChatMemory {
    
    private Cache<String, List<Message>> cache;
    
    @PostConstruct
    public void init() {
        // 初始化Caffeine缓存
        cache = Caffeine.newBuilder()
                .maximumSize(10000) // 最大缓存条目数
                .expireAfterWrite(24, TimeUnit.HOURS) // 写入后24小时过期
                .recordStats() // 记录缓存统计信息
                .build();
        
        log.info("CaffeineChatMemory初始化完成");
    }
    
    @Override
    public void add(String conversationId, List<Message> messages) {
        try {
            // 获取现有的消息列表
            List<Message> existingMessages = get(conversationId, Integer.MAX_VALUE);
            
            // 添加新消息
            existingMessages.addAll(messages);
            
            // 保存到缓存
            cache.put(conversationId, existingMessages);
            
            log.debug("成功将{}条消息添加到会话{}", messages.size(), conversationId);
        } catch (Exception e) {
            log.error("保存消息到Caffeine缓存时发生错误", e);
            throw new RuntimeException("Failed to save messages to Caffeine cache", e);
        }
    }
    
    @Override
    public List<Message> get(String conversationId, int lastN) {
        try {
            List<Message> messages = cache.getIfPresent(conversationId);
            
            if (messages == null) {
                return new ArrayList<>();
            }
            
            // 返回最新的N条消息
            if (lastN < messages.size()) {
                return messages.subList(messages.size() - lastN, messages.size());
            }
            
            return new ArrayList<>(messages); // 返回副本以防止外部修改
        } catch (Exception e) {
            log.error("从Caffeine缓存获取消息时发生错误", e);
            throw new RuntimeException("Failed to get messages from Caffeine cache", e);
        }
    }
    
    @Override
    public void clear(String conversationId) {
        try {
            cache.invalidate(conversationId);
            log.debug("成功清除会话{}", conversationId);
        } catch (Exception e) {
            log.error("清除会话时发生错误", e);
            throw new RuntimeException("Failed to clear conversation", e);
        }
    }
}