package pangea.hiagent.tools;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.stereotype.Component;

/**
 * 图表生成工具
 * 用于根据数据生成各种类型的图表
 */
@Slf4j
@Component
public class ChartGenerationTool {
    
    /**
     * 生成柱状图
     * @param title 图表标题
     * @param xAxisLabels X轴标签数组
     * @param seriesData 数据系列数组
     * @param seriesName 数据系列名称
     * @return 图表生成结果
     */
    @Tool(description = "根据数据生成柱状图")
    public String generateBarChart(String title, String[] xAxisLabels, double[] seriesData, String seriesName) {
        log.debug("开始生成柱状图: {}", title);
        
        try {
            if (title == null || title.trim().isEmpty()) {
                log.warn("图表标题不能为空");
                return "错误：图表标题不能为空";
            }
            
            if (xAxisLabels == null || xAxisLabels.length == 0) {
                log.warn("X轴标签不能为空");
                return "错误：X轴标签不能为空";
            }
            
            if (seriesData == null || seriesData.length == 0) {
                log.warn("数据系列不能为空");
                return "错误：数据系列不能为空";
            }
            
            if (xAxisLabels.length != seriesData.length) {
                log.warn("X轴标签数量与数据系列数量不匹配");
                return "错误：X轴标签数量与数据系列数量不匹配";
            }
            
            // 生成图表描述
            StringBuilder chartDescription = new StringBuilder();
            chartDescription.append("柱状图生成成功：\n");
            chartDescription.append("标题: ").append(title).append("\n");
            chartDescription.append("数据系列: ").append(seriesName != null ? seriesName : "数据").append("\n");
            chartDescription.append("数据点数量: ").append(seriesData.length).append("\n");
            chartDescription.append("数据详情:\n");
            
            for (int i = 0; i < xAxisLabels.length; i++) {
                chartDescription.append("  ").append(xAxisLabels[i]).append(": ").append(seriesData[i]).append("\n");
            }
            
            log.info("柱状图生成完成，包含 {} 个数据点", seriesData.length);
            return chartDescription.toString();
        } catch (Exception e) {
            log.error("生成柱状图时发生错误: {}", e.getMessage(), e);
            return "生成柱状图时发生错误: " + e.getMessage();
        }
    }
    
    /**
     * 生成折线图
     * @param title 图表标题
     * @param xAxisLabels X轴标签数组
     * @param seriesData 数据系列数组
     * @param seriesName 数据系列名称
     * @return 图表生成结果
     */
    @Tool(description = "根据数据生成折线图")
    public String generateLineChart(String title, String[] xAxisLabels, double[] seriesData, String seriesName) {
        log.debug("开始生成折线图: {}", title);
        
        try {
            if (title == null || title.trim().isEmpty()) {
                log.warn("图表标题不能为空");
                return "错误：图表标题不能为空";
            }
            
            if (xAxisLabels == null || xAxisLabels.length == 0) {
                log.warn("X轴标签不能为空");
                return "错误：X轴标签不能为空";
            }
            
            if (seriesData == null || seriesData.length == 0) {
                log.warn("数据系列不能为空");
                return "错误：数据系列不能为空";
            }
            
            if (xAxisLabels.length != seriesData.length) {
                log.warn("X轴标签数量与数据系列数量不匹配");
                return "错误：X轴标签数量与数据系列数量不匹配";
            }
            
            // 生成图表描述
            StringBuilder chartDescription = new StringBuilder();
            chartDescription.append("折线图生成成功：\n");
            chartDescription.append("标题: ").append(title).append("\n");
            chartDescription.append("数据系列: ").append(seriesName != null ? seriesName : "数据").append("\n");
            chartDescription.append("数据点数量: ").append(seriesData.length).append("\n");
            chartDescription.append("数据详情:\n");
            
            for (int i = 0; i < xAxisLabels.length; i++) {
                chartDescription.append("  ").append(xAxisLabels[i]).append(": ").append(seriesData[i]).append("\n");
            }
            
            log.info("折线图生成完成，包含 {} 个数据点", seriesData.length);
            return chartDescription.toString();
        } catch (Exception e) {
            log.error("生成折线图时发生错误: {}", e.getMessage(), e);
            return "生成折线图时发生错误: " + e.getMessage();
        }
    }
    
    /**
     * 生成饼图
     * @param title 图表标题
     * @param labels 饼图各部分标签数组
     * @param values 饼图各部分数值数组
     * @return 图表生成结果
     */
    @Tool(description = "根据数据生成饼图")
    public String generatePieChart(String title, String[] labels, double[] values) {
        log.debug("开始生成饼图: {}", title);
        
        try {
            if (title == null || title.trim().isEmpty()) {
                log.warn("图表标题不能为空");
                return "错误：图表标题不能为空";
            }
            
            if (labels == null || labels.length == 0) {
                log.warn("标签不能为空");
                return "错误：标签不能为空";
            }
            
            if (values == null || values.length == 0) {
                log.warn("数值不能为空");
                return "错误：数值不能为空";
            }
            
            if (labels.length != values.length) {
                log.warn("标签数量与数值数量不匹配");
                return "错误：标签数量与数值数量不匹配";
            }
            
            // 计算总值
            double total = 0;
            for (double value : values) {
                total += value;
            }
            
            // 生成图表描述
            StringBuilder chartDescription = new StringBuilder();
            chartDescription.append("饼图生成成功：\n");
            chartDescription.append("标题: ").append(title).append("\n");
            chartDescription.append("数据项数量: ").append(values.length).append("\n");
            chartDescription.append("总计: ").append(total).append("\n");
            chartDescription.append("数据详情:\n");
            
            for (int i = 0; i < labels.length; i++) {
                double percentage = total > 0 ? (values[i] / total) * 100 : 0;
                chartDescription.append("  ").append(labels[i]).append(": ").append(values[i])
                    .append(" (").append(String.format("%.2f", percentage)).append("%)\n");
            }
            
            log.info("饼图生成完成，包含 {} 个数据项", values.length);
            return chartDescription.toString();
        } catch (Exception e) {
            log.error("生成饼图时发生错误: {}", e.getMessage(), e);
            return "生成饼图时发生错误: " + e.getMessage();
        }
    }
}