package pangea.hiagent.websocket;

import com.alibaba.fastjson2.JSON;
import com.microsoft.playwright.*;
import com.microsoft.playwright.options.LoadState;
import org.springframework.web.socket.*;
import org.springframework.web.socket.handler.TextWebSocketHandler;
import pangea.hiagent.dto.DomSyncData;

import java.io.ByteArrayOutputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.zip.GZIPOutputStream;

/**
 * DOM同步的WebSocket处理器
 */
public class DomSyncHandler extends TextWebSocketHandler {
    // 存储连接的前端客户端（线程安全）
    private static final ConcurrentMap<WebSocketSession, String> clients = new ConcurrentHashMap<>();

    // Playwright核心实例
    private Playwright playwright;
    private Browser browser;
    private Page page;
    private BrowserContext context;
    
    // 压缩和分片配置
    private static final int CHUNK_SIZE = 64 * 1024; // 64KB分片大小
    private static final boolean ENABLE_COMPRESSION = true; // 是否启用压缩
    
    // 连接数限制
    private static final int MAX_CONNECTIONS_PER_USER = 5; // 每用户最大连接数
    private static final int MAX_COMMANDS_PER_SECOND = 10; // 每秒最大指令数
    
    // 用户连接计数
    private static final ConcurrentMap<String, Integer> userConnections = new ConcurrentHashMap<>();
    
    // 指令频率限制
    private static final ConcurrentMap<String, Long> lastCommandTimes = new ConcurrentHashMap<>();
    private static final ConcurrentMap<String, Integer> commandCounts = new ConcurrentHashMap<>();
    
    // 统计信息
    private static final ConcurrentMap<String, Long> messageCounters = new ConcurrentHashMap<>();
    
    // 增加计数器
    private void incrementCounter(String counterName) {
        messageCounters.merge(counterName, 1L, Long::sum);
    }
    
    // 获取计数器值
    public static long getCounter(String counterName) {
        return messageCounters.getOrDefault(counterName, 0L);
    }
    
    // 重置计数器
    public static void resetCounter(String counterName) {
        messageCounters.put(counterName, 0L);
    }
    
    // 初始化Playwright与页面
    public DomSyncHandler() {
        initPlaywright();
        initPageListener();
    }

    /**
     * 初始化Playwright（服务器端无头模式，适配生产环境）
     */
    private void initPlaywright() {
        try {
            System.out.println("正在初始化Playwright...");
            playwright = Playwright.create();
            // 启动Chrome无头实例，添加必要参数
            browser = playwright.chromium().launch(new BrowserType.LaunchOptions()
                    .setHeadless(true) // 无头模式，无界面
                    .setArgs(java.util.Arrays.asList(
                            "--no-sandbox", // 服务器端必须，否则Chrome启动失败
                            "--disable-dev-shm-usage", // 解决Linux下/dev/shm空间不足的问题
                            "--disable-gpu", // 禁用GPU，服务器端无需
                            "--remote-allow-origins=*", // 允许跨域请求（处理外部样式/脚本）
                            "--disable-web-security", // 禁用网络安全检查（便于测试）
                            "--allow-running-insecure-content" // 允许不安全内容
                    )));
            
            // 创建浏览器上下文（隔离环境）
            context = browser.newContext(new Browser.NewContextOptions()
                    .setViewportSize(1920, 1080) // 设置视口大小，与前端一致
                    .setUserAgent("Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36")); // 设置用户代理
            
            page = context.newPage();
            
            // 设置默认超时时间，避免长时间等待
            page.setDefaultTimeout(10000); // 10秒超时
            
            System.out.println("Playwright初始化成功");
        } catch (Exception e) {
            System.err.println("Playwright初始化失败: " + e.getMessage());
            e.printStackTrace();
            
            // 尝试重新初始化
            try {
                System.out.println("尝试重新初始化Playwright...");
                destroy(); // 先清理现有资源
                Thread.sleep(1000); // 等待1秒
                
                playwright = Playwright.create();
                browser = playwright.chromium().launch(new BrowserType.LaunchOptions()
                        .setHeadless(true)
                        .setArgs(java.util.Arrays.asList(
                                "--no-sandbox",
                                "--disable-dev-shm-usage",
                                "--disable-gpu",
                                "--remote-allow-origins=*")));
                
                context = browser.newContext(new Browser.NewContextOptions()
                        .setViewportSize(1920, 1080));
                page = context.newPage();
                page.setDefaultTimeout(10000); // 10秒超时
                System.out.println("Playwright重新初始化成功");
            } catch (Exception re) {
                System.err.println("Playwright重新初始化失败: " + re.getMessage());
                re.printStackTrace();
            }
        }
    }

    /**
     * 初始化页面监听事件（核心：捕获DOM、样式、脚本变化）
     */
    private void initPageListener() {
        // 初始化统计计数器
        messageCounters.put("domChanges", 0L);
        messageCounters.put("websocketMessages", 0L);
        messageCounters.put("sseEvents", 0L);
        messageCounters.put("errors", 0L);
        
        // 1. 页面加载完成后，推送完整的DOM、样式、脚本（初始化）
        page.onLoad(page -> {
            System.out.println("页面加载完成: " + page.url());
            incrementCounter("pageLoads");
            sendFullDomAndResourceToClients();
        });

        // 2. 监听DOM变化（使用MutationObserver），推送增量更新
        page.evaluate("() => {\n" +
                "  // 创建MutationObserver监听DOM变化\n" +
                "  const observer = new MutationObserver((mutations) => {\n" +
                "    // 将变化的DOM节点转为字符串，发送给Playwright\n" +
                "    const changes = mutations.map(mutation => ({\n" +
                "      type: mutation.type,\n" +
                "      target: mutation.target.outerHTML,\n" +
                "      addedNodes: Array.from(mutation.addedNodes).map(node => node.outerHTML || ''),\n" +
                "      removedNodes: Array.from(mutation.removedNodes).map(node => node.outerHTML || '')\n" +
                "    }));\n" +
                "    // 调用Playwright的暴露函数，传递DOM变化数据\n" +
                "    window.domChanged(JSON.stringify(changes));\n" +
                "  });\n" +
                "  // 配置监听：监听所有节点的添加/删除、属性变化、子节点变化\n" +
                "  observer.observe(document.body, {\n" +
                "    childList: true,\n" +
                "    attributes: true,\n" +
                "    subtree: true,\n" +
                "    characterData: true\n" +
                "  });\n" +
                "}");

        // 3. 暴露Playwright函数，接收前端的DOM变化数据
        page.exposeFunction("domChanged", args -> {
            try {
                if (args.length > 0 && args[0] instanceof String) {
                    String changes = (String) args[0];
                    if (changes != null && !changes.isEmpty()) {
                        incrementCounter("domChanges");
                        sendIncrementalDomToClients(changes);
                    }
                }
            } catch (Exception e) {
                String errorMsg = "处理DOM变化失败: " + e.getMessage();
                System.err.println(errorMsg);
                e.printStackTrace();
                incrementCounter("errors");
                sendErrorToClients(errorMsg);
            }
            return null;
        });

        // 4. 监听WebSocket连接
        page.onWebSocket(webSocket -> {
            System.out.println("检测到WebSocket连接: " + webSocket.url());
            
            // 监听WebSocket的消息接收事件
            webSocket.onFrameReceived(frame -> {
                try {
                    incrementCounter("websocketMessages");
                    // 封装WebSocket消息，推送给前端
                    DomSyncData wsData = new DomSyncData(
                            "ws", // WebSocket类型
                            webSocket.url(), // WebSocket地址
                            frame.text(), // 消息内容
                            "",
                            getCurrentPageUrl()
                    );
                    String jsonData = JSON.toJSONString(wsData);
                    broadcastMessage(jsonData);
                } catch (Exception e) {
                    String errorMsg = "处理WebSocket消息失败: " + e.getMessage();
                    System.err.println(errorMsg);
                    e.printStackTrace();
                    incrementCounter("errors");
                    sendErrorToClients(errorMsg);
                }
            });
            
            // 监听WebSocket关闭事件
            webSocket.onClose(closedWebSocket -> {
                try {
                    String closeInfo = "WebSocket连接已关闭: " + closedWebSocket.url();
                    System.out.println(closeInfo);
                    
                    DomSyncData wsCloseData = new DomSyncData(
                            "ws-close", // WebSocket关闭类型
                            closedWebSocket.url(), // WebSocket地址
                            closeInfo, // 关闭信息
                            "",
                            getCurrentPageUrl()
                    );
                    String jsonData = JSON.toJSONString(wsCloseData);
                    broadcastMessage(jsonData);
                } catch (Exception e) {
                    String errorMsg = "处理WebSocket关闭事件失败: " + e.getMessage();
                    System.err.println(errorMsg);
                    e.printStackTrace();
                    incrementCounter("errors");
                    sendErrorToClients(errorMsg);
                }
            });
            
            // 监听WebSocket帧发送事件
            webSocket.onFrameSent(frame -> {
                try {
                    // 记录发送的WebSocket帧（可用于调试）
                    System.out.println("WebSocket发送帧到: " + webSocket.url() + ", 内容长度: " + 
                        (frame.text() != null ? frame.text().length() : 0));
                } catch (Exception e) {
                    // 静默处理，避免影响主流程
                    System.err.println("记录WebSocket发送帧失败: " + e.getMessage());
                }
            });
            
            // 注意：Playwright Java WebSocket API 可能不支持 onError 回调
            // 如果需要错误处理，可以通过其他方式实现
        });

        // 5. 监听SSE（Server-Sent Events）连接
        page.onResponse(response -> {
            try {
                // 检查是否为SSE响应
                String contentType = response.headers().get("content-type");
                if (contentType != null && contentType.contains("text/event-stream")) {
                    System.out.println("检测到SSE响应: " + response.url());
                    incrementCounter("sseResponses");
                    
                    // 添加页面错误监听器来捕获SSE相关的错误
                    response.frame().page().onPageError(error -> {
                        try {
                            String errorMsg = "页面错误 (可能与SSE相关): " + error;
                            System.err.println(errorMsg);
                            incrementCounter("errors");
                            // 发送错误信息给客户端
                            DomSyncData errorData = new DomSyncData(
                                    "sse-error", // SSE错误类型
                                    response.url(), // SSE地址
                                    errorMsg, // 错误信息
                                    "",
                                    getCurrentPageUrl()
                            );
                            broadcastMessage(JSON.toJSONString(errorData));
                        } catch (Exception e) {
                            System.err.println("处理SSE页面错误失败: " + e.getMessage());
                            e.printStackTrace();
                            incrementCounter("errors");
                        }
                    });
                    
                    // 尝试通过控制台消息捕获SSE数据
                    response.frame().page().onConsoleMessage(consoleMessage -> {
                        try {
                            // 检查是否可能是SSE相关的消息
                            String text = consoleMessage.text();
                            if (text != null && (text.contains("event:") || text.contains("data:") || text.contains("id:"))) {
                                System.out.println("检测到可能的SSE数据: " + text);
                                incrementCounter("sseEvents");
                                
                                // 处理SSE数据
                                DomSyncData sseDataObj = new DomSyncData(
                                        "sse", // SSE类型
                                        response.url(), // SSE地址
                                        text, // SSE内容
                                        "",
                                        getCurrentPageUrl()
                                );
                                broadcastMessage(JSON.toJSONString(sseDataObj));
                            }
                        } catch (Exception e) {
                            System.err.println("处理SSE控制台消息失败: " + e.getMessage());
                            e.printStackTrace();
                            incrementCounter("errors");
                        }
                    });
                    
                    // 尝试通过请求完成事件来捕获SSE数据
                    response.frame().page().onRequestFinished(request -> {
                        try {
                            // 检查是否是SSE请求
                            if (request.url().equals(response.url())) {
                                System.out.println("SSE请求完成: " + request.url());
                                // 可以在这里添加额外的处理逻辑
                            }
                        } catch (Exception e) {
                            System.err.println("处理SSE请求完成事件失败: " + e.getMessage());
                            e.printStackTrace();
                            incrementCounter("errors");
                        }
                    });
                }
            } catch (Exception e) {
                String errorMsg = "检查SSE响应失败: " + e.getMessage();
                System.err.println(errorMsg);
                e.printStackTrace();
                incrementCounter("errors");
                sendErrorToClients(errorMsg);
            }
        });

        // 6. 监听页面导航事件，导航后重新初始化
        page.onFrameNavigated(frame -> {
            System.out.println("检测到页面导航: " + frame.url());
            incrementCounter("navigations");
            // 导航完成后，等待页面加载
            try {
                page.waitForLoadState(LoadState.LOAD);
                System.out.println("页面加载完成: " + frame.url());
                // 发送更新后的DOM内容到客户端
                sendFullDomAndResourceToClients();
            } catch (Exception e) {
                String errorMsg = "页面加载状态等待失败: " + e.getMessage();
                System.err.println(errorMsg);
                e.printStackTrace();
                incrementCounter("errors");
                sendErrorToClients(errorMsg);
            }
        });
        
        // 7. 监听页面错误事件
        page.onPageError(error -> {
            try {
                String errorMsg = "页面错误: " + error;
                System.err.println(errorMsg);
                incrementCounter("errors");
                sendErrorToClients(errorMsg);
            } catch (Exception e) {
                System.err.println("处理页面错误事件失败: " + e.getMessage());
                e.printStackTrace();
            }
        });
        
        // 8. 监听控制台消息事件
        page.onConsoleMessage(message -> {
            try {
                String consoleMsg = "控制台消息 [" + message.type() + "]: " + message.text();
                System.out.println(consoleMsg);
                
                // 如果是错误类型的消息，也发送给客户端
                if ("error".equals(message.type())) {
                    incrementCounter("consoleErrors");
                    sendErrorToClients(consoleMsg);
                }
            } catch (Exception e) {
                System.err.println("处理控制台消息事件失败: " + e.getMessage());
                e.printStackTrace();
            }
        });
    }

    /**
     * 推送完整的DOM、样式、脚本给所有客户端（初始化时调用）
     */
    private void sendFullDomAndResourceToClients() {
        try {
            // 1. 获取页面完整DOM（包含所有节点）
            String fullDom = page.content();

            // 2. 获取页面所有样式（内联样式+外部样式表内容）
            String fullStyle = getPageAllStyles();

            // 3. 获取页面所有脚本（内联脚本+外部脚本的URL，前端自行加载）
            String fullScript = getPageAllScripts();

            // 4. 封装数据
            DomSyncData data = new DomSyncData(
                    "init", // 初始化类型
                    fullDom,
                    fullStyle,
                    fullScript,
                    getCurrentPageUrl() // 当前页面URL
            );

            // 5. 序列化为JSON并推送
            String jsonData = JSON.toJSONString(data);
            broadcastMessage(jsonData);

        } catch (Exception e) {
            e.printStackTrace();
            sendErrorToClients("获取完整DOM失败：" + e.getMessage());
        }
    }

    /**
     * 获取页面所有样式（内联+外部）
     * 注意：外部样式表通过Playwright请求获取内容，避免前端跨域问题
     */
    private String getPageAllStyles() {
        try {
            Object result = page.evaluate("async () => {" +
                    "let styleText = '';" +
                    "// 1. 获取内联样式（<style>标签）" +
                    "document.querySelectorAll('style').forEach(style => {" +
                    "   styleText += style.textContent + '\\n';" +
                    "});" +
                    "// 2. 获取外部样式表（<link rel=\"stylesheet\">）" +
                    "const linkElements = document.querySelectorAll('link[rel=\"stylesheet\"]');" +
                    "for (let link of linkElements) {" +
                    "   try {" +
                    "       const response = await fetch(link.href);" +
                    "       if (response.ok) {" +
                    "           const cssText = await response.text();" +
                    "           styleText += cssText + '\\n';" +
                    "       }" +
                    "   } catch (e) {" +
                    "       console.error('加载外部样式表失败：', link.href, e);" +
                    "   }" +
                    "}" +
                    "return styleText;" +
                    "}");
            return result != null ? result.toString() : "";
        } catch (Exception e) {
            System.err.println("获取页面样式失败: " + e.getMessage());
            e.printStackTrace();
            return "";
        }
    }

    /**
     * 获取页面所有脚本（内联脚本+外部脚本URL，前端按需加载）
     * 外部脚本不直接获取内容，避免体积过大，前端通过URL加载（需处理跨域）
     */
    private String getPageAllScripts() {
        try {
            Object result = page.evaluate("() => {" +
                    "let scriptData = {" +
                    "   inline: [], // 内联脚本内容" +
                    "   external: [] // 外部脚本URL" +
                    "};" +
                    "// 1. 获取内联脚本（<script>标签，无src属性）" +
                    "document.querySelectorAll('script:not([src])').forEach(script => {" +
                    "   scriptData.inline.push(script.textContent);" +
                    "});" +
                    "// 2. 获取外部脚本URL（<script>标签，有src属性）" +
                    "document.querySelectorAll('script[src]').forEach(script => {" +
                    "   scriptData.external.push(script.src);" +
                    "});" +
                    "return JSON.stringify(scriptData);" +
                    "}");
            return result != null ? result.toString() : "{}";
        } catch (Exception e) {
            System.err.println("获取页面脚本失败: " + e.getMessage());
            e.printStackTrace();
            return "{}";
        }
    }

    /**
     * 推送增量DOM变化给所有客户端（DOM更新时调用）
     */
    private void sendIncrementalDomToClients(String changes) {
        try {
            DomSyncData data = new DomSyncData(
                    "update", // 增量更新类型
                    changes, // DOM变化数据
                    "", // 样式无增量，空字符串
                    "", // 脚本无增量，空字符串
                    getCurrentPageUrl()
            );
            String jsonData = JSON.toJSONString(data);
            broadcastMessage(jsonData);
        } catch (Exception e) {
            e.printStackTrace();
            sendErrorToClients("推送增量DOM失败：" + e.getMessage());
        }
    }
    
    /**
     * 广播消息给所有客户端（带压缩和分片）
     */
    private void broadcastMessage(String message) {
        if (ENABLE_COMPRESSION && message.getBytes().length > CHUNK_SIZE) {
            // 启用压缩且消息较大时，进行压缩和分片
            try {
                byte[] compressedData = compressData(message);
                sendChunkedData(compressedData);
            } catch (Exception e) {
                e.printStackTrace();
                // 压缩失败时，直接发送原始数据
                sendRawMessage(message);
            }
        } else {
            // 不启用压缩或消息较小时，直接发送
            sendRawMessage(message);
        }
    }
    
    /**
     * 发送原始消息（不分片）
     */
    private void sendRawMessage(String message) {
        TextMessage textMessage = new TextMessage(message);
        for (WebSocketSession client : clients.keySet()) {
            try {
                if (client.isOpen()) {
                    client.sendMessage(textMessage);
                }
            } catch (Exception e) {
                e.printStackTrace();
                // 发送失败时移除客户端
                clients.remove(client);
            }
        }
    }
    
    /**
     * 压缩数据（使用GZIP）
     */
    private byte[] compressData(String data) throws Exception {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream gzipOut = new GZIPOutputStream(baos);
        gzipOut.write(data.getBytes("UTF-8"));
        gzipOut.close();
        return baos.toByteArray();
    }
    
    /**
     * 发送分片数据
     */
    private void sendChunkedData(byte[] data) {
        try {
            int totalChunks = (int) Math.ceil((double) data.length / CHUNK_SIZE);
            
            for (int i = 0; i < totalChunks; i++) {
                int start = i * CHUNK_SIZE;
                int end = Math.min(start + CHUNK_SIZE, data.length);
                byte[] chunk = java.util.Arrays.copyOfRange(data, start, end);
                
                // 构造分片消息
                DomSyncData chunkData = new DomSyncData(
                    "chunk", // 分片类型
                    String.valueOf(i), // 当前分片索引
                    String.valueOf(totalChunks), // 总分片数
                    java.util.Base64.getEncoder().encodeToString(chunk), // 分片内容（Base64编码）
                    ""
                );
                
                String jsonData = JSON.toJSONString(chunkData);
                sendRawMessage(jsonData);
            }
        } catch (Exception e) {
            System.err.println("发送分片数据失败: " + e.getMessage());
            e.printStackTrace();
            sendErrorToClients("发送分片数据失败: " + e.getMessage());
        }
    }
    
    /**
     * 发送错误信息给所有客户端
     */
    private void sendErrorToClients(String errorMessage) {
        DomSyncData errorData = new DomSyncData(
                "error", // 错误类型
                errorMessage, // 错误信息
                "", 
                "",
                getCurrentPageUrl()
        );
        String jsonData = JSON.toJSONString(errorData);
        broadcastMessage(jsonData);
    }

    // ===================== WebSocket生命周期方法 =====================
    /**
     * 客户端连接建立时触发
     */
    @Override
    public void afterConnectionEstablished(WebSocketSession session) {
        // 生产环境：此处添加Token认证逻辑（参考之前的权限控制部分）
        // 从会话属性中获取用户ID
        String userId = (String) session.getAttributes().get("userId");
        if (userId == null) {
            userId = "anonymous"; // 默认匿名用户
        }
        
        // 检查连接数限制
        Integer currentConnections = userConnections.getOrDefault(userId, 0);
        if (currentConnections >= MAX_CONNECTIONS_PER_USER) {
            try {
                session.close(CloseStatus.POLICY_VIOLATION.withReason("超过最大连接数限制"));
                return;
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        
        // 增加用户连接数
        userConnections.put(userId, currentConnections + 1);
        
        clients.put(session, session.getId());
        System.out.println("客户端连接成功：" + session.getId() + "，用户：" + userId);
        
        // 连接建立后不立即推送DOM，而是等待客户端发送navigate指令
        // 这样可以避免在WebSocket连接建立但iframe还未准备好时推送数据
        System.out.println("WebSocket连接已建立，等待客户端发送导航指令...");
    }

    /**
     * 处理客户端发送的指令（如导航、点击、输入）
     */
    @Override
    protected void handleTextMessage(WebSocketSession session, TextMessage message) {
        String payload = message.getPayload();
        
        // 从会话属性中获取用户ID
        String userId = (String) session.getAttributes().get("userId");
        if (userId == null) {
            userId = "anonymous"; // 默认匿名用户
        }
        
        // 指令频率限制
        long currentTime = System.currentTimeMillis();
        Long lastTime = lastCommandTimes.getOrDefault(userId, 0L);
        Integer commandCount = commandCounts.getOrDefault(userId, 0);
        
        // 检查是否超过每秒最大指令数
        if (currentTime - lastTime < 1000) {
            // 同一秒内
            if (commandCount >= MAX_COMMANDS_PER_SECOND) {
                sendErrorToClients("指令执行过于频繁，请稍后再试");
                return;
            }
            commandCounts.put(userId, commandCount + 1);
        } else {
            // 新的一秒
            lastCommandTimes.put(userId, currentTime);
            commandCounts.put(userId, 1);
        }
        
        try {
            // 检查Playwright实例是否有效
            if (!isPlaywrightInstanceValid()) {
                sendErrorToClients("Playwright实例未初始化或已关闭，请刷新页面重试");
                return;
            }
            
            // 检查WebSocket连接状态
            if (!session.isOpen()) {
                sendErrorToClients("WebSocket连接已关闭");
                return;
            }
            
            // 指令格式：指令类型:参数（如navigate:https://www.baidu.com）
            // 特殊处理navigate命令，避免URL中的冒号导致分割错误
            String command;
            String param;
            
            int firstColonIndex = payload.indexOf(':');
            if (firstColonIndex != -1) {
                command = payload.substring(0, firstColonIndex);
                param = payload.substring(firstColonIndex + 1);
            } else {
                command = payload;
                param = "";
            }
            
            switch (command) {
                case "navigate":
                    // 导航到指定URL
                    if (param == null || param.trim().isEmpty()) {
                        sendErrorToClients("导航URL不能为空");
                        return;
                    }
                    
                    // 验证URL格式
                    if (!isValidUrl(param)) {
                        sendErrorToClients("无效的URL格式: " + param);
                        return;
                    }
                    
                    try {
                        System.out.println("正在导航到URL: " + param);
                        page.navigate(param);
                        System.out.println("导航成功: " + param);
                    } catch (Exception e) {
                        String errorMsg = "导航失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "stats":
                    // 获取统计信息
                    try {
                        String stats = getStatisticsSummary();
                        DomSyncData statsData = new DomSyncData(
                                "stats", // 统计信息类型
                                stats, // 统计信息内容
                                "", 
                                "",
                                page != null ? page.url() : ""
                        );
                        String jsonData = JSON.toJSONString(statsData);
                        broadcastMessage(jsonData);
                    } catch (Exception e) {
                        String errorMsg = "获取统计信息失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "reset-stats":
                    // 重置统计信息
                    try {
                        resetAllCounters();
                        sendErrorToClients("统计信息已重置");
                    } catch (Exception e) {
                        String errorMsg = "重置统计信息失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "click":
                    // 点击指定选择器的元素（如#su）
                    try {
                        // 检查选择器是否为容器元素本身
                        if ("#dom-view".equals(param)) {
                            sendErrorToClients("不能对容器元素 #dom-view 执行点击操作");
                        } else {
                            page.locator(param).click();
                        }
                    } catch (Exception e) {
                        String errorMsg = "点击元素失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "dblclick":
                    // 双击指定选择器的元素
                    try {
                        // 检查选择器是否为容器元素本身
                        if ("#dom-view".equals(param)) {
                            sendErrorToClients("不能对容器元素 #dom-view 执行双击操作");
                        } else {
                            page.locator(param).dblclick();
                        }
                    } catch (Exception e) {
                        String errorMsg = "双击元素失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "hover":
                    // 悬停在指定选择器的元素上
                    try {
                        // 检查选择器是否为容器元素本身，避免对容器元素执行hover操作
                        if ("#dom-view".equals(param)) {
                            // 忽略对容器元素本身的hover操作，不报错但也不执行
                            System.out.println("忽略对容器元素 #dom-view 的hover操作");
                        } else {
                            page.locator(param).hover();
                        }
                    } catch (Exception e) {
                        String errorMsg = "悬停元素失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "mousedown":
                    // 鼠标按下指定选择器的元素
                    try {
                        // 检查选择器是否为容器元素本身
                        if ("#dom-view".equals(param)) {
                            sendErrorToClients("不能对容器元素 #dom-view 执行鼠标按下操作");
                        } else {
                            page.mouse().down(); // 使用page.mouse()而不是locator.mouse()
                        }
                    } catch (Exception e) {
                        String errorMsg = "鼠标按下失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "mouseup":
                    // 鼠标释放指定选择器的元素
                    try {
                        // 检查选择器是否为容器元素本身
                        if ("#dom-view".equals(param)) {
                            sendErrorToClients("不能对容器元素 #dom-view 执行鼠标释放操作");
                        } else {
                            page.mouse().up(); // 使用page.mouse()而不是locator.mouse()
                        }
                    } catch (Exception e) {
                        String errorMsg = "鼠标释放失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "focus":
                    // 聚焦到指定元素
                    try {
                        // 检查选择器是否为容器元素本身
                        if ("#dom-view".equals(param)) {
                            sendErrorToClients("不能对容器元素 #dom-view 执行聚焦操作");
                        } else {
                            page.locator(param).focus();
                        }
                    } catch (Exception e) {
                        String errorMsg = "聚焦元素失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "blur":
                    // 失去焦点
                    try {
                        // 检查选择器是否为容器元素本身
                        if ("#dom-view".equals(param)) {
                            sendErrorToClients("不能对容器元素 #dom-view 执行失去焦点操作");
                        } else {
                            page.locator(param).blur();
                        }
                    } catch (Exception e) {
                        String errorMsg = "失去焦点失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "select":
                    // 选择下拉选项（格式：选择器:值）
                    try {
                        String[] selectParts = param.split(":", 2);
                        if (selectParts.length == 2) {
                            page.locator(selectParts[0]).selectOption(selectParts[1]);
                        }
                    } catch (Exception e) {
                        String errorMsg = "选择选项失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "keydown":
                    // 键盘按下事件（格式：键码）
                    try {
                        page.keyboard().down(param);
                    } catch (Exception e) {
                        String errorMsg = "键盘按下失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "keyup":
                    // 键盘释放事件（格式：键码）
                    try {
                        page.keyboard().up(param);
                    } catch (Exception e) {
                        String errorMsg = "键盘释放失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "keypress":
                    // 键盘按键事件（格式：键码）
                    try {
                        page.keyboard().press(param);
                    } catch (Exception e) {
                        String errorMsg = "按键事件失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "scroll":
                    // 滚动页面（格式：y坐标，如500）
                    try {
                        int scrollY = Integer.parseInt(param);
                        page.evaluate("window.scrollTo(0, " + scrollY + ")");
                    } catch (Exception e) {
                        String errorMsg = "滚动页面失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "type":
                    // 输入内容（格式：选择器:内容，如#kw:Java Playwright）
                    try {
                        // 使用indexOf和substring替代split，避免内容中的冒号导致分割错误
                        int colonIndex = param.indexOf(':');
                        if (colonIndex != -1 && colonIndex < param.length() - 1) {
                            String selector = param.substring(0, colonIndex);
                            String content = param.substring(colonIndex + 1);
                            
                            // 检查选择器是否为容器元素本身
                            if ("#dom-view".equals(selector)) {
                                sendErrorToClients("不能对容器元素 #dom-view 执行输入操作");
                            } else {
                                Locator inputLocator = page.locator(selector);
                                inputLocator.fill(content); // 使用fill替代type，因为type方法已被弃用
                            }
                        }
                    } catch (Exception e) {
                        String errorMsg = "输入内容失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "input":
                    // 更丰富的输入场景（格式：选择器:内容）
                    try {
                        // 使用indexOf和substring替代split，避免内容中的冒号导致分割错误
                        int colonIndex = param.indexOf(':');
                        if (colonIndex != -1 && colonIndex < param.length() - 1) {
                            String selector = param.substring(0, colonIndex);
                            String content = param.substring(colonIndex + 1);
                            
                            // 检查选择器是否为容器元素本身
                            if ("#dom-view".equals(selector)) {
                                sendErrorToClients("不能对容器元素 #dom-view 执行输入操作");
                            } else {
                                Locator inputLocator = page.locator(selector);
                                inputLocator.fill(content); // 使用fill替代type，支持更丰富的输入
                            }
                        }
                    } catch (Exception e) {
                        String errorMsg = "输入内容失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;

                default:
                    sendErrorToClients("未知指令：" + command);
            }
        } catch (Exception e) {
            e.printStackTrace();
            sendErrorToClients("指令执行失败：" + e.getMessage());
        }
    }

    /**
     * 客户端连接关闭时触发
     */
    @Override
    public void afterConnectionClosed(WebSocketSession session, CloseStatus status) {
        clients.remove(session);
        System.out.println("客户端断开连接：" + session.getId());
        
        // 从会话属性中获取用户ID
        String userId = (String) session.getAttributes().get("userId");
        if (userId == null) {
            userId = "anonymous"; // 默认匿名用户
        }
        
        // 减少用户连接数
        Integer currentConnections = userConnections.getOrDefault(userId, 0);
        if (currentConnections > 0) {
            userConnections.put(userId, currentConnections - 1);
        }
    }

    /**
     * 处理WebSocket传输错误
     */
    @Override
    public void handleTransportError(WebSocketSession session, Throwable exception) {
        clients.remove(session);
        System.out.println("客户端传输错误：" + session.getId() + "，错误信息：" + exception.getMessage());
    }

    /**
     * 验证URL格式是否有效
     *
     * @param url 要验证的URL
     * @return 如果URL有效返回true，否则返回false
     */
    private boolean isValidUrl(String url) {
        if (url == null || url.trim().isEmpty()) {
            return false;
        }
        
        // 检查是否以http://或https://开头
        if (!url.toLowerCase().startsWith("http://") && !url.toLowerCase().startsWith("https://")) {
            return false;
        }
        
        // 简单的URL格式检查
        try {
            new java.net.URL(url);
            return true;
        } catch (Exception e) {
            return false;
        }
    }
    
    /**
     * 销毁资源（Spring Boot关闭时调用）
     */
    public void destroy() {
        try {
            // 关闭所有客户端连接
            for (WebSocketSession session : clients.keySet()) {
                try {
                    if (session.isOpen()) {
                        session.close(CloseStatus.NORMAL);
                    }
                } catch (Exception e) {
                    System.err.println("关闭WebSocket会话失败: " + e.getMessage());
                }
            }
            clients.clear();
        } catch (Exception e) {
            System.err.println("清理客户端连接失败: " + e.getMessage());
        }

        try {
            // 关闭Playwright资源
            if (page != null && !page.isClosed()) {
                page.close();
            }
        } catch (Exception e) {
            System.err.println("关闭页面失败: " + e.getMessage());
        }
        
        try {
            if (context != null) {
                context.close();
            }
        } catch (Exception e) {
            System.err.println("关闭浏览器上下文失败: " + e.getMessage());
        }
        
        try {
            if (browser != null && browser.isConnected()) {
                browser.close();
            }
        } catch (Exception e) {
            System.err.println("关闭浏览器失败: " + e.getMessage());
        }
        
        try {
            if (playwright != null) {
                playwright.close();
            }
        } catch (Exception e) {
            System.err.println("关闭Playwright失败: " + e.getMessage());
        }
        
        System.out.println("Playwright资源已清理完毕");
    }
    
    /**
     * 获取统计信息摘要
     * 
     * @return 包含所有统计信息的JSON字符串
     */
    public String getStatisticsSummary() {
        try {
            Map<String, Object> stats = new HashMap<>();
            stats.put("clients", clients.size());
            stats.put("userConnections", new HashMap<>(userConnections));
            stats.put("domChanges", getCounter("domChanges"));
            stats.put("websocketMessages", getCounter("websocketMessages"));
            stats.put("sseEvents", getCounter("sseEvents"));
            stats.put("errors", getCounter("errors"));
            stats.put("pageLoads", getCounter("pageLoads"));
            stats.put("navigations", getCounter("navigations"));
            stats.put("consoleErrors", getCounter("consoleErrors"));
            stats.put("sseResponses", getCounter("sseResponses"));
            stats.put("timestamp", System.currentTimeMillis());
            
            if (page != null) {
                stats.put("currentPage", page.url());
            }
            
            return JSON.toJSONString(stats);
        } catch (Exception e) {
            System.err.println("获取统计信息失败: " + e.getMessage());
            e.printStackTrace();
            return "{\"error\":\"获取统计信息失败\"}";
        }
    }
    
    /**
     * 重置所有统计计数器
     */
    public void resetAllCounters() {
        try {
            String[] counters = {"domChanges", "websocketMessages", "sseEvents", "errors", "pageLoads", "navigations", "consoleErrors", "sseResponses"};
            for (String counter : counters) {
                resetCounter(counter);
            }
            System.out.println("所有统计计数器已重置");
        } catch (Exception e) {
            System.err.println("重置统计计数器失败: " + e.getMessage());
            e.printStackTrace();
        }
    }
    
    /**
     * 检查Playwright实例是否有效
     * 
     * @return 如果Playwright实例有效返回true，否则返回false
     */
    private boolean isPlaywrightInstanceValid() {
        try {
            return playwright != null && browser != null && page != null && !page.isClosed();
        } catch (Exception e) {
            System.err.println("检查Playwright实例状态失败: " + e.getMessage());
            return false;
        }
    }
    
    /**
     * 安全地获取当前页面URL
     * 
     * @return 当前页面URL，如果无法获取则返回空字符串
     */
    private String getCurrentPageUrl() {
        try {
            return page != null ? page.url() : "";
        } catch (Exception e) {
            System.err.println("获取当前页面URL失败: " + e.getMessage());
            return "";
        }
    }

}