package pangea.hiagent.tools;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.web.util.HtmlUtils;

/**
 * 字符串处理工具类
 * 提供字符串处理和转换功能
 */
@Slf4j
@Component
public class StringProcessingTools {

    
    @Tool(description = "将文本转换为大写")
    public String toUpperCase(String text) {
        log.debug("执行文本转大写操作");
        if (text == null) {
            String errorMsg = "输入文本不能为空";
            log.warn(errorMsg);
            return errorMsg;
        }
        String result = text.toUpperCase();
        log.debug("转换结果: {}", result);
        return result;
    }
    
    @Tool(description = "将文本转换为小写")
    public String toLowerCase(String text) {
        log.debug("执行文本转小写操作");
        if (text == null) {
            String errorMsg = "输入文本不能为空";
            log.warn(errorMsg);
            return errorMsg;
        }
        String result = text.toLowerCase();
        log.debug("转换结果: {}", result);
        return result;
    }
    
    @Tool(description = "反转文本")
    public String reverseText(String text) {
        log.debug("执行文本反转操作");
        if (text == null) {
            String errorMsg = "输入文本不能为空";
            log.warn(errorMsg);
            return errorMsg;
        }
        String result = new StringBuilder(text).reverse().toString();
        log.debug("反转结果: {}", result);
        return result;
    }
    
    @Tool(description = "统计文本中的字符数量")
    public int countCharacters(String text) {
        log.debug("执行字符统计操作");
        if (text == null) {
            log.warn("输入文本不能为空");
            return 0;
        }
        int count = text.length();
        log.debug("字符数量: {}", count);
        return count;
    }
    
    @Tool(description = "安全的字符串处理，防止XSS攻击")
    public String sanitizeString(String input) {
        log.debug("安全处理字符串，长度: {}", input != null ? input.length() : 0);
        if (input == null) {
            String errorMsg = "输入无效：字符串不能为空";
            log.warn("输入文本不能为空");
            return errorMsg;
        }
        
        if (input.length() > 1000) {
            String errorMsg = "输入无效：字符串长度不能超过1000个字符";
            log.warn("输入字符串长度超过限制: {}", input.length());
            return errorMsg;
        }
        
        // 清理输入，防止XSS攻击
        String cleanInput = HtmlUtils.htmlEscape(input);
        String result = "处理后的字符串：" + cleanInput;
        log.debug("处理后字符串长度: {}", cleanInput.length());
        return result;
    }
}