package pangea.hiagent.service;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.env.Environment;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import pangea.hiagent.model.User;
import pangea.hiagent.repository.UserRepository;
import pangea.hiagent.utils.JwtUtil;

import java.util.Arrays;

/**
 * 认证服务类
 * 负责用户的注册、登录和认证逻辑
 */
@Slf4j
@Service
public class AuthService {
    
    private final UserRepository userRepository;
    private final PasswordEncoder passwordEncoder;
    private final JwtUtil jwtUtil;
    private final Environment environment;
    
    public AuthService(UserRepository userRepository, PasswordEncoder passwordEncoder, JwtUtil jwtUtil, Environment environment) {
        this.userRepository = userRepository;
        this.passwordEncoder = passwordEncoder;
        this.jwtUtil = jwtUtil;
        this.environment = environment;
    }
    
    /**
     * 用户注册
     */
    @Transactional
    public User register(String username, String password, String email) {
        log.info("用户注册: {}", username);
        
        // 检查用户名是否存在
        LambdaQueryWrapper<User> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(User::getUsername, username);
        User existingUser = userRepository.selectOne(wrapper);
        
        if (existingUser != null) {
            throw new RuntimeException("用户名已存在");
        }
        
        // 注意：生产环境中不应记录密码相关信息
        log.debug("用户 {} 注册请求已接收", username);
        // 创建新用户
        User user = User.builder()
                .username(username)
                .password(passwordEncoder.encode(password))
                .email(email)
                .status("active")
                .role("user")
                .build();
        
        userRepository.insert(user);
        return user;
    }
    
    /**
     * 用户登录
     */
    public String login(String username, String password) {
        log.info("用户登录: {}", username);
        
        // 查询用户
        LambdaQueryWrapper<User> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(User::getUsername, username);
        User user = userRepository.selectOne(wrapper);
        
        if (user == null) {
            log.warn("登录失败: 用户 {} 不存在", username);
            throw new RuntimeException("用户不存在");
        }
        
        // 检查是否为开发环境，如果是则允许任意密码
        boolean isDevEnvironment = Arrays.asList(environment.getActiveProfiles()).contains("dev") || 
                                  Arrays.asList(environment.getDefaultProfiles()).contains("default");
        
        if (isDevEnvironment) {
            log.info("开发环境: 跳过密码验证");
        } else {
            // 验证密码
            // 注意：生产环境中不应记录密码相关信息
            // 使用BCryptPasswordEncoder验证密码
            boolean passwordMatch = passwordEncoder.matches(password, user.getPassword());
            
            if (!passwordMatch) {
                log.warn("登录失败: 用户 {} 密码错误", username);
                throw new RuntimeException("密码错误");
            }
        }
        
        // 检查用户状态
        if (!"active".equals(user.getStatus())) {
            log.warn("登录失败: 用户 {} 已被禁用", username);
            throw new RuntimeException("用户已禁用");
        }
        
        // 更新最后登录时间
        user.setLastLoginTime(System.currentTimeMillis());
        userRepository.updateById(user);
        
        // 生成Token
        String token = jwtUtil.generateToken(user.getId());
        log.info("用户 {} 登录成功，生成Token: {}", username, token);
        return token;
    }
    
    /**
     * 获取用户信息
     */
    public User getUserById(String userId) {
        return userRepository.selectById(userId);
    }
    
    /**
     * 验证Token
     */
    public boolean validateToken(String token) {
        return jwtUtil.validateToken(token);
    }
    
    /**
     * 从Token获取用户ID
     */
    public String getUserIdFromToken(String token) {
        return jwtUtil.getUserIdFromToken(token);
    }
}