package pangea.hiagent.auth;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;
import pangea.hiagent.model.AuthMode;
import pangea.hiagent.model.OAuth2Account;
import pangea.hiagent.model.OAuth2Provider;
import pangea.hiagent.model.User;
import pangea.hiagent.repository.UserRepository;
import pangea.hiagent.repository.OAuth2AccountRepository;
import pangea.hiagent.repository.OAuth2ProviderRepository;
import pangea.hiagent.utils.JwtUtil;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

/**
 * OAuth2.0 授权码模式认证策略实现
 * 支持标准 OAuth2.0 授权码流程（Authorization Code Grant）
 */
@Slf4j
@Component
public class OAuth2AuthenticationStrategy implements AuthenticationStrategy {
    
    private final RestTemplate restTemplate;
    private final ObjectMapper objectMapper;
    private final UserRepository userRepository;
    private final OAuth2AccountRepository oauth2AccountRepository;
    private final OAuth2ProviderRepository oauth2ProviderRepository;
    private final PasswordEncoder passwordEncoder;
    private final JwtUtil jwtUtil;
    
    public OAuth2AuthenticationStrategy(RestTemplate restTemplate, ObjectMapper objectMapper,
                                       UserRepository userRepository,
                                       OAuth2AccountRepository oauth2AccountRepository,
                                       OAuth2ProviderRepository oauth2ProviderRepository,
                                       PasswordEncoder passwordEncoder, JwtUtil jwtUtil) {
        this.restTemplate = restTemplate;
        this.objectMapper = objectMapper;
        this.userRepository = userRepository;
        this.oauth2AccountRepository = oauth2AccountRepository;
        this.oauth2ProviderRepository = oauth2ProviderRepository;
        this.passwordEncoder = passwordEncoder;
        this.jwtUtil = jwtUtil;
    }
    
    @Override
    public String getName() {
        return "OAuth2 Authorization Code Strategy";
    }
    
    @Override
    public boolean supports(String authMode) {
        return AuthMode.OAUTH2_AUTHORIZATION_CODE.getCode().equals(authMode);
    }
    
    /**
     * 执行 OAuth2 认证
     * @param credentials 包含以下字段：
     *   - authorizationCode: OAuth2 授权码
     *   - providerName: OAuth2 提供者名称
     *   - state: 防CSRF令牌（可选）
     * @return JWT Token
     */
    @Override
    public String authenticate(Map<String, Object> credentials) {
        String authorizationCode = (String) credentials.get("authorizationCode");
        String providerName = (String) credentials.get("providerName");
        String state = (String) credentials.get("state");
        
        if (authorizationCode == null || authorizationCode.trim().isEmpty()) {
            log.warn("OAuth2 认证失败: 授权码为空");
            throw new RuntimeException("授权码不能为空");
        }
        
        if (providerName == null || providerName.trim().isEmpty()) {
            log.warn("OAuth2 认证失败: 提供者名称为空");
            throw new RuntimeException("提供者名称不能为空");
        }
        
        log.info("执行 OAuth2 认证: providerName={}, authorizationCode={}", providerName, authorizationCode);
        
        try {
            // 获取 OAuth2 提供者配置
            LambdaQueryWrapper<OAuth2Provider> providerWrapper = new LambdaQueryWrapper<>();
            providerWrapper.eq(OAuth2Provider::getProviderName, providerName)
                          .eq(OAuth2Provider::getEnabled, 1);
            OAuth2Provider provider = oauth2ProviderRepository.selectOne(providerWrapper);
            
            if (provider == null) {
                log.error("OAuth2 认证失败: 未找到配置的提供者 {}", providerName);
                throw new RuntimeException("未找到配置的 OAuth2 提供者");
            }
            
            // 使用授权码交换访问令牌
            String accessToken = exchangeCodeForToken(provider, authorizationCode);
            
            // 获取用户信息
            Map<String, Object> userInfo = fetchUserInfo(provider, accessToken);
            
            // 获取或创建用户
            User user = findOrCreateUser(provider, userInfo);
            
            // 保存或更新 OAuth2 账户关联
            saveOAuth2Account(user, provider, userInfo, accessToken);
            
            // 生成 JWT Token
            String token = jwtUtil.generateToken(user.getId());
            log.info("OAuth2 认证成功，用户: {}, 提供者: {}, 生成Token: {}", user.getUsername(), providerName, token);
            
            return token;
            
        } catch (Exception e) {
            log.error("OAuth2 认证过程中出错: providerName={}, authorizationCode={}, 错误堆栈: ", providerName, authorizationCode, e);
            throw new RuntimeException("OAuth2 认证失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 使用授权码交换访问令牌
     */
    private String exchangeCodeForToken(OAuth2Provider provider, String authorizationCode) {
        try {
            log.debug("开始令牌交换: 提供者={}, tokenUrl={}", provider.getProviderName(), provider.getTokenUrl());
            
            Map<String, String> body = new HashMap<>();
            body.put("grant_type", "authorization_code");
            body.put("code", authorizationCode);
            body.put("client_id", provider.getClientId());
            body.put("client_secret", provider.getClientSecret());
            body.put("redirect_uri", provider.getRedirectUri());
            
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);
            
            // 构造表单数据
            String formBody = body.entrySet().stream()
                .map(e -> e.getKey() + "=" + e.getValue())
                .reduce((a, b) -> a + "&" + b)
                .orElse("");
            
            HttpEntity<String> request = new HttpEntity<>(formBody, headers);
            
            ResponseEntity<String> response = restTemplate.postForEntity(provider.getTokenUrl(), request, String.class);
            
            if (response.getStatusCode().is2xxSuccessful() && response.getBody() != null) {
                JsonNode jsonNode = objectMapper.readTree(response.getBody());
                String accessToken = jsonNode.get("access_token").asText();
                log.debug("令牌交换成功: 提供者={}, accessToken={}", provider.getProviderName(), 
                         accessToken.substring(0, Math.min(20, accessToken.length())) + "...");
                return accessToken;
            } else {
                log.error("令牌交换失败: 提供者={}, statusCode={}, responseBody={}", 
                         provider.getProviderName(), response.getStatusCode(), response.getBody());
                throw new RuntimeException("令牌交换失败");
            }
            
        } catch (IOException e) {
            log.error("令牌交换过程中出错: 提供者={}, 错误堆栈: ", provider.getProviderName(), e);
            throw new RuntimeException("令牌交换异常: " + e.getMessage(), e);
        }
    }
    
    /**
     * 从 OAuth2 提供者获取用户信息
     */
    private Map<String, Object> fetchUserInfo(OAuth2Provider provider, String accessToken) {
        try {
            log.debug("开始获取用户信息: 提供者={}, userinfoUrl={}", provider.getProviderName(), provider.getUserinfoUrl());
            
            HttpHeaders headers = new HttpHeaders();
            headers.setBearerAuth(accessToken);
            headers.set("Accept", MediaType.APPLICATION_JSON_VALUE);
            
            HttpEntity<String> request = new HttpEntity<>(headers);
            
            ResponseEntity<String> response = restTemplate.postForEntity(provider.getUserinfoUrl(), request, String.class);
            
            if (response.getStatusCode().is2xxSuccessful() && response.getBody() != null) {
                Map<String, Object> userInfo = objectMapper.readValue(response.getBody(), Map.class);
                log.debug("获取用户信息成功: 提供者={}", provider.getProviderName());
                return userInfo;
            } else {
                log.error("获取用户信息失败: 提供者={}, statusCode={}", provider.getProviderName(), response.getStatusCode());
                throw new RuntimeException("获取用户信息失败");
            }
            
        } catch (IOException e) {
            log.error("获取用户信息过程中出错: 提供者={}, 错误堆栈: ", provider.getProviderName(), e);
            throw new RuntimeException("获取用户信息异常: " + e.getMessage(), e);
        }
    }
    
    /**
     * 查找或创建用户
     */
    private User findOrCreateUser(OAuth2Provider provider, Map<String, Object> userInfo) {
        String remoteUserId = extractRemoteUserId(provider, userInfo);
        String remoteEmail = (String) userInfo.get("email");
        String remoteUsername = (String) userInfo.get("name");
        
        // 查找现有的 OAuth2 账户
        LambdaQueryWrapper<OAuth2Account> accountWrapper = new LambdaQueryWrapper<>();
        accountWrapper.eq(OAuth2Account::getProviderName, provider.getProviderName())
                     .eq(OAuth2Account::getRemoteUserId, remoteUserId);
        OAuth2Account existingAccount = oauth2AccountRepository.selectOne(accountWrapper);
        
        if (existingAccount != null) {
            // 返回已关联的用户
            return userRepository.selectById(existingAccount.getUserId());
        }
        
        // 如果用户邮箱存在，尝试关联现有用户
        if (remoteEmail != null && !remoteEmail.isEmpty()) {
            LambdaQueryWrapper<User> userWrapper = new LambdaQueryWrapper<>();
            userWrapper.eq(User::getEmail, remoteEmail);
            User existingUser = userRepository.selectOne(userWrapper);
            if (existingUser != null) {
                return existingUser;
            }
        }
        
        // 创建新用户
        String newUsername = provider.getProviderName() + "_" + remoteUserId;
        String randomPassword = generateRandomPassword();
        
        User newUser = User.builder()
                .username(newUsername)
                .password(passwordEncoder.encode(randomPassword))
                .email(remoteEmail)
                .nickname(remoteUsername)
                .status("active")
                .role("user")
                .build();
        
        userRepository.insert(newUser);
        log.info("创建新用户: username={}, 来自提供者: {}", newUsername, provider.getProviderName());
        
        return newUser;
    }
    
    /**
     * 保存或更新 OAuth2 账户关联
     */
    private void saveOAuth2Account(User user, OAuth2Provider provider, Map<String, Object> userInfo, String accessToken) {
        String remoteUserId = extractRemoteUserId(provider, userInfo);
        
        LambdaQueryWrapper<OAuth2Account> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(OAuth2Account::getUserId, user.getId())
              .eq(OAuth2Account::getProviderName, provider.getProviderName());
        OAuth2Account existingAccount = oauth2AccountRepository.selectOne(wrapper);
        
        long now = System.currentTimeMillis();
        
        try {
            String profileData = objectMapper.writeValueAsString(userInfo);
            
            if (existingAccount != null) {
                // 更新现有账户
                existingAccount.setAccessToken(accessToken);
                existingAccount.setRemoteUsername((String) userInfo.get("name"));
                existingAccount.setRemoteEmail((String) userInfo.get("email"));
                existingAccount.setProfileData(profileData);
                existingAccount.setLastLoginAt(now);
                oauth2AccountRepository.updateById(existingAccount);
                log.debug("更新 OAuth2 账户: userId={}, providerName={}", user.getId(), provider.getProviderName());
            } else {
                // 创建新账户关联
                OAuth2Account newAccount = OAuth2Account.builder()
                        .userId(user.getId())
                        .providerName(provider.getProviderName())
                        .remoteUserId(remoteUserId)
                        .remoteUsername((String) userInfo.get("name"))
                        .remoteEmail((String) userInfo.get("email"))
                        .accessToken(accessToken)
                        .scope(provider.getScope())
                        .profileData(profileData)
                        .linkedAt(now)
                        .lastLoginAt(now)
                        .createdAt(now)
                        .updatedAt(now)
                        .deleted(0)
                        .build();
                
                newAccount.setId(UUID.randomUUID().toString());
                oauth2AccountRepository.insert(newAccount);
                log.debug("创建新 OAuth2 账户关联: userId={}, providerName={}", user.getId(), provider.getProviderName());
            }
        } catch (IOException e) {
            log.error("保存 OAuth2 账户时出错: userId={}, providerName={}, 错误堆栈: ", 
                     user.getId(), provider.getProviderName(), e);
            throw new RuntimeException("保存 OAuth2 账户失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 提取远程用户 ID
     */
    private String extractRemoteUserId(OAuth2Provider provider, Map<String, Object> userInfo) {
        // 默认使用 "id" 或 "sub" 字段，可根据具体提供者调整
        Object id = userInfo.get("id");
        if (id == null) {
            id = userInfo.get("sub");
        }
        return id != null ? id.toString() : userInfo.get("email").toString();
    }
    
    /**
     * 生成随机密码
     */
    private String generateRandomPassword() {
        return UUID.randomUUID().toString().replace("-", "").substring(0, 16);
    }
    
    @Override
    public boolean verify(String token) {
        return jwtUtil.validateToken(token);
    }
}
