package pangea.hiagent.common.config;

import com.baomidou.mybatisplus.core.handlers.MetaObjectHandler;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.reflection.MetaObject;
import org.springframework.stereotype.Component;
import pangea.hiagent.common.utils.UserUtils;
import java.time.LocalDateTime;

/**
 * MyBatis Plus自动填充配置类
 * 用于自动填充实体类中的公共字段
 */
@Slf4j
@Component
public class MetaObjectHandlerConfig implements MetaObjectHandler {

    /**
     * 插入时自动填充
     * @param metaObject 元对象
     */
    @Override
    public void insertFill(MetaObject metaObject) {
        log.debug("开始执行插入自动填充");
        
        // 自动填充创建时间
        if (metaObject.hasSetter("createdAt")) {
            Object createdAt = getFieldValByName("createdAt", metaObject);
            if (createdAt == null) {
                this.strictInsertFill(metaObject, "createdAt", LocalDateTime.class, LocalDateTime.now());
                log.debug("自动填充createdAt字段");
            }
        }
        
        // 自动填充更新时间
        if (metaObject.hasSetter("updatedAt")) {
            Object updatedAt = getFieldValByName("updatedAt", metaObject);
            if (updatedAt == null) {
                this.strictInsertFill(metaObject, "updatedAt", LocalDateTime.class, LocalDateTime.now());
                log.debug("自动填充updatedAt字段");
            }
        }
        
        // 自动填充创建人
        if (metaObject.hasSetter("createdBy")) {
            Object createdBy = getFieldValByName("createdBy", metaObject);
            if (createdBy == null) {
                String userId = getCurrentUserIdWithContext();
                if (userId != null) {
                    this.strictInsertFill(metaObject, "createdBy", String.class, userId);
                    log.debug("自动填充createdBy字段: {}", userId);
                } else {
                    log.warn("无法获取当前用户ID，createdBy字段未填充");
                }
            }
        }
        
        // 自动填充更新人
        if (metaObject.hasSetter("updatedBy")) {
            Object updatedBy = getFieldValByName("updatedBy", metaObject);
            if (updatedBy == null) {
                String userId = getCurrentUserIdWithContext();
                if (userId != null) {
                    this.strictInsertFill(metaObject, "updatedBy", String.class, userId);
                    log.debug("自动填充updatedBy字段: {}", userId);
                } else {
                    log.warn("无法获取当前用户ID，updatedBy字段未填充");
                }
            }
        }
    }

    /**
     * 更新时自动填充
     * @param metaObject 元对象
     */
    @Override
    public void updateFill(MetaObject metaObject) {
        log.debug("开始执行更新自动填充");
        
        // 自动填充更新时间
        if (metaObject.hasSetter("updatedAt")) {
            // 更新时总是设置updatedAt字段
            this.strictUpdateFill(metaObject, "updatedAt", LocalDateTime.class, LocalDateTime.now());
            log.debug("自动填充updatedAt字段");
        }
        
        // 自动填充更新人
        if (metaObject.hasSetter("updatedBy")) {
            Object updatedBy = getFieldValByName("updatedBy", metaObject);
            // 如果updatedBy为空或者需要强制更新，则填充当前用户ID
            if (updatedBy == null) {
                String userId = getCurrentUserIdWithContext();
                if (userId != null) {
                    this.strictUpdateFill(metaObject, "updatedBy", String.class, userId);
                    log.debug("自动填充updatedBy字段: {}", userId);
                } else {
                    log.warn("无法获取当前用户ID，updatedBy字段未填充");
                }
            }
        }
    }
    
    /**
     * 获取当前用户ID，支持异步线程上下文
     * 该方法支持以下场景：
     * 1. 优先从ThreadLocal获取（支持异步线程）
     * 2. 从SecurityContext获取（支持同步请求和AsyncUserContextDecorator传播）
     * 3. 从请求中解析Token获取用户ID
     * 
     * @return 用户ID，如果无法获取则返回null
     */
    private String getCurrentUserIdWithContext() {
        try {
            // 直接调用UserUtils.getCurrentUserIdStatic()，该方法已经包含了所有获取用户ID的方式
            // 并且优先从ThreadLocal获取，支持异步线程
            String userId = UserUtils.getCurrentUserIdStatic();
            if (userId != null) {
                log.debug("成功获取用户ID: {}", userId);
                return userId;
            }
            
            log.warn("无法通过任何方式获取当前用户ID，createdBy/updatedBy字段将不被填充");
            return null;
        } catch (Exception e) {
            log.error("获取用户ID时发生异常", e);
            return null;
        }
    }
}