package pangea.hiagent.agent.processor;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.model.ChatModel;
import org.springframework.ai.chat.model.StreamingChatModel;
import org.springframework.ai.chat.prompt.Prompt;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import pangea.hiagent.model.Agent;
import pangea.hiagent.rag.RagService;
import pangea.hiagent.web.dto.AgentRequest;

import java.util.function.Consumer;

/**
 * 普通Agent处理器实现类
 * 职责：处理普通Agent的请求，调用LLM模型生成回复，支持RAG增强
 */
@Slf4j
@Service
public class NormalAgentProcessor extends BaseAgentProcessor {
    
    @Autowired(required = false)
    private RagService ragService;
    
    @Override
    public String processRequest(Agent agent, AgentRequest request, String userId) {
        log.info("使用普通Agent处理请求");
        
        try {
            // 处理请求的通用前置逻辑
            String ragResponse = handlePreProcessing(agent, request.getUserMessage(), userId, ragService, null);
            if (ragResponse != null) {
                return ragResponse;
            }
            
            // 获取系统提示词
            String systemPrompt = getSystemPrompt(agent);
            
            // 构建Prompt，使用Agent配置的历史记录长度
            int historyLength = getHistoryLength(agent);
            String sessionId = generateSessionId(agent, userId);
            Prompt prompt = buildPrompt(systemPrompt, historyLength, sessionId);
            
            // 根据Agent配置获取对应的ChatModel
            ChatModel chatModel = agentService.getChatModelForAgent(agent);
            
            if (chatModel == null) {
                log.error("无法获取Agent的聊天模型");
                return "[错误] 无法获取Agent的聊天模型，请检查模型配置";
            }
            
            // 使用对应模型进行调用
            org.springframework.ai.chat.model.ChatResponse chatResponse = chatModel.call(prompt);
            
            // 提取助理回复
            String responseContent = "";
            if (chatResponse.getResult() != null && chatResponse.getResult().getOutput() != null) {
                responseContent = chatResponse.getResult().getOutput().getText();
                log.info("模型调用成功，响应内容长度: {}", responseContent.length());
            } else {
                log.warn("模型返回空响应");
            }
            
            // 将助理回复添加到ChatMemory
            addAssistantMessageToMemory(sessionId, responseContent);
            
            return responseContent;
        } catch (Exception e) {
            return agentErrorHandler.handleSyncError(e, "模型调用失败");
        }
    }
    
    @Override
    public void processStreamRequest(AgentRequest request, Agent agent, String userId, Consumer<String> tokenConsumer) {
        try {
            log.info("使用普通Agent处理流式请求");
            
            // 处理请求的通用前置逻辑
            String ragResponse = handlePreProcessing(agent, request.getUserMessage(), userId, ragService, tokenConsumer);
            if (ragResponse != null) {
                return;
            }
            
            // 获取系统提示词
            String systemPrompt = getSystemPrompt(agent);
            
            // 构建Prompt，使用Agent配置的历史记录长度
            int historyLength = getHistoryLength(agent);
            String sessionId = generateSessionId(agent, userId);
            Prompt prompt = buildPrompt(systemPrompt, historyLength, sessionId);
            
            // 获取流式模型
            StreamingChatModel streamingChatModel = getStreamingChatModel(agent);
            if (streamingChatModel == null) {
                log.warn("当前模型不支持流式输出");
                handleModelNotSupportStream(tokenConsumer);
                return;
            }
            
            // 流式处理
            handleStreamingResponse(tokenConsumer, prompt, streamingChatModel, sessionId);
        } catch (Exception e) {
            agentErrorHandler.handleStreamError(e, tokenConsumer, "普通Agent流式处理失败");
            agentErrorHandler.ensureCompletionCallback(tokenConsumer, "处理请求时发生错误: " + e.getMessage());
        }
    }
    
    /**
     * 处理模型不支持流式输出的情况
     * 
     * @param tokenConsumer token消费者
     */
    private void handleModelNotSupportStream(Consumer<String> tokenConsumer) {
        String errorMessage = "[错误] 当前模型不支持流式输出";
        // 发送错误信息
        agentErrorHandler.sendErrorMessage(tokenConsumer, errorMessage);
        // 确保在异常情况下也调用完成回调
        agentErrorHandler.ensureCompletionCallback(tokenConsumer, errorMessage);
    }
    
    @Override
    public String getProcessorType() {
        return "normal";
    }
}