package pangea.hiagent.controller;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;
import pangea.hiagent.core.AgentChatService;
import pangea.hiagent.workpanel.SseEventManager;
import pangea.hiagent.dto.AgentRequest;
import pangea.hiagent.dto.ChatRequest;
import pangea.hiagent.model.Agent;
import pangea.hiagent.service.AgentService;
import pangea.hiagent.utils.AsyncUserContextDecorator;
import pangea.hiagent.utils.UserUtils;

import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * Agent 对话控制器
 * 提供Agent的对话交互功能，支持流式输出
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/agent")
public class AgentChatController {
    
    @Autowired
    private AgentService agentService;
    
    @Autowired
    private AgentChatService agentChatService;
    
    @Autowired
    private SseEventManager sseEventManager;
    
    private final ExecutorService executorService = new ThreadPoolExecutor(
            10, // 核心线程数
            50, // 最大线程数
            60L, // 空闲线程存活时间
            TimeUnit.SECONDS, // 时间单位
            new LinkedBlockingQueue<>(100), // 任务队列
            new ThreadPoolExecutor.CallerRunsPolicy() // 拒绝策略
    );
    
    /**
     * 流式对话接口
     * 支持 SSE (Server-Sent Events) 格式的流式输出
     * 
     * @param agentId Agent ID
     * @param chatRequest 对话请求
     * @return SSE emitter
     */
    @PostMapping("/chat-stream")
    public SseEmitter chatStream(
            @RequestParam String agentId,
            @RequestBody ChatRequest chatRequest,
            HttpServletResponse response) {
        
        // 将ChatRequest转换为AgentRequest
        AgentRequest request = new AgentRequest();
        request.setUserMessage(chatRequest.getMessage());
        
        log.info("开始处理流式对话请求，AgentId: {}, 用户消息: {}", agentId, request.getUserMessage());
        
        // 检查用户消息是否为空
        if (request.getUserMessage() == null || request.getUserMessage().trim().isEmpty()) {
            log.error("用户消息不能为空");
            SseEmitter emitter = new SseEmitter(300000L);
            try {
                // 检查响应是否已经提交
                if (!response.isCommitted()) {
                    Map<String, Object> errorData = new HashMap<>();
                    errorData.put("error", "用户消息不能为空");
                    errorData.put("timestamp", System.currentTimeMillis());
                    emitter.send(SseEmitter.event()
                            .name("error")
                            .data(errorData)
                            .build());
                    emitter.complete();
                } else {
                    log.warn("响应已经提交，无法发送用户消息为空错误");
                    emitter.complete();
                }
            } catch (IOException e) {
                log.error("发送用户消息为空错误失败", e);
                emitter.completeWithError(e);
            }
            return emitter;
        }
        
        String userId = getCurrentUserId();
        if (userId == null) {
            log.error("用户未认证");
            SseEmitter emitter = new SseEmitter(300000L);
            try {
                // 检查响应是否已经提交
                if (!response.isCommitted()) {
                    Map<String, Object> errorData = new HashMap<>();
                    errorData.put("error", "用户未认证，请重新登录");
                    errorData.put("timestamp", System.currentTimeMillis());
                    emitter.send(SseEmitter.event()
                            .name("error")
                            .data(errorData)
                            .build());
                    emitter.complete();
                } else {
                    log.warn("响应已经提交，无法发送未认证错误");
                    emitter.complete();
                }
            } catch (IOException e) {
                log.error("发送未认证错误失败", e);
                // 检查响应是否已经提交
                if (!response.isCommitted()) {
                    emitter.completeWithError(e);
                } else {
                    // 响应已提交，只能简单完成
                    emitter.complete();
                }
            } catch (IllegalStateException e) {
                log.warn("Emitter已经完成: {}", e.getMessage());
                // emitter已经完成，无需进一步操作
            }
            return emitter;
        }
        
        // 创建 SSE emitter
        SseEmitter emitter = new SseEmitter(300000L); // 5分钟超时，与前端保持一致
        
        // 设置超时回调
        emitter.onTimeout(() -> {
            log.warn("SSE连接超时，AgentId: {}, 用户ID: {}", agentId, userId);
            try {
                emitter.complete();
            } catch (IllegalStateException e) {
                log.warn("Emitter已经完成: {}", e.getMessage());
            } catch (Exception e) {
                log.error("关闭SSE连接时发生错误", e);
            }
        });
        
        // 异步处理对话，避免阻塞HTTP连接
        // 使用AsyncUserContextDecorator包装任务以传播用户上下文
        executorService.execute(AsyncUserContextDecorator.wrapWithContext(() -> {
            try {
                // 获取Agent信息
                Agent agent = agentService.getAgent(agentId);
                if (agent == null) {
                    log.error("Agent不存在: {}", agentId);
                    // 检查响应是否已经提交
                    if (!response.isCommitted()) {
                        sseEventManager.sendError(emitter, "Agent不存在");
                    } else {
                        log.warn("响应已经提交，无法发送Agent不存在错误");
                        try {
                            emitter.complete();
                        } catch (Exception ex) {
                            log.warn("关闭emitter时发生异常", ex);
                        }
                    }
                    return;
                }
                
                // 检查权限（可选）
                if (!agent.getOwner().equals(userId) && !isAdmin(userId)) {
                    log.warn("用户 {} 无权访问 Agent {}", userId, agentId);
                    // 检查响应是否已经提交
                    if (!response.isCommitted()) {
                        sseEventManager.sendError(emitter, "无权限访问该Agent");
                    } else {
                        log.warn("响应已经提交，无法发送权限错误");
                        try {
                            emitter.complete();
                        } catch (Exception ex) {
                            log.warn("关闭emitter时发生异常", ex);
                        }
                    }
                    return;
                }
                
                // 根据Agent配置选择处理方式
                if (agent.getEnableReAct() != null && agent.getEnableReAct()) {
                    // 使用ReAct Agent处理
                    log.info("使用ReAct Agent处理对话");
                    agentChatService.processReActAgentStreamWithSse(
                            request,
                            agent,
                            userId,
                            emitter,
                            sseEventManager,
                            null, // ReActService 会从 Spring 容器中注入
                            (agentParam, requestParam, userIdParam, responseContent, emitterParam, isCompleted, sseEventManagerParam) -> {
                                // 保存对话记录和发送完成事件
                                try {
                                    // 使用现有的sendEvent方法发送完成事件
                                    Map<String, Object> completeData = new HashMap<>();
                                    completeData.put("message", "对话完成");
                                    completeData.put("type", "complete");
                                    completeData.put("fullText", responseContent); // 添加完整文本内容
                                    sseEventManagerParam.sendEvent(emitterParam, "complete", completeData, isCompleted);
                                    // 确保标记为已完成
                                    isCompleted.set(true);
                                } catch (IOException e) {
                                    log.error("发送完成事件失败", e);
                                    isCompleted.set(true); // 出错时也标记为已完成
                                }
                            }
                    );
                } else {
                    // 使用普通Agent处理
                    log.info("使用普通Agent处理对话");
                    agentChatService.processNormalAgentStreamWithSse(
                            request,
                            agent,
                            userId,
                            emitter,
                            sseEventManager,
                            (token, fullText) -> {
                                // 创建token事件数据
                                java.util.Map<String, Object> data = new java.util.HashMap<>();
                                data.put("type", "token");
                                data.put("content", token);
                                return data;
                            },
                            (agentParam, requestParam, userIdParam, responseContent, emitterParam, isCompleted, sseEventManagerParam) -> {
                                // 保存对话记录和发送完成事件
                                try {
                                    // 使用现有的sendEvent方法发送完成事件
                                    Map<String, Object> completeData = new HashMap<>();
                                    completeData.put("message", "对话完成");
                                    completeData.put("type", "complete");
                                    completeData.put("fullText", responseContent); // 添加完整文本内容
                                    sseEventManagerParam.sendEvent(emitterParam, "complete", completeData, isCompleted);
                                    // 确保标记为已完成
                                    isCompleted.set(true);
                                } catch (IOException e) {
                                    log.error("发送完成事件失败", e);
                                    isCompleted.set(true); // 出错时也标记为已完成
                                }
                            }
                    );
                }
                
            } catch (Exception e) {
                log.error("处理流式对话失败", e);
                // 检查响应是否已经提交
                if (!response.isCommitted()) {
                    // sendError方法内部已经处理了所有可能的异常，无需再次捕获IOException
                    sseEventManager.sendError(emitter, "处理请求时发生错误: " + e.getMessage());
                } else {
                    log.warn("响应已经提交，无法发送处理错误: {}", e.getMessage());
                    try {
                        emitter.complete();
                    } catch (Exception ex) {
                        log.warn("关闭emitter时发生异常", ex);
                    }
                }
            }
        }));
        
        // 设置 emitter 的回调
        emitter.onCompletion(() -> log.debug("SSE连接完成"));
        emitter.onTimeout(() -> {
            log.warn("SSE连接超时，准备关闭连接");
            // complete方法内部已经处理了所有可能的异常，无需再次捕获IOException
            try {
                emitter.complete();
                log.info("SSE连接已成功关闭");
            } catch (IllegalStateException e) {
                log.warn("Emitter已经完成: {}", e.getMessage());
            } catch (Exception e) {
                log.error("关闭SSE连接时发生错误", e);
            }
        });
        emitter.onError(throwable -> log.error("SSE连接错误", throwable));
        
        return emitter;
    }
    
    /**
     * 获取当前认证用户ID
     */
    private String getCurrentUserId() {
        return UserUtils.getCurrentUserId();
    }
    
    /**
     * 检查用户是否是管理员
     */
    private boolean isAdmin(String userId) {
        // 使用Spring Security获取当前认证用户ID
        String currentUserId = getCurrentUserId();
        // 这里可以添加更复杂的权限检查逻辑，比如查询数据库或配置文件
        return "admin".equals(currentUserId) || "user-001".equals(currentUserId);
    }
    
    /**
     * 用于测试的方法，暴露私有方法getCurrentUserId
     */
    public String getCurrentUserIdForTesting() {
        return getCurrentUserId();
    }
}