package pangea.hiagent.websocket;

import com.alibaba.fastjson2.JSON;
import com.microsoft.playwright.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.socket.*;
import org.springframework.web.socket.handler.TextWebSocketHandler;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * 优化的DOM同步WebSocket处理器（门面类）
 * 协调各个子模块完成DOM同步功能
 */
@Slf4j
public class DomSyncHandler extends org.springframework.web.socket.handler.AbstractWebSocketHandler {
    // 注入Playwright管理器
    @Autowired
    private pangea.hiagent.core.PlaywrightManager playwrightManager;
    
    // 各个子模块
    private WebSocketConnectionManager connectionManager;
    private DomSyncService domSyncService;
    private CommandProcessor commandProcessor;
    private StatisticsService statisticsService;
    private BinaryMessageSender messageSender;
    
    // 共享的数据结构
    private static final ConcurrentMap<String, Long> messageCounters = new ConcurrentHashMap<>();
    private static final ConcurrentMap<String, Long> lastCommandTimes = new ConcurrentHashMap<>();
    private static final ConcurrentMap<String, Integer> commandCounts = new ConcurrentHashMap<>();
    private static final ConcurrentMap<String, Integer> userConnections = new ConcurrentHashMap<>();
    
    public DomSyncHandler() {
        // 初始化各个子模块
        initializeSubModules();
    }
    
    /**
     * 初始化各个子模块
     */
    private void initializeSubModules() {
        // 初始化连接管理器
        connectionManager = new WebSocketConnectionManager(playwrightManager);
        
        // 初始化DOM同步服务
        domSyncService = new DomSyncService(messageCounters);
        
        // 初始化消息发送器
        messageSender = new BinaryMessageSender(connectionManager);
        
        // 初始化统计信息服务
        statisticsService = new StatisticsService(messageCounters, userConnections);
        
        // 初始化指令处理器
        commandProcessor = new CommandProcessor(lastCommandTimes, commandCounts, messageCounters);
    }
    
    /**
     * 设置PlaywrightManager
     * 此方法用于Spring注入
     */
    public void setPlaywrightManager(pangea.hiagent.core.PlaywrightManager playwrightManager) {
        this.playwrightManager = playwrightManager;
        log.debug("已设置PlaywrightManager实例");
    }

    // ===================== WebSocket生命周期方法 =====================
    /**
     * 客户端连接建立时触发
     */
    @Override
    public void afterConnectionEstablished(WebSocketSession session) {
        // 将PlaywrightManager传递给DomSyncService
        domSyncService.setPlaywrightManager(playwrightManager);
        
        // 通过连接管理器处理连接建立
        connectionManager.handleConnectionEstablished(session, domSyncService);
    }

    /**
     * 处理客户端发送的指令（如导航、点击、输入）
     */
    @Override
    public void handleTextMessage(WebSocketSession session, TextMessage message) throws Exception {
        // 从会话属性中获取用户ID（来自JWT认证）
        String userId = (String) session.getAttributes().get("userId");
        if (userId == null || userId.isEmpty()) {
            // 如果没有有效的用户ID，使用默认值
            userId = "unknown-user";
            log.warn("WebSocket消息处理缺少有效的用户认证信息，使用默认用户ID: {}", userId);
        }
        
        // 通过指令处理器处理指令
        commandProcessor.processCommand(
            message.getPayload(), 
            domSyncService.getCurrentPage(), // 获取当前页面对象
            messageSender, 
            statisticsService, 
            userId
        );
    }

    /**
     * 客户端连接关闭时触发
     */
    @Override
    public void afterConnectionClosed(WebSocketSession session, CloseStatus status) {
        // 通过连接管理器处理连接关闭
        connectionManager.handleConnectionClosed(session, domSyncService);
    }

    /**
     * 处理WebSocket传输错误
     */
    @Override
    public void handleBinaryMessage(WebSocketSession session, BinaryMessage message) throws Exception {
        // 我们不期望接收二进制消息，如果有则忽略
        log.warn("接收到意外的二进制消息，大小: {} 字节", message.getPayloadLength());
    }
    
    @Override
    public void handleTransportError(WebSocketSession session, Throwable exception) {
        // 通过连接管理器处理传输错误
        connectionManager.handleTransportError(session, exception);
    }
    
    /**
     * 销毁资源（Spring Boot关闭时调用）
     */
    public void destroy() {
        log.debug("开始清理DomSyncHandler资源...");
        
        try {
            // 销毁各个子模块
            if (connectionManager != null) {
                connectionManager.destroy();
            }
            
            if (messageSender != null) {
                messageSender.destroy();
            }
            
            // 销毁DOM同步服务
            if (domSyncService != null) {
                String userId = domSyncService.getCurrentUserId();
                domSyncService.destroy(userId, playwrightManager);
            }
        } catch (Exception e) {
            log.error("清理DomSyncHandler资源失败", e);
        }
    }
    
    /**
     * 获取统计信息摘要
     * 
     * @return 包含所有统计信息的JSON字符串
     */
    public String getStatisticsSummary() {
        try {
            if (statisticsService != null) {
                return statisticsService.getStatisticsSummary();
            }
            
            Map<String, Object> stats = new HashMap<>();
            stats.put("error", "统计服务未初始化");
            return JSON.toJSONString(stats);
        } catch (Exception e) {
            log.error("获取统计信息失败", e);
            return "{\"error\":\"获取统计信息失败\"}";
        }
    }
    
    /**
     * 重置所有统计计数器
     */
    public void resetAllCounters() {
        try {
            if (statisticsService != null) {
                statisticsService.resetAllCounters();
                log.debug("所有统计计数器已重置");
            }
        } catch (Exception e) {
            log.error("重置统计计数器失败", e);
        }
    }
    
    /**
     * 获取计数器值
     */
    public static long getCounter(String counterName) {
        return messageCounters.getOrDefault(counterName, 0L);
    }
    
    /**
     * 重置计数器
     */
    public static void resetCounter(String counterName) {
        messageCounters.put(counterName, 0L);
    }
}