package pangea.hiagent.llm;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.model.ChatModel;
import org.springframework.stereotype.Component;
import pangea.hiagent.model.LlmConfig;

/**
 * DeepSeek模型适配器
 * 实现DeepSeek模型的创建和配置（使用OpenAI兼容API）
 */
@Slf4j
@Component
public class DeepSeekModelAdapter extends AbstractOpenAiCompatibleAdapter {
    
    private static final String DEEPSEEK_DEFAULT_BASE_URL = "https://api.deepseek.com";
    
    @Override
    public ChatModel createChatModel(LlmConfig config) {
        if (!validateConfig(config)) {
            throw new IllegalArgumentException("无效的DeepSeek配置");
        }
        
        // 记录API密钥长度信息（脱敏显示）
        if (config.getApiKey() != null) {
            log.info("创建DeepSeek ChatModel，模型名称: {}, API密钥长度: {}", config.getModelName(), config.getApiKey().length());
        } else {
            log.info("创建DeepSeek ChatModel，模型名称: {}, API密钥为空", config.getModelName());
        }
        
        return createOpenAiChatModel(config, DEEPSEEK_DEFAULT_BASE_URL);
    }
    
    @Override
    public String getProviderName() {
        return "deepseek";
    }
    
    @Override
    public boolean validateConfig(LlmConfig config) {
        // 修改验证逻辑，允许在没有API密钥的情况下启动，但会给出警告
        if (config == null || !config.getEnabled()) {
            return false;
        }
        
        // 如果启用了配置但没有API密钥，允许启动但会在使用时抛出异常
        if (config.getApiKey() == null || config.getApiKey().isEmpty()) {
            log.warn("DeepSeek配置已启用但未设置API密钥，将在实际使用时抛出异常");
            return true;
        }
        
        return true;
    }
}