package pangea.hiagent.tool.impl;

import com.microsoft.playwright.*;
import com.microsoft.playwright.options.LoadState;
import com.microsoft.playwright.options.WaitUntilState;
import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import pangea.hiagent.workpanel.playwright.PlaywrightManager;

import java.util.Base64;
import java.util.List;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.function.Function;

/**
 * Playwright网页自动化工具类
 * 提供基于Playwright的网页内容抓取、交互操作、截图等功能
 */
@Slf4j
@Component
public class PlaywrightWebTools {

    // 注入Playwright管理器
    @Autowired
    private PlaywrightManager playwrightManager;
    
    // 浏览器实例（从Playwright管理器获取）
    private Browser browser;
    
    /**
     * 延迟初始化浏览器实例引用
     */
    private void initializeIfNeeded() {
        if (browser == null) {
            try {
                log.info("正在初始化Playwright网页工具...");
                // 从Playwright管理器获取共享的浏览器实例
                this.browser = playwrightManager.getBrowser();
                log.info("Playwright网页工具初始化成功");
            } catch (Exception e) {
                log.error("Playwright网页工具初始化失败: ", e);
            }
        }
    }
    
    /**
     * 处理失败情况
     * @param message 错误消息
     * @return 错误结果
     */
    private String handleFailure(String message) {
        log.warn(message);
        return "抱歉，" + message;
    }

    /**
     * 在页面上执行操作的通用方法
     * @param url 网页地址
     * @param pageFunction 页面操作函数
     * @return 操作结果
     */
    private String executeWithPage(String url, Function<Page, String> pageFunction) {
        initializeIfNeeded();
        try (BrowserContext context = browser.newContext();
             Page page = context.newPage()) {
            // 导航到指定URL，等待页面加载完成
            page.navigate(url, new Page.NavigateOptions().setWaitUntil(WaitUntilState.LOAD));
            
            // 执行具体操作
            return pageFunction.apply(page);
        } catch (Exception e) {
            log.error("页面操作失败: ", e);
            return "操作失败：" + e.getMessage();
        }
    }
    
    /**
     * 在页面上执行需要CSS选择器的操作的通用方法
     * @param url 网页地址
     * @param selector CSS选择器
     * @param pageFunction 页面操作函数
     * @return 操作结果
     */
    private String executeWithPageAndSelector(String url, String selector, Function<Page, String> pageFunction) {
        // 参数校验
        if (url == null || url.isEmpty() || selector == null || selector.isEmpty()) {
            return handleFailure("URL和选择器不能为空");
        }
        
        try (BrowserContext context = browser.newContext();
             Page page = context.newPage()) {
            // 导航到指定URL，等待页面加载完成
            page.navigate(url, new Page.NavigateOptions().setWaitUntil(WaitUntilState.LOAD));
            
            // 检查元素是否存在
            Locator locator = page.locator(selector);
            if (locator.count() == 0) {
                return handleFailure("未找到匹配的元素：" + selector);
            }
            
            // 执行具体操作
            return pageFunction.apply(page);
        } catch (Exception e) {
            log.error("页面操作失败: ", e);
            return "操作失败：" + e.getMessage();
        }
    }

    /**
     * 工具方法：获取指定URL的网页全部文本内容（去除HTML标签）
     * @param url 网页地址（必填）
     * @return 网页纯文本内容
     */
    @Tool(description = "获取网页纯文本内容")
    public String getWebPageText(String url) {
        log.debug("获取网页纯文本内容: {}", url);
        
        // 空值校验，增强健壮性
        if (url == null || url.isEmpty()) {
            return handleFailure("URL不能为空，请提供有效的网页地址");
        }

        return executeWithPage(url, page -> {
            // 导航到指定URL，等待页面加载完成
            page.navigate(url, new Page.NavigateOptions().setWaitUntil(WaitUntilState.LOAD));
            // 获取页面全部文本（Playwright的innerText会自动去除HTML标签，保留文本结构）
            return page.locator("body").innerText();
        });
    }
    
    /**
     * 工具方法：获取网页中指定CSS选择器的元素内容
     * @param url 网页地址
     * @param cssSelector CSS选择器（如#title、.content）
     * @return 元素的文本内容
     */
    @Tool(description = "获取网页指定元素的内容")
    public String getWebElementText(String url, String cssSelector) {
        log.debug("获取网页指定元素内容: {}, 选择器: {}", url, cssSelector);
        
        return executeWithPageAndSelector(url, cssSelector, page -> {
            Locator locator = page.locator(cssSelector);
            return locator.innerText();
        });
    }
    
    /**
     * 工具方法：在网页指定输入框中输入文本
     * @param url 网页地址
     * @param inputSelector 输入框的CSS选择器（如#search-input）
     * @param text 要输入的文本
     * @return 操作结果
     */
    @Tool(description = "在网页输入框中输入文本")
    public String inputTextToWebElement(String url, String inputSelector, String text) {
        log.debug("在网页输入框中输入文本: {}, 选择器: {}, 文本: {}", url, inputSelector, text);
        
        // 参数校验
        if (url == null || url.isEmpty() || inputSelector == null || inputSelector.isEmpty() || text == null) {
            return handleFailure("URL、输入框选择器和输入文本不能为空");
        }
        
        try (BrowserContext context = browser.newContext();
             Page page = context.newPage()) {
            page.navigate(url, new Page.NavigateOptions().setWaitUntil(WaitUntilState.LOAD));
            Locator inputLocator = page.locator(inputSelector);
            if (inputLocator.count() == 0) {
                return handleFailure("未找到输入框：" + inputSelector);
            }
            // 聚焦并输入文本
            inputLocator.click();
            inputLocator.fill(text);
            log.debug("文本输入成功: {}", text);
            return "文本输入成功：" + text;
        } catch (Exception e) {
            log.error("输入文本失败: ", e);
            return "输入文本失败：" + e.getMessage();
        }
    }
    
    /**
     * 工具方法：点击网页中的指定元素（按钮、链接等）
     * @param url 网页地址
     * @param elementSelector 元素的CSS选择器
     * @return 操作结果
     */
    @Tool(description = "点击网页指定元素")
    public String clickWebElement(String url, String elementSelector) {
        log.debug("点击网页指定元素: {}, 选择器: {}", url, elementSelector);
        
        return executeWithPageAndSelector(url, elementSelector, page -> {
            Locator locator = page.locator(elementSelector);
            // 点击元素，等待导航完成（如果是链接/提交按钮）
            locator.click();
            page.waitForLoadState(LoadState.LOAD);
            return "元素点击成功，当前页面URL：" + page.url();
        });
    }
    
    /**
     * 工具方法：获取网页全屏截图并保存到指定路径
     * @param url 网页地址
     * @param savePath 截图保存路径（如D:/screenshots/page.png）
     * @return 操作结果
     */
    @Tool(description = "获取网页全屏截图")
    public String captureWebPageScreenshot(String url, String savePath) {
        log.debug("获取网页全屏截图: {}, 保存路径: {}", url, savePath);
        
        // 参数校验
        if (url == null || url.isEmpty() || savePath == null || savePath.isEmpty()) {
            return handleFailure("URL和保存路径不能为空");
        }
        
        try (BrowserContext context = browser.newContext();
             Page page = context.newPage()) {
            page.navigate(url, new Page.NavigateOptions().setWaitUntil(WaitUntilState.LOAD));
            // 截取全屏并保存
            page.screenshot(new Page.ScreenshotOptions().setPath(java.nio.file.Paths.get(savePath)).setFullPage(true));
            
            // 将截图文件转换为Base64编码，用于前端预览显示
            String base64Image = "";
            try {
                Path imagePath = java.nio.file.Paths.get(savePath);
                byte[] imageBytes = Files.readAllBytes(imagePath);
                base64Image = Base64.getEncoder().encodeToString(imageBytes);
            } catch (Exception e) {
                log.warn("截图Base64编码转换失败: {}", e.getMessage());
            }
            
            log.debug("截图成功，保存路径: {}", savePath);
            
            // 如果有Base64图像数据，则将其添加到结果中以供前端预览
            String result = "截图成功，保存路径：" + savePath;
            if (!base64Image.isEmpty()) {
                result += "\n预览图像Base64: data:image/png;base64," + base64Image;
            }
            return result;
        } catch (Exception e) {
            log.error("截图失败: ", e);
            return "截图失败：" + e.getMessage();
        }
    }
    
    /**
     * 工具方法：提取网页中的所有超链接（a标签的href属性）
     * @param url 网页地址
     * @return 链接列表，以逗号分隔
     */
    @Tool(description = "提取网页中的所有链接")
    public String extractAllLinksFromPage(String url) {
        log.debug("提取网页中的所有链接: {}", url);
        
        // 参数校验
        if (url == null || url.isEmpty()) {
            return handleFailure("URL不能为空");
        }
        
        return executeWithPage(url, page -> {
            // 获取所有a标签的href属性
            Object result = page.locator("a").evaluateAll("elements => elements.map(el => el.href)");
            // 安全地进行类型转换
            List<?> rawList;
            if (result instanceof List) {
                rawList = (List<?>) result;
            } else {
                log.warn("预期返回List类型，但实际返回: {}", result != null ? result.getClass().getName() : "null");
                return "获取链接失败：返回类型错误";
            }
            
            // 安全地转换为List<String>
            List<String> links = rawList.stream()
                .map(item -> item != null ? item.toString() : "")
                .filter(str -> !str.isEmpty())
                .toList();
            
            return links.isEmpty() ? "未找到任何链接" : String.join(", ", links);
        });
    }
    
    /**
     * 工具方法：在网页中执行自定义JavaScript代码，获取返回结果
     * @param url 网页地址
     * @param jsCode 要执行的JS代码（如"document.title"、"window.innerWidth"）
     * @return JS执行结果
     */
    @Tool(description = "执行网页JavaScript代码")
    public String executeJavaScriptOnPage(String url, String jsCode) {
        log.debug("执行网页JavaScript代码: {}, JS代码: {}", url, jsCode);
        
        // 参数校验
        if (url == null || url.isEmpty() || jsCode == null || jsCode.isEmpty()) {
            return handleFailure("URL和JS代码不能为空");
        }
        
        try (BrowserContext context = browser.newContext();
             Page page = context.newPage()) {
            page.navigate(url, new Page.NavigateOptions().setWaitUntil(WaitUntilState.LOAD));
            // 执行JS代码并获取结果
            Object result = page.evaluate(jsCode);
            String resultStr = "JS执行结果：" + (result == null ? "null" : result.toString());
            log.debug("JS执行成功，结果: {}", resultStr);
            return resultStr;
        } catch (Exception e) {
            log.error("执行JS失败: ", e);
            return "执行JS失败：" + e.getMessage();
        }
    }
    

}