package pangea.hiagent.tool.impl;

import com.microsoft.playwright.*;
import com.microsoft.playwright.options.LoadState;
import com.microsoft.playwright.options.WaitUntilState;
import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.File;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;

import pangea.hiagent.web.service.ToolConfigService;
import pangea.hiagent.workpanel.playwright.PlaywrightManager;

/**
 * 海信绩效系统流程审批工具类
 * 专门负责海信绩效管理系统(HIPMS)的审批功能，需要先通过HisenseSsoLoginTool登录
 * 该工具专注于流程审批操作，不处理登录逻辑
 */
@Slf4j
@Component
public class HisensePerformanceApprovalTool {

    // SSO登录页面URL
    private static final String SSO_LOGIN_URL = "https://sso.hisense.com/login/";
    
    // 绩效管理系统待审批页面URL
    private static final String PERFORMANCE_PENDING_URL = "https://hipms.hisense.com/PBC/PBCsubordinateTaskBook";

    // 注入Playwright管理器
    @Autowired
    private PlaywrightManager playwrightManager;
    
    @Autowired
    private ToolConfigService toolConfigService;
    
    // 存储目录路径
    private static final String STORAGE_DIR = "storage";

    /**
     * 从数据库获取SSO用户名
     * 
     * @return SSO用户名
     */
    public String getSsoUsername() {
        log.debug("从数据库获取SSO用户名");
        return toolConfigService.getParamValue("hisenseSsoLogin", "ssoUsername");
    }

    /**
     * 工具方法：自动查找所有待审批的绩效流程的网址
     * 
     * @return 待审批流程的URL列表
     */
    @Tool(description = "自动查找所有待审批的绩效流程的网址，需要先使用HisenseSsoLoginTool登录")
    public List<String> checkHisensePerformancePendingTasks() {
        String ssoUsername = getSsoUsername();
        log.info("开始为用户 {} 查找所有待审批的绩效流程", ssoUsername);
        
        long startTime = System.currentTimeMillis();
        
        // 参数校验
        if (ssoUsername == null || ssoUsername.isEmpty()) {
            String errorMsg = "用户ID不能为空";
            log.error(errorMsg);
            return List.of(errorMsg);
        }
        
        Page page = null;
        
        try {
            // 获取用户专用的浏览器上下文
            BrowserContext userContext = playwrightManager.getUserContext(ssoUsername);
            
            // 创建新页面
            page = userContext.newPage();
            
            // 访问待审批列表页面
            log.info("正在访问绩效系统待审批页面: {}", PERFORMANCE_PENDING_URL);
            page.navigate(PERFORMANCE_PENDING_URL, new Page.NavigateOptions().setWaitUntil(WaitUntilState.NETWORKIDLE));
            
            // 检查是否重定向到了SSO登录页面
            String currentUrl = page.url();
            log.info("当前页面URL: {}", currentUrl);
            
            if (currentUrl.startsWith(SSO_LOGIN_URL)) {
                String errorMsg = "用户未登录或会话已过期，请先使用HisenseSsoLoginTool进行登录";
                log.error(errorMsg);
                return List.of(errorMsg);
            }
            
            // 等待页面完全加载完成
            page.waitForLoadState(LoadState.NETWORKIDLE);
            
            // 等待按钮加载完成
            try {
                page.waitForSelector("button[data-v-991781fe] span", 
                    new Page.WaitForSelectorOptions().setState(com.microsoft.playwright.options.WaitForSelectorState.VISIBLE).setTimeout(10000));
            } catch (com.microsoft.playwright.TimeoutError e) {
                log.info("在预期时间内未找到待审批项目，可能没有待审批的流程");
                return List.of("没有找到待审批的流程");
            }
            
            ArrayList<String> urls = new ArrayList<>();
            
            // 查找所有"审核"按钮（第一种）
            Locator auditButtons = page.locator("button[data-v-991781fe][type='button']:has(span:text('审核'))");
            int auditButtonCount = auditButtons.count();
            log.info("找到 {} 个'审核'按钮", auditButtonCount);
            
            // 查找所有"复评"按钮（第二种）
            Locator reviewButtons = page.locator("button[data-v-991781fe][type='button']:has(span:text('复评'))");
            int reviewButtonCount = reviewButtons.count();
            log.info("找到 {} 个'复评'按钮", reviewButtonCount);
            
            // 处理"审核"按钮，获取跳转地址
            for (int i = 0; i < auditButtonCount; i++) {
                try {
                    Locator auditButton = auditButtons.nth(i);
                    
                    // 尝试获取按钮的href属性或从父元素获取
                    String href = auditButton.getAttribute("href");
                    if (href != null && !href.isEmpty()) {
                        log.info("获取到审核流程URL: {}", href);
                        urls.add(href);
                    } else {
                        // 如果按钮没有href属性，点击按钮后获取新页面URL
                        log.debug("审核按钮无href属性，将通过点击获取URL");
                        
                        // 监听新的页面打开事件
                        Page newPage = page.context().waitForPage(() -> {
                            auditButton.click();
                        });
                        
                        String newPageUrl = newPage.url();
                        log.info("获取到审核流程URL (通过点击): {}", newPageUrl);
                        urls.add(newPageUrl);
                        
                        // 关闭新页面，返回原页面
                        newPage.close();
                        page.bringToFront();
                    }
                } catch (Exception e) {
                    log.warn("处理审核按钮时发生异常: {}", e.getMessage());
                }
            }
            
            // 处理"复评"按钮，获取跳转地址
            for (int i = 0; i < reviewButtonCount; i++) {
                try {
                    Locator reviewButton = reviewButtons.nth(i);
                    
                    // 尝试获取按钮的href属性或从父元素获取
                    String href = reviewButton.getAttribute("href");
                    if (href != null && !href.isEmpty()) {
                        log.info("获取到复评流程URL: {}", href);
                        urls.add(href);
                    } else {
                        // 如果按钮没有href属性，点击按钮后获取新页面URL
                        log.debug("复评按钮无href属性，将通过点击获取URL");
                        
                        // 监听新的页面打开事件
                        Page newPage = page.context().waitForPage(() -> {
                            reviewButton.click();
                        });
                        
                        String newPageUrl = newPage.url();
                        log.info("获取到复评流程URL (通过点击): {}", newPageUrl);
                        urls.add(newPageUrl);
                        
                        // 关闭新页面，返回原页面
                        newPage.close();
                        page.bringToFront();
                    }
                } catch (Exception e) {
                    log.warn("处理复评按钮时发生异常: {}", e.getMessage());
                }
            }
            
            long endTime = System.currentTimeMillis();
            log.info("待审批流程查找完成，共 {} 个流程，耗时: {} ms", urls.size(), endTime - startTime);
            
            if (urls.isEmpty()) {
                return List.of("没有找到待审批的流程");
            }
            
            return urls;
        } catch (Exception e) {
            long endTime = System.currentTimeMillis();
            String errorMsg = "查找待审批流程失败: " + e.getMessage();
            log.error("查找待审批流程失败，耗时: {} ms", endTime - startTime, e);
            
            // 如果页面对象存在，截图保存错误页面
            if (page != null) {
                try {
                    takeScreenshotAndSave(page, "performance_pending_fail_" + ssoUsername);
                } catch (Exception screenshotException) {
                    log.warn("截图保存失败: {}", screenshotException.getMessage());
                }
            }
            
            return List.of(errorMsg);
        } finally {
            // 不立即关闭页面，让服务器完成处理
            // 保留页面、上下文和浏览器实例供后续操作使用
            if (page != null) {
                try {
                    log.debug("保留页面实例供后续使用");
                } catch (Exception e) {
                    log.warn("处理页面实例时发生异常: {}", e.getMessage());
                }
            }
        }
    }

    /**
     * 工具方法：获取海信绩效系统的审批页面内容
     * 
     * @param approvalUrl 审批页面URL
     * @return 页面内容（HTML文本）
     */
    @Tool(description = "获取海信绩效系统的审批页面内容，需要先使用HisenseSsoLoginTool登录")
    public String getHisensePerformancePageContent(String approvalUrl) {
        String ssoUsername = getSsoUsername();
        log.info("开始为用户 {} 获取绩效审批页面内容，URL: {}", ssoUsername, approvalUrl);
        
        long startTime = System.currentTimeMillis();
        
        // 参数校验
        if (ssoUsername == null || ssoUsername.isEmpty()) {
            String errorMsg = "用户ID不能为空";
            log.error(errorMsg);
            return errorMsg;
        }
        
        if (approvalUrl == null || approvalUrl.isEmpty()) {
            String errorMsg = "审批页面URL不能为空";
            log.error(errorMsg);
            return errorMsg;
        }

        Page page = null;
        
        try {
            // 获取用户专用的浏览器上下文
            BrowserContext userContext = playwrightManager.getUserContext(ssoUsername);
            
            // 创建新页面
            page = userContext.newPage();
            
            // 访问审批页面
            log.info("正在访问审批页面: {}", approvalUrl);
            page.navigate(approvalUrl, new Page.NavigateOptions().setWaitUntil(WaitUntilState.NETWORKIDLE));
            
            // 检查是否重定向到了SSO登录页面
            String currentUrl = page.url();
            log.info("当前页面URL: {}", currentUrl);
            
            if (currentUrl.startsWith(SSO_LOGIN_URL)) {
                String errorMsg = "用户未登录或会话已过期，请先使用HisenseSsoLoginTool进行登录";
                log.error(errorMsg);
                return errorMsg;
            }
            
            // 等待页面完全加载完成
            page.waitForLoadState(LoadState.NETWORKIDLE);
            
            // 提取页面内容
            String content = page.locator("body").innerText();
            long endTime = System.currentTimeMillis();
            log.info("成功获取绩效审批页面内容，耗时: {} ms", endTime - startTime);
            
            return content;
        } catch (Exception e) {
            long endTime = System.currentTimeMillis();
            String errorMsg = "获取绩效审批页面内容失败: " + e.getMessage();
            log.error("获取绩效审批页面内容失败，耗时: {} ms", endTime - startTime, e);
            return errorMsg;
        } finally {
            // 释放页面资源，但保留浏览器上下文供后续使用
            if (page != null) {
                try {
                    page.close();
                } catch (Exception e) {
                    log.warn("关闭页面时发生异常: {}", e.getMessage());
                }
            }
        }
    }

    /**
     * 工具方法：处理绩效系统单个审批流程
     * 
     * @param approvalUrl 审批页面URL
     * @param isApproved 是否通过审批（true为通过，false为驳回）
     * @param approvalOpinion 审批意见
     * @return 处理结果
     */
    @Tool(description = "处理绩效系统单个审批流程，需要先使用HisenseSsoLoginTool登录")
    public String performSinglePerformanceApproval(String approvalUrl, boolean isApproved, String approvalOpinion) {
        String ssoUsername = getSsoUsername();
        log.info("开始为用户 {} 处理绩效审批，URL: {}, 是否通过: {}", ssoUsername, approvalUrl, isApproved);
        
        long startTime = System.currentTimeMillis();
        
        // 参数校验
        if (ssoUsername == null || ssoUsername.isEmpty()) {
            String errorMsg = "用户ID不能为空";
            log.error(errorMsg);
            return errorMsg;
        }
        
        if (approvalUrl == null || approvalUrl.isEmpty()) {
            String errorMsg = "审批页面URL不能为空";
            log.error(errorMsg);
            return errorMsg;
        }
        
        if (approvalOpinion == null || approvalOpinion.isEmpty()) {
            String errorMsg = "审批意见不能为空";
            log.error(errorMsg);
            return errorMsg;
        }
        
        Page page = null;
        
        try {
            // 获取用户专用的浏览器上下文
            BrowserContext userContext = playwrightManager.getUserContext(ssoUsername);
            
            // 创建新页面
            page = userContext.newPage();
            
            // 访问审批页面
            log.info("正在访问审批页面: {}", approvalUrl);
            page.navigate(approvalUrl, new Page.NavigateOptions().setWaitUntil(WaitUntilState.NETWORKIDLE));
            
            // 检查是否重定向到了SSO登录页面
            String currentUrl = page.url();
            log.info("当前页面URL: {}", currentUrl);
            
            if (currentUrl.startsWith(SSO_LOGIN_URL)) {
                String errorMsg = "用户未登录或会话已过期，请先使用HisenseSsoLoginTool进行登录";
                log.error(errorMsg);
                return errorMsg;
            }
            
            // 等待页面完全加载完成
            page.waitForLoadState(LoadState.NETWORKIDLE);
            
            // 等待关键元素加载完成，确保页面完全就绪
            try {
                page.waitForSelector("input[name='radioGroup'][type='radio'][class='ant-radio-input']", 
                    new Page.WaitForSelectorOptions().setState(com.microsoft.playwright.options.WaitForSelectorState.VISIBLE).setTimeout(5000));
            } catch (com.microsoft.playwright.TimeoutError e) {
                log.warn("关键审批元素未在预期时间内加载完成，继续执行审批操作");
            }
            
            // 执行审批操作
            performPerformanceApprovalOperation(page, isApproved, approvalOpinion);
            
            // 截图并保存
            takeScreenshotAndSave(page, "performance_approval_success_" + ssoUsername);
            
            long endTime = System.currentTimeMillis();
            log.info("绩效审批处理完成，耗时: {} ms", endTime - startTime);
            
            return "绩效审批处理成功";
        } catch (Exception e) {
            long endTime = System.currentTimeMillis();
            String errorMsg = "绩效审批处理失败: " + e.getMessage();
            log.error("绩效审批处理失败，耗时: {} ms", endTime - startTime, e);
            
            // 如果页面对象存在，截图保存错误页面
            if (page != null) {
                try {
                    takeScreenshotAndSave(page, "performance_approval_fail_" + ssoUsername);
                } catch (Exception screenshotException) {
                    log.warn("截图保存失败: {}", screenshotException.getMessage());
                }
            }
            
            return errorMsg;
        } finally {
            // 不立即关闭页面，让服务器完成审批处理
            // 保留页面、上下文和浏览器实例供后续操作使用
            if (page != null) {
                try {
                    log.debug("保留页面实例以等待服务器完成审批处理");
                } catch (Exception e) {
                    log.warn("处理页面实例时发生异常: {}", e.getMessage());
                }
            }
        }
    }

    /**
     * 执行绩效审批操作
     * 
     * @param page 当前页面对象
     * @param isApproved 是否通过审批（true为通过，false为驳回）
     * @param approvalOpinion 审批意见
     * @throws Exception 审批过程中的异常
     */
    private void performPerformanceApprovalOperation(Page page, boolean isApproved, String approvalOpinion) throws Exception {
        log.info("开始执行绩效审批操作，审批结果: {}", isApproved ? "通过" : "驳回");
        
        try {

            // 获取当前页面URL作为参考
                String originalUrl = page.url();
                log.debug("审批前页面URL: {}", originalUrl);
                
            // 根据页面元素动态确定radioValue
            String radioValue;
            if (isApproved) {
                // 如果isApproved为真，检查页面上是否存在特定的span元素
                Locator pendingReviewSpan = page.locator("span[data-v-3734a0eb][class='ant-tag ant-tag-orange']:has-text('待复评')");
                Locator pendingAuditSpan = page.locator("span[data-v-3734a0eb][class='ant-tag ant-tag-orange']:has-text('待审核')");
                
                if (pendingReviewSpan.count() > 0) {
                    radioValue = "0"; // 待复评时radioValue = 0
                    log.debug("检测到'待复评'标签，设置radioValue为0");
                } else if (pendingAuditSpan.count() > 0) {
                    radioValue = "1"; // 待审核时radioValue = 1
                    log.debug("检测到'待审核'标签，设置radioValue为1");
                } else {
                    // 如果没有找到特定标签，默认使用1
                    radioValue = "1";
                    log.debug("未检测到特定标签，使用默认radioValue为1");
                }
            } else {
                // 如果isApproved为假（驳回），使用4
                radioValue = "4";
            }
            
            String radioSelector = "input[name='radioGroup'][type='radio'][class='ant-radio-input'][value='" + radioValue + "']";
            log.debug("正在定位审批结果单选框: {}", radioSelector);
            
            Locator radioButton = page.locator(radioSelector);
            if (radioButton.count() > 0) {
                radioButton.click();
                log.info("审批结果单选框选择完成: {}", isApproved ? "通过" : "驳回");
            } else {
                throw new RuntimeException("未找到审批结果单选框");
            }

            // 定位审批意见输入框并填入内容
            String opinionTextareaSelector = "textarea[class='ant-input'][style*='width']";
            log.debug("正在定位审批意见输入框: {}", opinionTextareaSelector);
            
            Locator opinionTextarea = page.locator(opinionTextareaSelector);
            if (opinionTextarea.count() > 0) {
                opinionTextarea.fill(approvalOpinion);
                log.debug("审批意见输入完成");
            } else {
                log.warn("未找到审批意见输入框，尝试使用备选选择器");
                opinionTextareaSelector = "textarea[placeholder=''][class='ant-input']";
                opinionTextarea = page.locator(opinionTextareaSelector);
                if (opinionTextarea.count() > 0) {
                    opinionTextarea.fill(approvalOpinion);
                    log.debug("审批意见输入完成 (使用备选选择器)");
                } else {
                    throw new RuntimeException("未找到审批意见输入框");
                }
            }

            // 定位并点击提交按钮
            String submitButtonSelector = "button[data-v-deb7b464][type='button'].ant-btn.ant-btn-primary";
            log.debug("正在定位提交按钮: {}", submitButtonSelector);
                        
            Locator submitButton = page.locator(submitButtonSelector);
            if (submitButton.count() > 0) {
                submitButton.click();
                log.info("提交按钮点击完成");
            } else {
                throw new RuntimeException("未找到提交按钮");
            }
            
            // 等待页面导航完成（只要跳转离开当前审批页就算完成）
            try {
                log.debug("等待页面导航完成...");
                
                
                // 等待URL变化（表示已离开当前审批页面）
                page.waitForURL(url -> !url.equals(originalUrl), 
                    new Page.WaitForURLOptions().setTimeout(10000));
                
                log.info("审批操作执行完成，页面已跳转离开当前审批页");
            } catch (com.microsoft.playwright.TimeoutError e) {
                log.warn("等待页面跳转超时，但审批操作可能已成功提交: {}", e.getMessage());
            } catch (Exception e) {
                log.warn("等待页面导航时发生异常，但审批操作可能已成功提交: {}", e.getMessage());
            }
            
            // 记录跳转后的页面URL
            String currentUrl = page.url();
            log.info("审批后当前页面URL: {}", currentUrl);
            
        } catch (Exception e) {
            log.error("审批操作过程中发生异常", e);
            throw new RuntimeException("审批操作失败: " + e.getMessage(), e);
        }
    }

    /**
     * 截图并保存到存储目录
     * 
     * @param page 当前页面对象
     * @param fileName 文件名前缀
     */
    private void takeScreenshotAndSave(Page page, String fileName) {
        try {
            // 确保存储目录存在
            File storageDir = new File(STORAGE_DIR);
            if (!storageDir.exists()) {
                storageDir.mkdirs();
            }
            
            // 生成带时间戳的文件名
            String timestamp = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMdd_HHmmss"));
            String fullFileName = String.format("%s_%s.png", fileName, timestamp);
            String filePath = Paths.get(STORAGE_DIR, fullFileName).toString();
            
            // 截图并保存
            page.screenshot(new Page.ScreenshotOptions().setPath(Paths.get(filePath)));
            log.info("截图已保存至: {}", filePath);
        } catch (Exception e) {
            log.error("截图保存失败: {}", e.getMessage(), e);
        }
    }
}
