package pangea.hiagent.auth;

import lombok.extern.slf4j.Slf4j;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Component;
import pangea.hiagent.model.AuthMode;
import pangea.hiagent.model.User;
import pangea.hiagent.common.utils.JwtUtil;
import pangea.hiagent.web.repository.UserRepository;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;

import java.util.Arrays;
import java.util.Map;

/**
 * 本地用户名/密码认证策略实现
 * 支持基于本地数据库的用户名密码认证
 */
@Slf4j
@Component
public class LocalAuthenticationStrategy implements AuthenticationStrategy {
    
    private final UserRepository userRepository;
    private final PasswordEncoder passwordEncoder;
    private final JwtUtil jwtUtil;
    private final org.springframework.core.env.Environment environment;
    
    public LocalAuthenticationStrategy(UserRepository userRepository, PasswordEncoder passwordEncoder, 
                                      JwtUtil jwtUtil, org.springframework.core.env.Environment environment) {
        this.userRepository = userRepository;
        this.passwordEncoder = passwordEncoder;
        this.jwtUtil = jwtUtil;
        this.environment = environment;
    }
    
    @Override
    public String getName() {
        return "Local Authentication Strategy";
    }
    
    @Override
    public boolean supports(String authMode) {
        return AuthMode.LOCAL.getCode().equals(authMode);
    }
    
    /**
     * 执行本地用户认证
     * @param credentials 需要包含 username 和 password 字段
     * @return JWT Token
     */
    @Override
    public String authenticate(Map<String, Object> credentials) {
        String username = (String) credentials.get("username");
        String password = (String) credentials.get("password");
        
        if (username == null || username.trim().isEmpty()) {
            log.warn("本地认证失败: 用户名为空");
            throw new RuntimeException("用户名不能为空");
        }
        
        if (password == null || password.trim().isEmpty()) {
            log.warn("本地认证失败: 密码为空");
            throw new RuntimeException("密码不能为空");
        }
        
        log.info("执行本地认证: {}", username);
        
        // 查询用户
        LambdaQueryWrapper<User> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(User::getUsername, username);
        User user = userRepository.selectOne(wrapper);
        
        if (user == null) {
            log.warn("本地认证失败: 用户 {} 不存在", username);
            throw new RuntimeException("用户不存在");
        }
        
        // 检查用户状态
        if (!"active".equals(user.getStatus())) {
            log.warn("本地认证失败: 用户 {} 已被禁用", username);
            throw new RuntimeException("用户已禁用");
        }
        
        // 检查是否为开发环境，如果是则允许任意密码
        boolean isDevEnvironment = Arrays.asList(environment.getActiveProfiles()).contains("dev") || 
                                  Arrays.asList(environment.getDefaultProfiles()).contains("default");
        
        if (!isDevEnvironment) {
            // 验证密码
            boolean passwordMatch = passwordEncoder.matches(password, user.getPassword());
            
            if (!passwordMatch) {
                log.warn("本地认证失败: 用户 {} 密码错误", username);
                throw new RuntimeException("密码错误");
            }
        } else {
            log.info("开发环境: 跳过密码验证");
        }
        
        // 更新最后登录时间
        user.setLastLoginTime(System.currentTimeMillis());
        userRepository.updateById(user);
        
        // 生成 Token
        String token = jwtUtil.generateToken(user.getId());
        log.info("本地认证成功，用户: {}, 生成Token: {}", username, token);
        
        return token;
    }
    
    @Override
    public boolean verify(String token) {
        return jwtUtil.validateToken(token);
    }
}
