package pangea.hiagent.workpanel.event;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import pangea.hiagent.web.dto.*;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 事件去重服务
 * 统一处理事件去重逻辑，避免在多个地方重复实现
 */
@Slf4j
@Component
public class EventDeduplicationService {
    
    /**
     * 最近事件的缓存，用于去重检查
     */
    private final Map<String, WorkPanelEvent> recentEventsCache = new ConcurrentHashMap<>();
    
    /**
     * 缓存过期时间（毫秒），默认5秒
     */
    private static final long CACHE_EXPIRY_TIME = 5000L;
    
    /**
     * 最大缓存大小，防止内存溢出
     */
    private static final int MAX_CACHE_SIZE = 1000;
    
    /**
     * 上次清理缓存的时间
     */
    private volatile long lastCleanupTime = 0;
    
    /**
     * 清理间隔（毫秒）
     */
    private static final long CLEANUP_INTERVAL = 30000L;
    
    /**
     * 检查是否为重复事件
     */
    public boolean isDuplicateEvent(WorkPanelEvent event) {
        if (event == null) {
            return false;
        }
        
        // 定期清理过期缓存
        cleanupExpiredCache();
        
        // 生成事件唯一标识
        String eventKey = generateEventKey(event);
        WorkPanelEvent cachedEvent = recentEventsCache.get(eventKey);
        
        if (cachedEvent != null) {
            // 检查缓存是否过期
            long currentTime = System.currentTimeMillis();
            if ((currentTime - cachedEvent.getTimestamp()) <= CACHE_EXPIRY_TIME) {
                // 检查事件内容是否相同
                return isEventContentEqual(event, cachedEvent);
            } else {
                // 缓存过期，移除
                recentEventsCache.remove(eventKey);
            }
        }
        
        return false;
    }
    
    /**
     * 更新最近事件缓存
     */
    public void updateRecentEventsCache(WorkPanelEvent event) {
        if (event == null) {
            return;
        }
        
        // 控制缓存大小，防止内存溢出
        if (recentEventsCache.size() >= MAX_CACHE_SIZE) {
            // 移除最老的条目（简单实现，实际可以根据LRU算法优化）
            java.util.Iterator<Map.Entry<String, WorkPanelEvent>> iterator = recentEventsCache.entrySet().iterator();
            if (iterator.hasNext()) {
                iterator.next();
                iterator.remove();
            }
        }
        
        // 添加新条目
        String eventKey = generateEventKey(event);
        recentEventsCache.put(eventKey, event);
    }
    
    /**
     * 生成事件唯一标识
     */
    private String generateEventKey(WorkPanelEvent event) {
        StringBuilder key = new StringBuilder();
        key.append(event.getType() != null ? event.getType() : "");
        key.append("_");
        
        // 获取工具名称（如果是ToolEvent）
        String toolName = "";
        if (event instanceof ToolEvent) {
            toolName = ((ToolEvent) event).getToolName();
        }
        key.append(toolName != null ? toolName : "");
        key.append("_");
        
        // 获取思考类型（如果是ThoughtEvent）
        String thinkingType = "";
        if (event instanceof ThoughtEvent) {
            thinkingType = ((ThoughtEvent) event).getThinkingType();
        }
        key.append(thinkingType != null ? thinkingType : "");
        
        // 对于思考事件，添加内容摘要
        if ("thought".equals(event.getType()) && event instanceof ThoughtEvent) {
            ThoughtEvent thoughtEvent = (ThoughtEvent) event;
            if (thoughtEvent.getContent() != null) {
                // 取内容的前50个字符作为摘要
                String contentSummary = thoughtEvent.getContent().length() > 50 ? 
                    thoughtEvent.getContent().substring(0, 50) : thoughtEvent.getContent();
                key.append("_").append(contentSummary.hashCode());
            }
        }
        
        return key.toString();
    }
    
    /**
     * 检查两个事件的内容是否相等
     */
    private boolean isEventContentEqual(WorkPanelEvent event1, WorkPanelEvent event2) {
        if (event1 == event2) {
            return true;
        }
        
        if (event1 == null || event2 == null) {
            return false;
        }
        
        // 比较基本字段
        if (!java.util.Objects.equals(event1.getType(), event2.getType())) {
            return false;
        }
        
        // 比较工具名称（如果是ToolEvent）
        String toolName1 = "";
        String toolName2 = "";
        if (event1 instanceof ToolEvent) {
            toolName1 = ((ToolEvent) event1).getToolName();
        }
        if (event2 instanceof ToolEvent) {
            toolName2 = ((ToolEvent) event2).getToolName();
        }
        if (!java.util.Objects.equals(toolName1, toolName2)) {
            return false;
        }
        
        // 比较思考类型（如果是ThoughtEvent）
        String thinkingType1 = "";
        String thinkingType2 = "";
        if (event1 instanceof ThoughtEvent) {
            thinkingType1 = ((ThoughtEvent) event1).getThinkingType();
        }
        if (event2 instanceof ThoughtEvent) {
            thinkingType2 = ((ThoughtEvent) event2).getThinkingType();
        }
        if (!java.util.Objects.equals(thinkingType1, thinkingType2)) {
            return false;
        }
        
        // 比较内容字段（根据不同事件类型）
        if (event1 instanceof ThoughtEvent && event2 instanceof ThoughtEvent) {
            ThoughtEvent thought1 = (ThoughtEvent) event1;
            ThoughtEvent thought2 = (ThoughtEvent) event2;
            return java.util.Objects.equals(thought1.getContent(), thought2.getContent());
        }
        
        // 比较工具输入（确保都是ToolEvent）
        if (event1 instanceof ToolEvent && event2 instanceof ToolEvent) {
            if (!java.util.Objects.equals(((ToolEvent) event1).getToolInput(), 
                                         ((ToolEvent) event2).getToolInput())) {
                return false;
            }
            
            // 比较工具输出
            if (!java.util.Objects.equals(((ToolEvent) event1).getToolOutput(), 
                                         ((ToolEvent) event2).getToolOutput())) {
                return false;
            }
        } else if (event1 instanceof ResultEvent && event2 instanceof ResultEvent) {
            // 比较结果内容
            String content1 = ((ResultEvent) event1).getContent();
            String content2 = ((ResultEvent) event2).getContent();
            if (!java.util.Objects.equals(content1, content2)) {
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * 清理过期缓存
     */
    private void cleanupExpiredCache() {
        long currentTime = System.currentTimeMillis();
        
        // 检查是否需要执行清理
        if (currentTime - lastCleanupTime < CLEANUP_INTERVAL) {
            return;
        }
        
        // 更新上次清理时间
        lastCleanupTime = currentTime;
        
        // 清理过期条目
        recentEventsCache.entrySet().removeIf(entry -> {
            WorkPanelEvent event = entry.getValue();
            return event == null || (currentTime - event.getTimestamp()) > CACHE_EXPIRY_TIME;
        });
    }
    
    /**
     * 清空缓存
     */
    public void clearCache() {
        recentEventsCache.clear();
    }
}