import axios, { AxiosInstance, AxiosRequestConfig, AxiosResponse, InternalAxiosRequestConfig } from 'axios'
import { useAuthStore } from '@/stores/auth'
import { ElMessage } from 'element-plus'

// 创建一个 axios 实例
const request: AxiosInstance = axios.create({
  baseURL: '/api/v1', // 设置基础URL
  timeout: 120000, // 设置超时时间为2分钟，解决Dashboard页面请求超时问题
})

// 请求拦截器
request.interceptors.request.use(
  (config: InternalAxiosRequestConfig) => {
    // 从 localStorage 获取 token
    const token = localStorage.getItem('token')
    if (token) {
      config.headers.Authorization = `Bearer ${token}`
    }
    return config
  },
  (error: any) => {
    return Promise.reject(error)
  }
)

// 响应拦截器
request.interceptors.response.use(
  (response: AxiosResponse) => {
    // 对响应数据做点什么
    return response
  },
  (error: any) => {
    // 对响应错误做点什么
    if (error.response && error.response.status === 401) {
      // token 过期或无效，清除本地存储
      localStorage.removeItem('token')
      localStorage.removeItem('userInfo')
      
      // 使用 auth store 的 logout 方法
      const authStore = useAuthStore()
      authStore.logout()
      
      // 显示错误消息
      ElMessage.error('认证已过期，请重新登录')
      
      // 如果在浏览器环境中，跳转到登录页
      if (typeof window !== 'undefined') {
        // 延迟跳转，确保状态清理完成
        setTimeout(() => {
          window.location.href = '/login'
        }, 100)
      }
    } else if (error.response && error.response.status === 403) {
      // 权限不足
      ElMessage.error('权限不足，无法访问该资源')
    } else if (error.response && error.response.status >= 500) {
      // 服务器错误
      ElMessage.error('服务器内部错误，请稍后重试')
    } else if (error.response) {
      // 其他HTTP错误
      ElMessage.error(`请求失败: ${error.response.status} ${error.response.statusText}`)
    } else if (error.request) {
      // 网络错误
      ElMessage.error('网络连接失败，请检查网络设置')
    } else {
      // 其他错误
      ElMessage.error('请求失败: ' + error.message)
    }
    return Promise.reject(error)
  }
)

export default request