package pangea.hiagent.workpanel;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;
import pangea.hiagent.dto.WorkPanelEvent;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * SSE 事件管理器
 * 负责处理 SSE 连接、事件发送和连接管理
 */
@Slf4j
@Component
public class SseEventManager {

    private static final long SSE_TIMEOUT = 300000L; // 5分钟超时，与前端保持一致
    
    // 存储所有活动的 emitter
    private final List<SseEmitter> emitters = new CopyOnWriteArrayList<>();

    /**
     * 创建 SSE Emitter
     */
    public SseEmitter createEmitter() {
        SseEmitter emitter = new SseEmitter(SSE_TIMEOUT);
        registerCallbacks(emitter);
        emitters.add(emitter);
        return emitter;
    }
    
    /**
     * 获取所有活动的 emitter
     */
    public List<SseEmitter> getEmitters() {
        return emitters;
    }
    
    /**
     * 从列表中移除指定的 emitter
     */
    public void removeEmitter(SseEmitter emitter) {
        emitters.remove(emitter);
    }

    /**
     * 注册 SSE Emitter 回调
     */
    public void registerCallbacks(SseEmitter emitter) {
        emitter.onCompletion(() -> {
            log.debug("SSE连接已完成");
            removeEmitter(emitter);
        });
        emitter.onError((Throwable t) -> {
            // 区分客户端断开和真正的错误
            if (t instanceof IOException) {
                String message = t.getMessage();
                if (message != null && (message.contains("软件中止") || message.contains("中断") || message.contains("连接"))) {
                    // 客户端主动断开连接 - 这是正常行为
                    log.debug("SSE流式传输中客户端连接中断 - 异常类型: {}", t.getClass().getSimpleName());
                } else {
                    // 其他IO异常
                    log.debug("SSE连接发生IO异常", t);
                }
            } else if (t instanceof org.springframework.web.context.request.async.AsyncRequestNotUsableException) {
                // 异步请求不可用 - 客户端已断开
                log.debug("SSE异步请求不可用，客户端已断开连接");
            } else {
                // 其他真正的错误
                log.error("SSE连接发生错误", t);
            }
            removeEmitter(emitter);
        });
        emitter.onTimeout(() -> {
            log.debug("SSE连接超时");
            completeEmitter(emitter, new AtomicBoolean(true));
            removeEmitter(emitter);
        });
    }

    /**
     * 发送 SSE 事件
     * 增强版本：在发送前检查连接状态，防止在已断开的连接上继续尝试发送
     */
    public void sendEvent(SseEmitter emitter, String eventName, Object data, AtomicBoolean isCompleted) throws IOException {
        synchronized (emitter) {
            if (!isCompleted.get()) {
                try {
                    emitter.send(SseEmitter.event().name(eventName).data(data));
                    log.debug("SSE事件发送成功 [{}]", eventName);
                } catch (org.springframework.web.context.request.async.AsyncRequestNotUsableException e) {
                    // 客户端已断开连接或请求不再可用 - 这是正常的客户端中断行为
                    log.debug("异步请求不可用，客户端已断开连接[{}]: {}", eventName, e.getMessage());
                    isCompleted.set(true);
                } catch (java.io.IOException e) {
                    // IO异常 - 客户端连接中断
                    String errorDetail = e.getMessage();
                    boolean isNormalDisconnect = errorDetail != null && 
                        (errorDetail.contains("软件中止") || 
                         errorDetail.contains("中断") || 
                         errorDetail.contains("连接") ||
                         errorDetail.contains("Socket") ||
                         errorDetail.contains("Pipe") ||
                         errorDetail.contains("closed") ||
                         errorDetail.contains("Software caused connection abort"));
                    
                    if (isNormalDisconnect) {
                        // 正常的客户端断开连接 - 调试级别
                        log.debug("客户端连接中断，SSE事件[{}]发送失败: {}", eventName, errorDetail);
                    } else {
                        // 异常的IO错误 - 警告级别，需要关注
                        log.warn("异常IO错误，SSE事件[{}]发送失败: {}", eventName, errorDetail);
                    }
                    isCompleted.set(true);
                } catch (IllegalStateException e) {
                    // Emitter已完成或无效状态 - 调试级别
                    log.debug("Emitter状态异常，无法发送SSE事件[{}]: {}", eventName, e.getMessage());
                    isCompleted.set(true);
                } catch (java.lang.RuntimeException e) {
                    // 处理其他运行时异常
                    if (e.getCause() instanceof org.springframework.web.context.request.async.AsyncRequestNotUsableException) {
                        log.debug("RuntimeException根因是AsyncRequestNotUsableException，SSE事件[{}]发送失败: {}", eventName, e.getCause().getMessage());
                        isCompleted.set(true);
                    } else if (e.getCause() instanceof java.io.IOException) {
                        String causeMsg = e.getCause().getMessage();
                        boolean isNormalDisconnect = causeMsg != null && 
                            (causeMsg.contains("软件中止") || 
                             causeMsg.contains("中断") || 
                             causeMsg.contains("连接") ||
                             causeMsg.contains("Socket") ||
                             causeMsg.contains("Pipe") ||
                             causeMsg.contains("closed") ||
                             causeMsg.contains("Software caused connection abort"));
                        
                        if (isNormalDisconnect) {
                            log.debug("RuntimeException根因是IOException，SSE事件[{}]发送失败: {}", eventName, causeMsg);
                        } else {
                            log.warn("RuntimeException根因是IOException，SSE事件[{}]发送失败: {}", eventName, causeMsg);
                        }
                        isCompleted.set(true);
                    } else if (e.getMessage() != null && e.getMessage().contains("response has already been committed")) {
                        log.debug("响应已提交，无法发送SSE事件[{}]", eventName);
                        isCompleted.set(true);
                    } else {
                        // 其他运行时异常 - 记录关键信息便于诊断
                        log.warn("SSE事件[{}]发送失败 - RuntimeException - 异常消息: {}", eventName, e.getMessage());
                        isCompleted.set(true);
                    }
                } catch (Exception e) {
                    // 其他异常 - 根据类型决定日志级别
                    String msg = e.getMessage() != null ? e.getMessage() : "";
                    if (msg.contains("response has already been committed")) {
                        log.debug("响应已提交，无法发送SSE事件[{}]", eventName);
                    } else if (msg.contains("closed") || msg.contains("Closed") || msg.contains("Broken") || msg.contains("Software caused connection abort")) {
                        log.debug("连接已关闭，无法发送SSE事件[{}]", eventName);
                    } else {
                        // 未知异常 - 记录便于排查
                        log.warn("SSE事件[{}]发送失败 - 未知异常 - 异常类型: {} - 消息: {}", 
                                eventName, 
                                e.getClass().getSimpleName(), 
                                msg);
                    }
                    isCompleted.set(true);
                }
            } else {
                // 连接已完成，不再发送
                if (log.isTraceEnabled()) {
                    log.trace("连接已完成，跳过发送SSE事件: {}", eventName);
                }
            }
        }
    }

    /**
     * 发送工作面板事件
     */
    public void sendWorkPanelEvent(SseEmitter emitter, WorkPanelEvent event, AtomicBoolean isCompleted) throws IOException {
        // 先快速检查是否已完成，避免不必要的构造数据对象
        if (isCompleted.get()) {
            if (log.isTraceEnabled()) {
                log.trace("连接已完成，跳过发送工作面板事件");
            }
            return;
        }
        
        synchronized (emitter) {
            if (!isCompleted.get()) {
                try {
                    Map<String, Object> data = new HashMap<>();
                    data.put("eventType", event.getEventType());
                    data.put("timestamp", event.getTimestamp());
                    data.put("content", event.getContent());
                    data.put("thinkingType", event.getThinkingType());
                    data.put("toolName", event.getToolName());
                    data.put("toolAction", event.getToolAction());
                    data.put("toolInput", event.getToolInput());
                    data.put("toolOutput", event.getToolOutput());
                    data.put("toolStatus", event.getToolStatus());
                    data.put("logLevel", event.getLogLevel());
                    data.put("executionTime", event.getExecutionTime());
                    
                    // 添加embed事件相关字段
                    data.put("embedUrl", event.getEmbedUrl());
                    data.put("embedType", event.getEmbedType());
                    data.put("embedTitle", event.getEmbedTitle());
                    data.put("embedHtmlContent", event.getEmbedHtmlContent());
                    
                    // 确保所有事件类型都包含type字段，保证前端可以正确识别事件类型
                    data.put("type", event.getEventType());
                    
                    // 修复：对于最终答案类型的思考事件，添加特殊标记
                    if ("thinking".equals(event.getEventType()) && "final_answer".equals(event.getThinkingType())) {
                        data.put("isFinalAnswer", true);
                    }
                    
                    String eventName = getEventNameForWorkPanel(event.getEventType());
                    sendEvent(emitter, eventName, data, isCompleted);
                } catch (org.springframework.web.context.request.async.AsyncRequestNotUsableException e) {
                    // 客户端已断开连接 - 这是正常行为
                    log.debug("异步请求不可用，客户端已断开连接: {}", e.getMessage());
                    isCompleted.set(true);
                } catch (Exception e) {
                    if (e instanceof IOException || (e.getCause() instanceof IOException)) {
                        log.debug("客户端连接已断开，无法发送工作面板事件");
                        isCompleted.set(true);
                    } else if (e.getMessage() != null && e.getMessage().contains("response has already been committed")) {
                        log.debug("响应已提交，无法发送工作面板事件");
                        isCompleted.set(true);
                    } else {
                        // 其他异常类型，降级为debug日志
                        log.debug("发送工作面板事件失败: {}", e.getMessage());
                        isCompleted.set(true);
                    }
                }
            }
        }
    }

    /**
     * 根据事件类型获取SSE事件名称
     */
    private String getEventNameForWorkPanel(String eventType) {
        return switch (eventType) {
            case "thinking" -> "thinking";
            case "tool_call" -> "tool_call";
            case "tool_result" -> "tool_result";
            case "tool_error" -> "tool_error";
            case "log" -> "log";
            case "embed" -> "embed";
            default -> "workpanel_event";
        };
    }

    /**
     * 启动心跳保活
     */
    public void startHeartbeat(SseEmitter emitter, AtomicBoolean isCompleted) {
        ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(1);
        scheduler.scheduleAtFixedRate(() -> {
            if (!isCompleted.get()) {
                try {
                    // 发送心跳事件
                    emitter.send(SseEmitter.event().name("keepalive").data(""));
                    log.debug("心跳事件已发送");
                } catch (Exception e) {
                    // 心跳发送失败，标记完成并关闭调度器
                    String errorMsg = e.getMessage();
                    boolean isNormalDisconnect = errorMsg != null && 
                        (errorMsg.contains("软件中止") || 
                         errorMsg.contains("中断") || 
                         errorMsg.contains("连接") ||
                         errorMsg.contains("Socket") ||
                         errorMsg.contains("Pipe") ||
                         errorMsg.contains("closed") ||
                         errorMsg.contains("Software caused connection abort"));
                    
                    if (isNormalDisconnect) {
                        log.debug("心跳发送失败，客户端连接已断开: {}", errorMsg);
                    } else {
                        log.warn("心跳发送失败: {}", errorMsg);
                    }
                    isCompleted.set(true);
                    scheduler.shutdown();
                }
            } else {
                // 已完成，关闭调度器
                scheduler.shutdown();
            }
        }, 30, 30, TimeUnit.SECONDS); // 每30秒发送一次心跳
        
        // 添加一个更频繁的轻量级心跳，用于检测连接状态
        ScheduledExecutorService lightScheduler = Executors.newScheduledThreadPool(1);
        lightScheduler.scheduleAtFixedRate(() -> {
            if (!isCompleted.get()) {
                try {
                    // 发送轻量级心跳事件
                    emitter.send(SseEmitter.event().name("ping").data(""));
                    if (log.isTraceEnabled()) {
                        log.trace("轻量级心跳事件已发送");
                    }
                } catch (Exception e) {
                    // 心跳发送失败，标记完成并关闭调度器
                    String errorMsg = e.getMessage();
                    boolean isNormalDisconnect = errorMsg != null && 
                        (errorMsg.contains("软件中止") || 
                         errorMsg.contains("中断") || 
                         errorMsg.contains("连接") ||
                         errorMsg.contains("Socket") ||
                         errorMsg.contains("Pipe") ||
                         errorMsg.contains("closed") ||
                         errorMsg.contains("Software caused connection abort"));
                    
                    if (isNormalDisconnect) {
                        log.debug("轻量级心跳发送失败，客户端连接已断开: {}", errorMsg);
                    } else {
                        log.warn("轻量级心跳发送失败: {}", errorMsg);
                    }
                    isCompleted.set(true);
                    lightScheduler.shutdown();
                }
            } else {
                // 已完成，关闭调度器
                lightScheduler.shutdown();
            }
        }, 5, 5, TimeUnit.SECONDS); // 每5秒发送一次轻量级心跳
    }

    /**
     * 检查 emitter 是否已经完成
     */
    private boolean isEmitterCompleted(SseEmitter emitter) {
        // 首先检查emitter是否为null
        if (emitter == null) {
            return true;
        }
        
        try {
            // 尝试发送一个空消息来检查emitter状态
            emitter.send(SseEmitter.event().name("ping").data(""));
            return false; // 如果发送成功，说明emitter未完成
        } catch (IllegalStateException | org.springframework.web.context.request.async.AsyncRequestNotUsableException e) {
            return true; // 如果抛出这些异常，说明emitter已完成
        } catch (Exception e) {
            // 其他异常也认为emitter可能已完成
            return true;
        }
    }
    
    /**
     * 安全地完成 emitter
     */
    public void completeEmitter(SseEmitter emitter, AtomicBoolean isCompleted) {
        if (!isCompleted.getAndSet(true)) {
            try {
                emitter.complete();
                log.debug("SSE连接已正常关闭");
            } catch (IllegalStateException e) {
                log.warn("Emitter已经完成: {}", e.getMessage());
            } catch (Exception e) {
                // 检查是否是响应已提交的异常
                if (e.getMessage() != null && e.getMessage().contains("response has already been committed")) {
                    log.debug("响应已提交，无法完成Emitter");
                } else {
                    log.error("关闭SSE连接时发生错误", e);
                }
            }
        } else {
            log.debug("Emitter已经标记为完成，无需再次关闭");
        }
    }

    /**
     * 发送错误信息到 SSE
     */
    public void sendError(SseEmitter emitter, String message) {
        // 检查参数
        if (emitter == null) {
            log.warn("无法发送错误信息，emitter为null");
            return;
        }
        
        if (message == null || message.isEmpty()) {
            log.warn("无法发送错误信息，消息为空");
            return;
        }
        
        // 检查emitter是否已经完成
        if (isEmitterCompleted(emitter)) {
            log.debug("Emitter已经完成，无法发送错误信息: {}", message);
            return;
        }
        
        try {
            Map<String, Object> errorData = new HashMap<>();
            errorData.put("error", message);
            errorData.put("timestamp", System.currentTimeMillis());
            emitter.send(SseEmitter.event().name("error").data(errorData));
            log.info("已发送错误信息到客户端: {}", message);
            completeEmitter(emitter, new AtomicBoolean(true));
        } catch (org.springframework.web.context.request.async.AsyncRequestNotUsableException e) {
            // 客户端已断开连接导致的异步请求不可用异常
            log.debug("客户端连接中断，无法发送错误信息: {}", e.getMessage());
        } catch (IOException e) {
            log.error("发送错误信息失败: {}", message, e);
            completeEmitter(emitter, new AtomicBoolean(true));
        } catch (IllegalStateException e) {
            log.warn("Emitter已经完成，无法发送错误信息: {}", message, e);
        } catch (java.lang.RuntimeException e) {
            // 处理客户端断开连接导致的运行时异常
            if (e.getCause() instanceof java.io.IOException) {
                log.debug("客户端连接中断，无法发送错误信息: {}", message);
            } else {
                // 检查是否是响应已提交的异常
                if (e.getMessage() != null && e.getMessage().contains("response has already been committed")) {
                    log.debug("响应已提交，无法发送错误信息: {}", message);
                } else {
                    log.error("发送错误信息时发生运行时异常: {}", message, e);
                }
            }
        } catch (Exception e) {
            // 检查是否是响应已提交的异常
            if (e.getMessage() != null && e.getMessage().contains("response has already been committed")) {
                log.debug("响应已提交，无法发送错误信息: {}", message);
            } else {
                log.error("发送错误信息时发生未知异常: {}", message, e);
            }
            completeEmitter(emitter, new AtomicBoolean(true));
        }
    }
}