// DOM同步服务
import { unescapeHtmlContent } from '@/utils/stringUtils';
import { addLog } from '@/utils/logUtils';

export class DomSyncService {
  private chunkBuffer: Map<number, string> = new Map();
  private totalChunks: number = 0;
  private receivedChunks: number = 0;
  private chunkTimeoutId: number | null = null;

  // 已参数化：handleChunkData() 方法（不再需要，改为使用二进制协议）
  // 此方法外氺了Base64解码步骤，现在二进制协议直接传输UTF-8数据

  // 处理DOM同步数据
  handleDomSyncData(data: any, iframeDoc: Document, onIframeEventListenersChanged: () => void) {
    try {
      addLog(`📥 DOM同步数据处理开始，数据类型: ${typeof data}`, 'debug');
      
      // 检查数据是否有效
      if (!data) {
        addLog('接收到空数据', 'error');
        return;
      }
      
      // 记录数据大小
      if (typeof data === 'string') {
        addLog(`接收到DOM数据，长度: ${data.length} 字符`, 'debug');
      }
      
      // 直接渲染DOM内容，不再使用type字段区分
      this.renderFullDomInIframe({ dom: data }, iframeDoc, onIframeEventListenersChanged);
      addLog('已初始化页面DOM', 'init');
    } catch (e) {
      addLog('处理DOM同步数据失败：' + (e as Error).message, 'error');
    }
  }

  // 绑定事件监听器的辅助方法
  private bindEventListeners(onIframeEventListenersChanged: () => void) {
    setTimeout(() => {
      try {
        onIframeEventListenersChanged();
        addLog('事件监听器重新绑定成功', 'info');
      } catch (rebindError) {
        addLog('事件监听器重新绑定失败: ' + (rebindError as Error).message, 'error');
      }
    }, 100);
  }

  // 在iframe中渲染完整DOM
  private renderFullDomInIframe(data: any, doc: Document, onIframeEventListenersChanged: () => void) {
    try {
      // 检查必要数据
      if (!data.dom) {
        addLog('缺少DOM数据', 'error');
        return;
      }
      
      const domContent = data.dom;
      
      // 注意：后端已经不再进行手动转义。fastjson2会自动处理JSON中的特殊字符
      // 因此，domContent已经是有效的HTML内容了
      let actualHtmlContent = domContent;
      
      // 验证HTML内容的基本有效性
      if (!actualHtmlContent || actualHtmlContent.trim().length < 10) {
        addLog('DOM内容无效或过短（' + (actualHtmlContent ? actualHtmlContent.length : 0) + '字节）', 'error');
        return;
      }
      
      addLog('开始在iframe中渲染DOM，内容长度: ' + actualHtmlContent.length + ' 字节', 'info');
      
      // 使用最简单有效的方法：直接设置innerHTML
      try {
        // 清空现有内容
        doc.documentElement.innerHTML = '';
        
        // 使用 DOMParser 解析HTML，更安全且不会导致脚本中断
        const parser = new DOMParser();
        const parsedDoc = parser.parseFromString(actualHtmlContent, 'text/html');
        
        if (parsedDoc && parsedDoc.documentElement) {
          // 导入新的documentElement
          const newHtmlElement = doc.importNode(parsedDoc.documentElement, true);
          doc.replaceChild(newHtmlElement, doc.documentElement);
          addLog('文档内容替换成功', 'info');
          
          // 重新绑定事件监听器
          this.bindEventListeners(onIframeEventListenersChanged);
          return;
        }
      } catch (error) {
        addLog('DOMParser方案失败: ' + (error as Error).message + '，尝试备用方案...', 'warn');
      }
      
      // 备用方案：使用 document.open/write
      try {
        addLog('尝试使用document.open/write方案...', 'info');
        
        // 禁用动画和过渡，减少视觉闪烁
        doc.documentElement.style.animation = 'none';
        doc.documentElement.style.transition = 'none';
        
        doc.open();
        doc.write(actualHtmlContent);
        doc.close();
        
        addLog('document.write方案成功', 'info');
        
        // 重新绑定事件监听器
        this.bindEventListeners(onIframeEventListenersChanged);
        return;
      } catch (writeError) {
        addLog('document.write方案失败: ' + (writeError as Error).message, 'error');
      }
      
      // 最后的尝试：直接设置innerHTML到body
      try {
        addLog('尝试最后的方案：直接使用innerHTML...', 'warn');
        if (doc.body) {
          doc.body.innerHTML = actualHtmlContent;
          this.bindEventListeners(onIframeEventListenersChanged);
        }
      } catch (lastError) {
        addLog('所有渲染方案都失败了，请检查HTML内容的有效性: ' + (lastError as Error).message, 'error');
      }
    } catch (e) {
      addLog('在iframe中渲染完整DOM失败：' + (e as Error).message, 'error');
      addLog('错误堆栈：' + (e as any).stack, 'error');
      addLog('DOM数据预览（前200字符）：' + (data.dom ? data.dom.substring(0, 200) + '...' : '无'), 'error');
    }
  }
  
  // 在iframe中更新增量 DOM（已移除，使用完整DOM更新替代）
  private updateIncrementalDomInIframe(data: any, doc: Document) {
    // 不再使用增量更新，改为完整DOM更新以简化实现
    addLog('增量DOM更新已被移除，使用完整DOM更新替代', 'warn');
  }
}