// 统一的事件处理服务
import type { TimelineEvent } from '../types/timeline';

export class EventProcessingService {

  /**
   * 标准化事件对象
   * @param event 原始事件数据
   * @returns 标准化的事件对象
   */
  normalizeEvent(event: any): TimelineEvent {
    // 确保时间戳存在
    const timestamp = event.timestamp || Date.now();

    // 根据事件类型创建相应类型的事件对象
    switch (event.type) {
      case 'thought':
        return {
          type: 'thought',
          title: event.title || '思考事件',
          content: event.content || '',
          thinkingType: event.thinkingType,
          metadata: event.metadata,
          timestamp
        };

      case 'tool_call':
        return {
          type: 'tool_call',
          title: event.title || '工具调用',
          toolName: event.toolName || '',
          toolAction: event.toolAction || '',
          toolInput: event.toolInput,
          toolStatus: event.toolStatus,
          metadata: event.metadata,
          timestamp
        };

      case 'tool_result':
        return {
          type: 'tool_result',
          title: event.title || '工具结果',
          toolName: event.toolName || '',
          toolAction: event.toolAction || '',
          toolOutput: event.toolOutput,
          toolStatus: event.toolStatus,
          executionTime: event.executionTime,
          metadata: event.metadata,
          timestamp
        };

      case 'tool_error':
        return {
          type: 'tool_error',
          title: event.title || '工具错误',
          toolName: event.toolName || '',
          errorMessage: event.errorMessage || '',
          errorCode: event.errorCode,
          metadata: event.metadata,
          timestamp
        };

      case 'embed':
        return {
          type: 'embed',
          title: event.title || '嵌入内容',
          embedUrl: event.embedUrl || '',
          embedType: event.embedType,
          embedTitle: event.embedTitle,
          embedHtmlContent: event.embedHtmlContent,
          metadata: event.metadata,
          timestamp
        };

      default:
        return {
          type: event.type || 'thought',
          title: event.title || '未命名事件',
          metadata: event.metadata,
          timestamp
        };
    }
  }

  /**
   * 处理事件类型转换
   * @param event 事件对象
   * @returns 处理后的事件对象
   */
  processEventType(event: any): any {
    // 处理thinking类型的事件，如果是final_answer则转换为result类型
    const processedEvent = { ...event };
    if (processedEvent.type === 'thought' && processedEvent.title === '最终答案') {
      processedEvent.type = 'result';
    }
    return processedEvent;
  }


}