package pangea.hiagent.service;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.messages.AssistantMessage;
import org.springframework.ai.chat.messages.Message;
import org.springframework.ai.chat.messages.SystemMessage;
import org.springframework.ai.chat.messages.UserMessage;
import org.springframework.ai.chat.model.ChatModel;
import org.springframework.ai.chat.model.ChatResponse;
import org.springframework.ai.chat.prompt.Prompt;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import pangea.hiagent.model.Agent;
import pangea.hiagent.model.Tool;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.HashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * ReAct Agent服务类
 * 负责实现ReAct Agent的核心逻辑
 */
@Slf4j
@Service
public class ReActAgentService {
    
    @Autowired
    private ChatModel chatModel;
    
    @Autowired
    private ToolService toolService;
    
    @Autowired
    private AgentService agentService;
    
    /**
     * 处理用户请求的主方法
     * 
     * @param agent Agent对象
     * @param userMessage 用户消息
     * @return 处理结果
     */
    public String processRequest(Agent agent, String userMessage) {
        log.info("开始处理ReAct Agent请求，Agent ID: {}, 用户消息: {}", agent.getId(), userMessage);
        
        // 初始化对话历史
        List<Message> conversationHistory = new ArrayList<>();
        
        // 添加系统提示词
        String systemPrompt = agent.getSystemPrompt() != null ? agent.getSystemPrompt() : 
            "You are a helpful AI assistant that can use tools to help answer questions. " +
            "Use the following format to think through problems step by step:\n\n" +
            "Thought: Think about what to do.\n" +
            "Action: Call a tool with parameters.\n" +
            "Observation: Observe the result of the tool call.\n\n" +
            "Continue this process until you have enough information to answer the question.\n" +
            "Finally, provide your answer in the format:\n" +
            "Final Answer: [Your answer here]";
        conversationHistory.add(new SystemMessage(systemPrompt));
        
        // 添加用户消息
        conversationHistory.add(new UserMessage(userMessage));
        
        // 执行ReAct循环
        for (int i = 0; i < 5; i++) { // 最多执行5轮
            log.info("执行ReAct循环第 {} 轮", i + 1);
            
            // 构建Prompt
            Prompt prompt = new Prompt(conversationHistory);
            
            // 调用模型
            ChatResponse response = chatModel.call(prompt);
            String aiResponse = response.getResult().getOutput().getText();
            
            log.info("模型响应: {}", aiResponse);
            
            // 检查是否需要工具调用
            if (needsToolCall(aiResponse)) {
                log.info("检测到需要工具调用");
                
                // 解析工具调用
                ToolCall toolCall = parseToolCall(aiResponse);
                
                if (toolCall != null) {
                    log.info("解析到工具调用: 工具名称={}, 参数={}", toolCall.getToolName(), toolCall.getParameters());
                    
                    // 执行工具调用
                    String toolResult = toolService.executeTool(toolCall.getToolName(), toolCall.getParameters());
                    
                    log.info("工具执行结果: {}", toolResult);
                    
                    // 将工具结果添加到对话历史
                    conversationHistory.add(new AssistantMessage(aiResponse));
                    conversationHistory.add(new UserMessage("Observation: " + toolResult));
                } else {
                    log.warn("工具调用解析失败，结束ReAct循环");
                    break;
                }
            } else {
                log.info("无需工具调用，返回最终答案");
                // 返回最终结果
                String finalAnswer = extractFinalAnswer(aiResponse);
                log.info("最终答案: {}", finalAnswer);
                return finalAnswer;
            }
        }
        
        log.warn("达到最大执行轮次限制");
        return "达到最大执行轮次限制";
    }
    
    /**
     * 判断是否需要工具调用
     * 
     * @param response 模型响应
     * @return 是否需要工具调用
     */
    private boolean needsToolCall(String response) {
        return response.contains("Action:");
    }
    
    /**
     * 解析工具调用
     * 
     * @param response 模型响应
     * @return 工具调用对象
     */
    private ToolCall parseToolCall(String response) {
        // 使用正则表达式解析Action行
        // 支持多种格式：Action: tool_name(param1=value1, param2=value2) 或 Action: tool_name()
        Pattern actionPattern = Pattern.compile("Action:\\s*(\\w+)\\s*\\(([^)]*)\\)");
        Matcher matcher = actionPattern.matcher(response);
        
        if (matcher.find()) {
            String toolName = matcher.group(1);
            String paramsStr = matcher.group(2);
            
            // 解析参数
            Map<String, Object> parameters = parseParameters(paramsStr);
            
            return new ToolCall(toolName, parameters);
        }
        
        return null;
    }
    
    /**
     * 解析参数字符串
     * 
     * @param paramsStr 参数字符串
     * @return 参数Map
     */
    private Map<String, Object> parseParameters(String paramsStr) {
        Map<String, Object> parameters = new HashMap<>();
        
        // 如果参数字符串为空，直接返回空Map
        if (paramsStr == null || paramsStr.trim().isEmpty()) {
            return parameters;
        }
        
        // 解析参数，支持带引号和不带引号的值
        // 格式: key1=value1, key2="value with spaces", key3='another value'
        String[] paramPairs = paramsStr.split(","); // 简化处理，按逗号分割
        
        for (String paramPair : paramPairs) {
            String[] parts = paramPair.split("=");
            if (parts.length == 2) {
                String key = parts[0].trim();
                String value = parts[1].trim();
                
                // 移除引号（如果存在）
                if ((value.startsWith("\"") && value.endsWith("\"")) || 
                    (value.startsWith("'") && value.endsWith("'"))) {
                    value = value.substring(1, value.length() - 1);
                }
                
                parameters.put(key, value);
            }
        }
        
        return parameters;
    }
    
    /**
     * 提取最终答案
     * 
     * @param response 模型响应
     * @return 最终答案
     */
    private String extractFinalAnswer(String response) {
        // 查找Final Answer:后面的文本
        Pattern answerPattern = Pattern.compile("Final Answer:\\s*(.*)", Pattern.DOTALL);
        Matcher matcher = answerPattern.matcher(response);
        
        if (matcher.find()) {
            return matcher.group(1).trim();
        }
        
        // 如果没有找到Final Answer格式，返回整个响应
        return response;
    }
    
    /**
     * 工具调用内部类
     */
    private static class ToolCall {
        private final String toolName;
        private final Map<String, Object> parameters;
        
        public ToolCall(String toolName, Map<String, Object> parameters) {
            this.toolName = toolName;
            this.parameters = parameters;
        }
        
        public String getToolName() {
            return toolName;
        }
        
        public Map<String, Object> getParameters() {
            return parameters;
        }
    }
}