package pangea.hiagent.web.controller;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import pangea.hiagent.document.KnowledgeBaseInitializationService;
import pangea.hiagent.web.dto.ApiResponse;
import pangea.hiagent.tool.ToolBeanNameInitializer;

/**
 * 系统管理控制器
 * 提供系统级别的管理功能，如手动触发初始化任务等
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/admin/system")
@Tag(name = "系统管理", description = "系统管理相关API")
public class SystemAdminController {

    @Autowired
    private ToolBeanNameInitializer toolBeanNameInitializer;

    @Autowired
    private KnowledgeBaseInitializationService knowledgeBaseInitializationService;

    /**
     * 手动触发工具Bean名称初始化
     * 
     * @return 操作结果
     */
    @PostMapping("/initialize-tool-beans")
    @Operation(summary = "初始化工具Bean", description = "手动触发工具Bean名称初始化任务")
    public ResponseEntity<ApiResponse<Void>> initializeToolBeans() {
        try {
            log.info("收到手动触发工具Bean初始化请求");
            toolBeanNameInitializer.initializeToolBeanNamesManually();
            log.info("工具Bean初始化完成");
            return ResponseEntity.ok(ApiResponse.success(null, "工具Bean初始化完成"));
        } catch (Exception e) {
            log.error("工具Bean初始化失败", e);
            return ResponseEntity.internalServerError()
                    .body(ApiResponse.error(500, "工具Bean初始化失败: " + e.getMessage()));
        }
    }

    /**
     * 手动触发知识库初始化
     * 
     * @return 操作结果
     */
    @PostMapping("/initialize-knowledge-bases")
    @Operation(summary = "初始化知识库", description = "手动触发知识库初始化任务")
    public ResponseEntity<ApiResponse<Void>> initializeKnowledgeBases() {
        try {
            log.info("收到手动触发知识库初始化请求");
            knowledgeBaseInitializationService.initializeKnowledgeBasesManually();
            log.info("知识库初始化完成");
            return ResponseEntity.ok(ApiResponse.success(null, "知识库初始化完成"));
        } catch (Exception e) {
            log.error("知识库初始化失败", e);
            return ResponseEntity.internalServerError()
                    .body(ApiResponse.error(500, "知识库初始化失败: " + e.getMessage()));
        }
    }
}