package pangea.hiagent.agent.processor;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import pangea.hiagent.model.Agent;
import java.util.List;
import java.util.Map;
import java.util.HashMap;
import java.util.StringJoiner;

/**
 * Agent处理器工厂类
 * 根据Agent类型创建相应的处理器实例
 */
@Slf4j
@Component
public class AgentProcessorFactory {
    
    @Autowired
    private List<AgentProcessor> agentProcessors;
    
    private final Map<String, AgentProcessor> processorCache = new HashMap<>();
    
    /**
     * 根据Agent获取对应的处理器
     * 
     * @param agent Agent对象
     * @return 对应的处理器实例
     */
    public AgentProcessor getProcessor(Agent agent) {
        if (agent == null) {
            throw new IllegalArgumentException("Agent不能为null");
        }
        
        // 生成缓存键
        String cacheKey = generateCacheKey(agent);
        
        // 先从缓存中获取
        AgentProcessor processor = processorCache.get(cacheKey);
        if (processor != null) {
            return processor;
        }
        
        // 根据Agent配置确定处理器类型
        String processorType = determineProcessorType(agent);
        
        // 查找对应的处理器
        for (AgentProcessor agentProcessor : agentProcessors) {
            if (agentProcessor.getProcessorType().equals(processorType)) {
                // 放入缓存
                processorCache.put(cacheKey, agentProcessor);
                return agentProcessor;
            }
        }
        
        // 如果没有找到匹配的处理器，抛出异常并提供更多信息以便调试
        String errorMsg = buildErrorMessage(processorType);
        log.error(errorMsg);
        throw new IllegalStateException(errorMsg);
    }
    
    /**
     * 构建错误信息
     * 
     * @param processorType 处理器类型
     * @return 错误信息
     */
    private String buildErrorMessage(String processorType) {
        StringBuilder errorMsg = new StringBuilder();
        errorMsg.append("未找到适合Agent的处理器，类型: ").append(processorType);
        errorMsg.append(", 可用处理器数量: ").append(agentProcessors != null ? agentProcessors.size() : 0);
        
        if (agentProcessors != null && !agentProcessors.isEmpty()) {
            StringJoiner joiner = new StringJoiner(", ");
            for (AgentProcessor processor : agentProcessors) {
                joiner.add(processor.getProcessorType());
            }
            errorMsg.append(", 可用处理器类型: ").append(joiner.toString());
        }
        
        return errorMsg.toString();
    }
    
    /**
     * 根据Agent配置确定处理器类型
     * 
     * @param agent Agent对象
     * @return 处理器类型
     */
    private String determineProcessorType(Agent agent) {
        // 如果启用了ReAct模式，则使用ReAct处理器
        if (Boolean.TRUE.equals(agent.getEnableReAct())) {
            return "react";
        }
        
        // 默认使用普通处理器
        return "normal";
    }
    
    /**
     * 生成缓存键
     * 
     * @param agent Agent对象
     * @return 缓存键
     */
    private String generateCacheKey(Agent agent) {
        return agent.getId() + "_" + (agent.getEnableReAct() != null ? agent.getEnableReAct() : false);
    }
    
    /**
     * 清除缓存
     */
    public void clearCache() {
        processorCache.clear();
    }
}