package pangea.hiagent.tool.impl;

import com.fasterxml.jackson.annotation.JsonClassDescription;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyDescription;
import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;
import org.springframework.http.ResponseEntity;

// 天气API响应数据结构
class WeatherApiResponse {
    public Result[] results;
    
    static class Result {
        public Location location;
        public Now now;
        public String last_update;
    }
    
    static class Location {
        public String id;
        public String name;
        public String country;
        public String path;
        public String timezone;
        public String timezone_offset;
    }
    
    static class Now {
        public String text; // 天气现象文字
        public String code; // 天气现象代码
        public String temperature; // 温度
        public String feels_like; // 体感温度
        public String pressure; // 气压
        public String humidity; // 相对湿度
        public String visibility; // 能见度
        public String wind_direction; // 风向文字
        public String wind_direction_degree; // 风向角度
        public String wind_speed; // 风速
        public String wind_scale; // 风力等级
        public String clouds; // 云量
        public String dew_point; // 露点温度
    }
}

/**
 * 天气查询工具类
 * 提供城市天气信息查询功能
 */
@Slf4j
@Component
public class WeatherFunction {
    
    private final RestTemplate restTemplate;
    
    public WeatherFunction(RestTemplate restTemplate) {
        this.restTemplate = restTemplate;
    }
    
    // API密钥常量
    private static final String API_KEY = "SNCqjZiDAIPXdhE_O";
    
    @JsonClassDescription("获取指定位置的天气信息")
    public record Request(
        @JsonProperty(required = true, value = "location")
        @JsonPropertyDescription("位置名称") 
        String location,
        
        @JsonProperty(required = false, value = "language")
        @JsonPropertyDescription("语言，默认值：zh-Hans") 
        String language,
        
        @JsonProperty(required = false, value = "unit")
        @JsonPropertyDescription("单位，默认值：c") 
        String unit
    ) {}
    
    @JsonClassDescription("天气信息响应")
    public record Response(
        @JsonPropertyDescription("位置名称") String locationName,
        @JsonPropertyDescription("温度") String temperature,
        @JsonPropertyDescription("体感温度") String feelsLike,
        @JsonPropertyDescription("湿度") String humidity,
        @JsonPropertyDescription("天气状况") String condition,
        @JsonPropertyDescription("风向") String windDirection,
        @JsonPropertyDescription("风力等级") String windScale,
        @JsonPropertyDescription("气压") String pressure,
        @JsonPropertyDescription("能见度") String visibility,
        @JsonPropertyDescription("数据更新时间") String lastUpdate
    ) {}
    
    @Tool(description = "获取指定位置的天气信息")
    public Response getWeather(Request request) {
        log.debug("查询天气信息，位置: {}, 语言: {}, 单位: {}", request.location, request.language, request.unit);
        
        try {
            // 构建API请求URL
            String baseUrl = "https://api.seniverse.com/v3/weather/now.json";
            
            // 设置默认参数
            String language = request.language != null ? request.language : "zh-Hans";
            String unit = request.unit != null ? request.unit : "c";
            
            // 构建完整URL
            String url = String.format("%s?key=%s&location=%s&language=%s&unit=%s", 
                                      baseUrl, API_KEY, request.location, language, unit);
            
            ResponseEntity<WeatherApiResponse> responseEntity = restTemplate.getForEntity(url, WeatherApiResponse.class);
            
            if (responseEntity.getStatusCode().is2xxSuccessful() && responseEntity.getBody() != null) {
                WeatherApiResponse apiResponse = responseEntity.getBody();
                
                if (apiResponse.results != null && apiResponse.results.length > 0) {
                    WeatherApiResponse.Result result = apiResponse.results[0];
                    WeatherApiResponse.Now now = result.now;
                    
                    // 构建响应数据
                    Response response = new Response(
                        result.location.name,
                        now.temperature + "°" + ("c".equals(unit) ? "C" : "F"),
                        now.feels_like + "°" + ("c".equals(unit) ? "C" : "F"),
                        now.humidity + "%",
                        now.text,
                        now.wind_direction,
                        now.wind_scale + "级",
                        now.pressure + "mb",
                        now.visibility + "km",
                        result.last_update
                    );
                    
                    log.debug("天气查询结果: {}", response);
                    return response;
                } else {
                    log.error("天气API返回结果为空");
                }
            } else {
                log.error("天气API调用失败，HTTP状态码: {}", responseEntity.getStatusCode());
            }
        } catch (Exception e) {
            log.error("天气API调用异常: ", e);
        }
        
        // 如果API调用失败，返回默认值
        Response response = new Response(
            request.location,
            "",
            "",
            "",
            "",
            "",
            "",
            "",
            "",
            ""
        );
        log.debug("天气查询结果(默认值): {}", response);
        return response;
    }
}