package pangea.hiagent.agent.react;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.client.ChatClient;
import org.springframework.ai.chat.messages.*;
import org.springframework.ai.chat.model.ChatResponse;
import org.springframework.ai.chat.prompt.Prompt;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import pangea.hiagent.agent.service.ErrorHandlerService;
import pangea.hiagent.agent.service.SseTokenEmitter;
import pangea.hiagent.agent.service.TokenConsumerWithCompletion;
import pangea.hiagent.memory.MemoryService;
import pangea.hiagent.model.Agent;
import pangea.hiagent.tool.AgentToolManager;
import pangea.hiagent.common.utils.UserUtils;
import java.util.List;
import java.util.ArrayList;
import java.util.function.Consumer;

/**
 * 简化的默认ReAct执行器实现
 */
@Slf4j
@Service
public class DefaultReactExecutor implements ReactExecutor {

    @Value("${hiagent.react.system-prompt}")
    private String defaultSystemPrompt;

    private final List<ReactCallback> reactCallbacks = new ArrayList<>();

    private final ReactCallback reactCallback;

    private MemoryService memoryService;

    private ErrorHandlerService errorHandlerService;

    private final AgentToolManager agentToolManager;

    public DefaultReactExecutor(ReactCallback reactCallback, AgentToolManager agentToolManager,
            MemoryService memoryService, ErrorHandlerService errorHandlerService) {
        this.reactCallback = reactCallback;
        this.agentToolManager = agentToolManager;
        this.memoryService = memoryService;
        this.errorHandlerService = errorHandlerService;
    }

    @Override
    public void addReactCallback(ReactCallback callback) {
        if (callback != null) {
            reactCallbacks.add(callback);
        }
    }

    @Override
    public String execute(ChatClient chatClient, String userInput, List<Object> tools, Agent agent) {
        // 调用带用户ID的方法，首先尝试获取当前用户ID
        String userId = UserUtils.getCurrentUserIdStatic();
        return execute(chatClient, userInput, tools, agent, userId);
    }

    @Override
    public String execute(ChatClient chatClient, String userInput, List<Object> tools, Agent agent, String userId) {
        log.info("开始执行ReAct流程，用户输入: {}", userInput);

        List<Object> agentTools = getAgentTools(agent);

        try {
            Prompt prompt = buildPromptWithHistory(defaultSystemPrompt, userInput, agent, userId);

            ChatResponse response = chatClient.prompt(prompt)
                    .tools(agentTools.toArray())
                    .call()
                    .chatResponse();

            String responseText = response.getResult().getOutput().getText();

            log.info("最终答案: {}", responseText);

            // 保存助手回复到内存，使用提供的用户ID
            saveAssistantResponseToMemory(agent, responseText, userId);

            return responseText;
        } catch (Exception e) {
            log.error("执行ReAct流程时发生错误", e);
            return handleReActError(e);
        }
    }

    /**
     * 处理ReAct执行过程中发生的错误
     * 
     * @param e 发生的异常
     * @return 错误处理结果
     */
    private String handleReActError(Exception e) {
        log.error("ReAct执行过程中发生错误", e);
        return errorHandlerService.handleSyncError(e, "处理ReAct请求时发生错误");
    }

    /**
     * 构建带有历史记录的提示词
     * 
     * @param systemPrompt 系统提示词
     * @param userInput    用户输入
     * @param agent        智能体对象
     * @param userId       用户ID（可选，如果为null则自动获取）
     * @return 构建好的提示词对象
     */
    private Prompt buildPromptWithHistory(String systemPrompt, String userInput, Agent agent, String userId) {
        List<org.springframework.ai.chat.messages.Message> messages = new ArrayList<>();

        messages.add(new SystemMessage(systemPrompt));

        if (agent != null) {
            try {
                // 如果没有提供用户ID，则尝试获取当前用户ID
                if (userId == null) {
                    userId = UserUtils.getCurrentUserIdStatic();
                }
                String sessionId = memoryService.generateSessionId(agent, userId);

                int historyLength = agent.getHistoryLength() != null ? agent.getHistoryLength() : 10;

                List<org.springframework.ai.chat.messages.Message> historyMessages = memoryService
                        .getHistoryMessages(sessionId, historyLength);

                messages.addAll(historyMessages);

                memoryService.addUserMessageToMemory(sessionId, userInput);
            } catch (Exception e) {
                log.warn("获取历史对话记录时发生错误: {}", e.getMessage());
            }
        }

        messages.add(new UserMessage(userInput));

        return new Prompt(messages);
    }

    @Override
    public void executeStream(ChatClient chatClient, String userInput, List<Object> tools,
            SseTokenEmitter tokenConsumer, Agent agent) {
        // 调用带用户ID的方法，但首先尝试获取当前用户ID
        String userId = UserUtils.getCurrentUserIdStatic();
        executeStream(chatClient, userInput, tools, tokenConsumer, agent, userId);
    }

    @Override
    public void executeStream(ChatClient chatClient, String userInput, List<Object> tools,
            SseTokenEmitter tokenConsumer, Agent agent, String userId) {
        log.info("使用stream()方法处理ReAct流程，支持真正的流式输出");

        List<Object> agentTools = getAgentTools(agent);

        StringBuilder fullResponse = new StringBuilder();

        try {
            Prompt prompt = buildPromptWithHistory(defaultSystemPrompt, userInput, agent, userId);

            chatClient.prompt(prompt)
                    .tools(agentTools.toArray())
                    .stream()
                    .chatResponse()
                    .subscribe(
                            chatResponse -> handleTokenResponse(chatResponse, tokenConsumer, fullResponse),
                            throwable -> handleStreamError(throwable, tokenConsumer),
                            () -> handleStreamCompletion(tokenConsumer, fullResponse, agent, userId));

        } catch (Exception e) {
            log.error("流式执行ReAct流程时发生错误", e);
            errorHandlerService.handleReactFlowError(e, tokenConsumer);
        }
    }

    /**
     * 处理流式响应中的单个token
     * 
     * @param chatResponse  聊天响应对象
     * @param tokenConsumer token消费者
     * @param fullResponse  完整响应构建器
     */
    private void handleTokenResponse(org.springframework.ai.chat.model.ChatResponse chatResponse,
            SseTokenEmitter tokenConsumer, StringBuilder fullResponse) {
        try {
            String token = chatResponse.getResult().getOutput().getText();

            if (isValidToken(token)) {
                fullResponse.append(token);

                if (tokenConsumer != null) {
                    tokenConsumer.accept(token);
                }

                reactCallback.onToken(token);
            }
        } catch (Exception e) {
            log.error("处理token时发生错误", e);
            errorHandlerService.handleReactFlowError(e, tokenConsumer);
        }
    }

    /**
     * 处理流式响应完成事件
     * 
     * @param tokenConsumer token消费者
     * @param fullResponse  完整响应内容
     * @param agent         智能体对象
     * @param userId        用户ID
     */
    private void handleStreamCompletion(SseTokenEmitter tokenConsumer, StringBuilder fullResponse, Agent agent,
            String userId) {
        try {
            log.info("流式处理完成");

            reactCallback.endStream();

            String responseStr = fullResponse.toString();
            saveAssistantResponseToMemory(agent, responseStr, userId);

            // 发送完成事件，触发对话历史保存到数据库
            tokenConsumer.onComplete(responseStr);

        } catch (Exception e) {
            log.error("处理流式完成回调时发生错误", e);
            handleCompletionError(tokenConsumer, e);
        }
    }

    /**
     * 将助手的回复保存到内存中
     * 
     * @param agent    智能体对象
     * @param response 助手的回复内容
     * @param userId   用户ID
     */
    private void saveAssistantResponseToMemory(Agent agent, String response, String userId) {
        if (agent != null) {
            try {
                String sessionId = memoryService.generateSessionId(agent, userId);
                memoryService.addAssistantMessageToMemory(sessionId, response);
            } catch (Exception e) {
                log.warn("保存助理回复到内存时发生错误: {}", e.getMessage());
            }
        }
    }

    /**
     * 处理完成事件时发生的错误
     * 
     * @param tokenConsumer token消费者
     * @param e             发生的异常
     */
    private void handleCompletionError(SseTokenEmitter tokenConsumer, Exception e) {
        if (tokenConsumer instanceof TokenConsumerWithCompletion) {
            try {
                String errorId = errorHandlerService.generateErrorId();
                String fullErrorMessage = errorHandlerService.buildFullErrorMessage("处理完成时发生错误", e, errorId, "ReAct");
                // 符合onComplete设计原则：在通讯操作失败后执行的最终操作
                // 触发条件：处理完成事件时发生异常
                // 通讯流程位置：错误处理流程的最终阶段，确保客户端收到完整的错误信息
                ((TokenConsumerWithCompletion) tokenConsumer).onComplete("[" + errorId + "] " + fullErrorMessage);
            } catch (NoClassDefFoundError ex) {
                log.error("TokenConsumerWithCompletion依赖类未找到，跳过完成回调: {}", ex.getMessage());
            } catch (Exception ex) {
                log.error("调用onComplete时发生错误", ex);
            }
        }
    }

    /**
     * 验证token是否有效
     * 
     * @param token 待验证的token
     * @return 如果token有效则返回true，否则返回false
     */
    private boolean isValidToken(String token) {
        return token != null && !token.isEmpty();
    }

    /**
     * 处理流式响应中的错误
     * 
     * @param throwable     异常对象
     * @param tokenConsumer token消费者
     */
    private void handleStreamError(Throwable throwable, SseTokenEmitter tokenConsumer) {
        errorHandlerService.handleStreamError(throwable, tokenConsumer, "ReAct流式处理");
    }

    // /**
    // * 发送完成事件
    // *
    // * @param tokenConsumer token消费者
    // * @param fullResponse 完整响应内容
    // */
    // private void sendCompletionEvent(SseTokenEmitter tokenConsumer, String
    // fullResponse) {
    // if (fullResponse == null) {
    // fullResponse = "";
    // }

    // if (tokenConsumer instanceof TokenConsumerWithCompletion) {
    // try {
    // // 符合onComplete设计原则：在所有通讯操作最终完成后执行
    // // 触发条件：ReAct执行流程完成，需要发送完整响应
    // // 通讯流程位置：执行流程的最终阶段，确保客户端收到完整的响应内容
    // ((TokenConsumerWithCompletion) tokenConsumer).onComplete(fullResponse);
    // } catch (NoClassDefFoundError e) {
    // log.error("TokenConsumerWithCompletion依赖类未找到，跳过完成回调: {}", e.getMessage());
    // // 如果类未找到，至少发送一个空消息以确保流的完整性
    // if (tokenConsumer != null) {
    // try {
    // tokenConsumer.accept("");
    // } catch (Exception ex) {
    // log.error("发送空消息也失败", ex);
    // }
    // }
    // } catch (Exception e) {
    // log.error("调用onComplete时发生错误", e);
    // }
    // } else if (tokenConsumer != null) {
    // tokenConsumer.accept("");
    // }
    // }

    /**
     * 获取智能体工具
     * 
     * @param agent 智能体对象
     * @return 智能体可用的工具列表
     */
    private List<Object> getAgentTools(Agent agent) {
        List<Object> tools = new ArrayList<>();

        if (agent != null) {
            try {
                tools = agentToolManager.getAvailableToolInstances(agent);
            } catch (Exception e) {
                log.error("获取工具实例时发生错误: {}", e.getMessage());
                // 发生异常时，tools 保持为空列表
            }
        }

        return tools;
    }
}