package pangea.hiagent.web.controller;

import lombok.extern.slf4j.Slf4j;
import pangea.hiagent.common.utils.UserUtils;
import pangea.hiagent.memory.MemoryService;
import pangea.hiagent.model.Agent;
import pangea.hiagent.web.dto.ApiResponse;
import pangea.hiagent.web.service.AgentService;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;



import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.HashMap;

/**
 * 内存管理控制器
 * 提供对话记忆和知识记忆的查询、管理和清理功能
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/memory")
public class MemoryController {

    @Autowired
    private MemoryService memoryService;

    @Autowired
    private AgentService agentService;

    /**
     * 获取对话记忆列表
     * 
     * @return 对话记忆列表
     */
    @GetMapping("/dialogue")
    public ApiResponse<List<Map<String, Object>>> getDialogueMemories() {
        try {
            String userId = UserUtils.getCurrentUserIdStatic();
            if (userId == null) {
                log.warn("用户未认证，无法获取对话记忆列表");
                return ApiResponse.error(401, "用户未认证");
            }

            log.info("用户 {} 请求获取对话记忆列表", userId);

            // 获取用户的所有Agent
            List<Agent> userAgents = agentService.getUserAgents(userId);            
            // 构建对话记忆信息列表
            List<Map<String, Object>> dialogueMemories = new ArrayList<>();
            for (Agent agent : userAgents) {
                String sessionId = memoryService.generateSessionId(agent, userId);
                
                Map<String, Object> memoryInfo = new HashMap<>();
                memoryInfo.put("id", sessionId);
                memoryInfo.put("agentName", agent.getName());
                memoryInfo.put("agentId", agent.getId());
                memoryInfo.put("userName", userId);
                memoryInfo.put("lastActive", "暂无数据"); // 实际应用中应从ChatMemory获取
                memoryInfo.put("messageCount", 0); // 实际应用中应从ChatMemory获取
                
                dialogueMemories.add(memoryInfo);
            }

            log.info("用户 {} 成功获取对话记忆列表，共 {} 条记录", userId, dialogueMemories.size());
            return ApiResponse.success(dialogueMemories, "获取对话记忆列表成功");
        } catch (Exception e) {
            log.error("获取对话记忆列表失败", e);
            return ApiResponse.error(500, "获取对话记忆列表失败: " + e.getMessage());
        }
    }

    /**
     * 获取知识记忆列表
     * 
     * @return 知识记忆列表
     */
    @GetMapping("/knowledge")
    public ApiResponse<List<Map<String, Object>>> getKnowledgeMemories() {
        try {
            String userId = UserUtils.getCurrentUserIdStatic();
            if (userId == null) {
                log.warn("用户未认证，无法获取知识记忆列表");
                return ApiResponse.error(401, "用户未认证");
            }

            log.info("用户 {} 请求获取知识记忆列表", userId);

            // 目前返回空列表，实际应用中应从知识库获取数据
            List<Map<String, Object>> knowledgeMemories = new ArrayList<>();

            log.info("用户 {} 成功获取知识记忆列表，共 {} 条记录", userId, knowledgeMemories.size());
            return ApiResponse.success(knowledgeMemories, "获取知识记忆列表成功");
        } catch (Exception e) {
            log.error("获取知识记忆列表失败", e);
            return ApiResponse.error(500, "获取知识记忆列表失败: " + e.getMessage());
        }
    }

    /**
     * 获取指定对话记忆的详细信息
     * 
     * @param agentId Agent ID
     * @return 对话记忆详情
     */
    @GetMapping("/dialogue/agent/{agentId}")
    public ApiResponse<Map<String, Object>> getDialogueMemoryDetail(@PathVariable String agentId) {
        try {
            String userId = UserUtils.getCurrentUserIdStatic();
            if (userId == null) {
                log.warn("用户未认证，无法获取对话记忆详情");
                return ApiResponse.error(401, "用户未认证");
            }
    
            // 获取Agent对象
            // 首先尝试直接根据ID获取Agent，如果失败则尝试解析为Long类型
            Agent agent = agentService.getAgent(agentId);
            if (agent == null) {
                try {
                    // 如果agentId是数字格式，则尝试通过数字ID获取
                    agent = agentService.getAgentById(Long.parseLong(agentId));
                } catch (NumberFormatException e) {
                    log.warn("无效的Agent ID格式: {}", agentId);
                }
            }
            
            if (agent == null) {
                log.warn("找不到ID为 {} 的Agent", agentId);
                return ApiResponse.error(404, "找不到对应的Agent");
            }
    
            // 使用MemoryService统一的生成会话 ID
            String sessionId = memoryService.generateSessionId(agent, userId);
            log.info("用户 {} 请求获取对话记忆详情，Agent ID: {}，会话 ID: {}", userId, agentId, sessionId);
    
            // 从数据库查询民史对话记录而非 ChatMemory
            java.util.List<pangea.hiagent.model.AgentDialogue> dialogueList = agentService.getDialogueHistory(agent.getId(), userId, 50);
                
            // 构建对话详情
            Map<String, Object> detail = new HashMap<>();
            detail.put("id", sessionId);
                
            // 转换消息列表格式
            java.util.List<Map<String, Object>> messages = new ArrayList<>();
            if (dialogueList != null && !dialogueList.isEmpty()) {
                for (pangea.hiagent.model.AgentDialogue dialogue : dialogueList) {
                    // 添加用户消息
                    if (dialogue.getUserMessage() != null && !dialogue.getUserMessage().trim().isEmpty()) {
                        Map<String, Object> userMsg = new HashMap<>();
                        userMsg.put("sender", "user");
                        userMsg.put("content", dialogue.getUserMessage());
                        userMsg.put("time", dialogue.getCreatedAt() != null ? 
                                   String.valueOf(dialogue.getCreatedAt().atZone(java.time.ZoneId.systemDefault()).toInstant().toEpochMilli()) : 
                                   String.valueOf(System.currentTimeMillis()));
                        messages.add(userMsg);
                    }
                        
                    // 添加助手回复
                    if (dialogue.getAgentResponse() != null && !dialogue.getAgentResponse().trim().isEmpty()) {
                        Map<String, Object> assistantMsg = new HashMap<>();
                        assistantMsg.put("sender", "assistant");
                        assistantMsg.put("content", dialogue.getAgentResponse());
                        assistantMsg.put("time", dialogue.getCreatedAt() != null ? 
                                        String.valueOf(dialogue.getCreatedAt().atZone(java.time.ZoneId.systemDefault()).toInstant().toEpochMilli()) : 
                                        String.valueOf(System.currentTimeMillis()));
                        messages.add(assistantMsg);
                    }
                }
            }
                
            detail.put("messages", messages);
    
            log.info("用户 {} 成功获取对话记忆详情，共 {} 条消息", userId, messages.size());
            return ApiResponse.success(detail, "获取对话记忆详情成功");
        } catch (Exception e) {
            log.error("获取对话记忆详情失败", e);
            return ApiResponse.error(500, "获取对话记忆详情失败: " + e.getMessage());
        }
    }
    /**
     * 清空指定对话记忆
     * 
     * @param sessionId 会话ID
     * @return 操作结果
     */
    @DeleteMapping("/dialogue/{sessionId}")
    public ApiResponse<Void> clearDialogueMemory(@PathVariable String sessionId) {
        try {
            String userId = UserUtils.getCurrentUserIdStatic();
            if (userId == null) {
                log.warn("用户未认证，无法清空对话记忆");
                return ApiResponse.error(401, "用户未认证");
            }

            log.info("用户 {} 请求清空对话记忆，会话ID: {}", userId, sessionId);

            // 验证会话ID是否属于当前用户
            if (!sessionId.startsWith(userId + "_")) {
                return ApiResponse.error(403, "无权操作该会话");
            }

            // 实际应用中应清除ChatMemory中的对应会话数据
            // memoryService.clearSession(sessionId);

            log.info("用户 {} 成功清空对话记忆", userId);
            return ApiResponse.success(null, "对话记忆已清空");
        } catch (Exception e) {
            log.error("清空对话记忆失败", e);
            return ApiResponse.error(500, "清空对话记忆失败: " + e.getMessage());
        }
    }

    /**
     * 删除指定知识记忆
     * 
     * @param id 知识记忆ID
     * @return 操作结果
     */
    @DeleteMapping("/knowledge/{id}")
    public ApiResponse<Void> deleteKnowledgeMemory(@PathVariable String id) {
        try {
            String userId = UserUtils.getCurrentUserIdStatic();
            if (userId == null) {
                log.warn("用户未认证，无法删除知识记忆");
                return ApiResponse.error(401, "用户未认证");
            }

            log.info("用户 {} 请求删除知识记忆，ID: {}", userId, id);

            // 实际应用中应从知识库删除对应的数据
            // knowledgeService.deleteById(id);

            log.info("用户 {} 成功删除知识记忆", userId);
            return ApiResponse.success(null, "知识记忆已删除");
        } catch (Exception e) {
            log.error("删除知识记忆失败", e);
            return ApiResponse.error(500, "删除知识记忆失败: " + e.getMessage());
        }
    }
}