package pangea.hiagent.agent.service;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;
import pangea.hiagent.model.Agent;
import pangea.hiagent.model.AgentDialogue;
// import pangea.hiagent.common.utils.ValidationUtils;  // 移除这个依赖
import pangea.hiagent.agent.processor.AgentProcessor;
import pangea.hiagent.agent.sse.UserSseService;
import pangea.hiagent.common.utils.LogUtils;
import pangea.hiagent.common.utils.UserUtils;
import pangea.hiagent.web.dto.AgentRequest;
import pangea.hiagent.web.service.AgentService;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 完成回调处理服务
 * 负责处理流式输出完成后的回调操作
 */
@Slf4j
@Service
public class CompletionHandlerService {
    
    @Autowired
    private AgentService agentService;
    
    @Autowired
    private UserSseService unifiedSseService;
    
    @Autowired
    private ErrorHandlerService errorHandlerService;
    
    /**
     * 处理完成回调
     *
     * @param emitter SSE发射器
     * @param processor Agent处理器
     * @param agent Agent对象
     * @param request Agent请求
     * @param userId 用户ID
     * @param fullContent 完整内容
     * @param isCompleted 完成状态标记
     */
    public void handleCompletion(SseEmitter emitter, AgentProcessor processor, Agent agent, 
                               AgentRequest request, String userId, 
                               String fullContent, AtomicBoolean isCompleted) {
        LogUtils.enterMethod("handleCompletion", emitter, processor, agent, request, userId);
        // 参数验证 - 内联验证逻辑，避免依赖ValidationUtils
        if (emitter == null) {
            log.error("SSE发射器不能为空");
            LogUtils.exitMethod("handleCompletion", "SSE发射器不能为空");
            return;
        }
        
        if (processor == null) {
            log.error("Agent处理器不能为空");
            LogUtils.exitMethod("handleCompletion", "Agent处理器不能为空");
            return;
        }
        
        if (agent == null) {
            log.error("Agent对象不能为空");
            LogUtils.exitMethod("handleCompletion", "Agent对象不能为空");
            return;
        }
        
        if (request == null) {
            log.error("Agent请求不能为空");
            LogUtils.exitMethod("handleCompletion", "Agent请求不能为空");
            return;
        }
        
        if (userId == null || userId.trim().isEmpty()) {
            log.error("用户ID不能为空");
            LogUtils.exitMethod("handleCompletion", "用户ID不能为空");
            return;
        }
        
        if (isCompleted == null) {
            log.error("完成状态标记不能为空");
            LogUtils.exitMethod("handleCompletion", "完成状态标记不能为空");
            return;
        }

        log.info("{} Agent处理完成，总字符数: {}", processor.getProcessorType(), fullContent != null ? fullContent.length() : 0);

        // 严格按照正确的SSE连接关闭顺序：
        // 1. 先完成所有业务处理：保存对话记录等
        // 2. 发送完成信号：发送[DONE]信号
        // 3. 关闭SSE连接：调用emitter.complete()方法完成连接
        // 4. 取消心跳任务：清理相关的ScheduledFuture心跳任务
        // 5. 移除连接映射：从连接管理器(userEmitters、emitterUsers、emitters)中移除连接映射
        
        // 1. 完成所有业务处理：保存对话记录
        Exception completionException = null;
        try {
            saveDialogue(agent, request, userId, fullContent);
            log.info("对话记录保存成功");
        } catch (Exception e) {
            log.error("保存对话记录失败", e);
            // 记录异常但不中断流程
            completionException = e;
        }
        
        log.debug("业务处理完成，准备发送完成信号");
        
        // 2. 发送完成信号：发送[DONE]信号
        try {
            // 检查emitter是否已经完成，避免向已完成的连接发送数据
            if (!unifiedSseService.isEmitterCompleted(emitter)) {
                // 发送完成信号
                emitter.send(SseEmitter.event().name("done").data("[DONE]").build());
                log.debug("完成信号已发送");
            } else {
                log.debug("SSE emitter已完成，跳过发送完成信号");
            }
        } catch (Exception e) {
            log.error("发送完成信号失败", e);
            completionException = e;
        }
        
        log.debug("完成信号发送完毕，准备关闭SSE连接");
        
        // 3. 关闭SSE连接：调用emitter.complete()方法完成连接
        try {
            // 检查emitter是否已经完成，避免重复关闭
            if (!unifiedSseService.isEmitterCompleted(emitter)) {
                unifiedSseService.completeEmitter(emitter, isCompleted);
                log.debug("SSE Emitter已关闭");
            } else {
                log.debug("SSE Emitter已完成，跳过关闭操作");
            }
        } catch (Exception e) {
            log.error("关闭Emitter时发生错误", e);
        }
        
        LogUtils.exitMethod("handleCompletion", "处理完成");
    }
    
    /**
     * 保存对话记录
     */
    public void saveDialogue(Agent agent, AgentRequest request, String userId, String responseContent) {
        LogUtils.enterMethod("saveDialogue", agent, request, userId);
        // 参数验证 - 内联验证逻辑，避免依赖ValidationUtils
        if (agent == null) {
            log.error("Agent对象不能为空");
            LogUtils.exitMethod("saveDialogue", "Agent对象不能为空");
            return;
        }
        
        if (request == null) {
            log.error("Agent请求不能为空");
            LogUtils.exitMethod("saveDialogue", "Agent请求不能为空");
            return;
        }
        
        if (userId == null || userId.trim().isEmpty()) {
            log.error("用户ID不能为空");
            LogUtils.exitMethod("saveDialogue", "用户ID不能为空");
            return;
        }
        
        try {
            // 创建对话记录
            AgentDialogue dialogue = AgentDialogue.builder()
                .agentId(request.getAgentId())
                .userMessage(request.getUserMessage())
                .agentResponse(responseContent)
                .userId(userId)
                .build();
            
            // 确保ID被设置
            if (dialogue.getId() == null || dialogue.getId().isEmpty()) {
                dialogue.setId(java.util.UUID.randomUUID().toString());
            }
            
            // 设置创建人和更新人信息
            // 在异步线程中获取用户ID
            String currentUserId = UserUtils.getCurrentUserIdInAsync();
            if (currentUserId == null) {
                currentUserId = userId; // 回退到传入的userId
            }
            dialogue.setCreatedBy(currentUserId);
            dialogue.setUpdatedBy(currentUserId);

            // 保存对话记录
            agentService.saveDialogue(dialogue);
            LogUtils.exitMethod("saveDialogue", "保存成功");
        } catch (Exception e) {
            log.error("保存对话记录失败", e);
            LogUtils.exitMethod("saveDialogue", e);
            throw new RuntimeException("保存对话记录失败", e);
        }
    }
}