package pangea.hiagent.agent.service;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import pangea.hiagent.agent.sse.UserSseService;
import pangea.hiagent.workpanel.event.EventService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;

/**
 * 统一错误处理服务类
 * 集中处理所有错误逻辑，减少重复代码
 */
@Slf4j
@Service
public class ErrorHandlerService {
    
    @Autowired
    private EventService eventService;
    
    @Autowired
    private ExceptionMonitoringService exceptionMonitoringService;
    
    @Autowired
    private UserSseService userSseService;
    
    /**
     * 生成错误跟踪ID
     * 
     * @return 错误跟踪ID
     */
    public String generateErrorId() {
        return java.util.UUID.randomUUID().toString().substring(0, 8);
    }
    
    /**
     * 构建完整的错误消息
     * 
     * @param errorMessage 基本错误信息
     * @param exception 异常对象
     * @param errorId 错误跟踪ID
     * @param processorType 处理器类型
     * @return 完整的错误消息
     */
    public String buildFullErrorMessage(String errorMessage, Exception exception, String errorId, String processorType) {
        StringBuilder fullErrorMessage = new StringBuilder();
        
        // 添加错误跟踪ID
        fullErrorMessage.append("[错误ID: ").append(errorId).append("] ");
        
        // 添加处理器类型（如果有）
        if (processorType != null && !processorType.isEmpty()) {
            fullErrorMessage.append("[").append(processorType).append("] ");
        }
        
        // 添加基本错误信息
        fullErrorMessage.append(errorMessage);
        
        // 添加异常信息（如果有）
        if (exception != null) {
            String exceptionMessage = exception.getMessage();
            if (exceptionMessage != null && !exceptionMessage.isEmpty()) {
                fullErrorMessage.append(": ").append(exceptionMessage);
            }
        }
        
        // 对于特定类型的错误，提供用户友好的提示
        if (exception != null && isUnauthorizedError(exception)) {
            fullErrorMessage.append(" 请检查您的API密钥配置。");
        } else if (exception != null && isTimeoutError(exception)) {
            fullErrorMessage.append(" 请稍后重试或联系技术支持。");
        }
        
        return fullErrorMessage.toString();
    }
    
    /**
     * 检查是否为未授权错误
     * 
     * @param exception 异常对象
     * @return 是否为未授权错误
     */
    public boolean isUnauthorizedError(Exception exception) {
        return exception != null && exception.getMessage() != null && 
               (exception.getMessage().contains("401") || 
                exception.getMessage().contains("Unauthorized") ||
                exception.getMessage().contains("API key"));
    }
    
    /**
     * 检查是否为超时错误
     * 
     * @param exception 异常对象
     * @return 是否为超时错误
     */
    public boolean isTimeoutError(Exception exception) {
        return exception != null && exception.getMessage() != null && 
               (exception.getMessage().contains("timeout") || 
                exception.getMessage().contains("timed out"));
    }
    
    /**
     * 处理聊天过程中的异常
     * 
     * @param emitter SSE发射器
     * @param errorMessage 错误信息
     * @param exception 异常对象
     * @param processorType 处理器类型（可选）
     */
    public void handleChatError(SseEmitter emitter, String errorMessage, Exception exception, String processorType) {
        // 参数验证
        if (errorMessage == null || errorMessage.isEmpty()) {
            errorMessage = "未知错误";
        }
            
        // 生成错误跟踪ID
        String errorId = generateErrorId();
            
        // 记录异常到监控服务
        if (exception != null) {
            exceptionMonitoringService.recordException(
                exception.getClass().getSimpleName(), 
                errorMessage, 
                java.util.Arrays.toString(exception.getStackTrace())
            );
        }
            
        log.error("[{}] {}处理聊天请求时发生错误: {}", 
                  errorId,
                  processorType != null ? processorType : "未知处理器", 
                  errorMessage, exception);
            
        try {
            // 检查emitter是否已经完成，避免向已完成的连接发送错误信息
            if (userSseService != null && !userSseService.isEmitterCompleted(emitter)) {
                String fullErrorMessage = buildFullErrorMessage(errorMessage, exception, errorId, processorType);
                eventService.sendErrorEvent(emitter, fullErrorMessage);
            } else {
                log.debug("[{}] SSE emitter已完成，跳过发送错误信息", errorId);
            }
        } catch (Exception sendErrorEx) {
            log.error("[{}] 发送错误信息失败", errorId, sendErrorEx);
        }
    }
    
    /**
     * 处理聊天过程中的异常（）
     * 
     * @param emitter SSE发射器
     * @param errorMessage 错误信息
     */
    public void handleChatError(SseEmitter emitter, String errorMessage) {
        // 参数验证
        if (errorMessage == null || errorMessage.isEmpty()) {
            errorMessage = "未知错误";
        }
        
        // 生成错误跟踪ID
        String errorId = generateErrorId();
        
        log.error("[{}] 处理聊天请求时发生错误: {}", errorId, errorMessage);
        
        try {
            // 检查emitter是否已经完成，避免向已完成的连接发送错误信息
            if (userSseService != null && !userSseService.isEmitterCompleted(emitter)) {
                String fullErrorMessage = buildFullErrorMessage(errorMessage, null, errorId, null);
                eventService.sendErrorEvent(emitter, fullErrorMessage);
            } else {
                log.debug("[{}] SSE emitter已完成，跳过发送错误信息", errorId);
            }
        } catch (Exception sendErrorEx) {
            log.error("[{}] 发送错误信息失败", errorId, sendErrorEx);
        }
    }
    
    /**
     * 处理Token处理过程中的异常
     * 
     * @param emitter SSE发射器
     * @param processorType 处理器类型
     * @param exception 异常对象
     * @param isCompleted 完成状态标记
     */
    public void handleTokenError(SseEmitter emitter, String processorType, Exception exception, AtomicBoolean isCompleted) {
        // 参数验证
        if (processorType == null || processorType.isEmpty()) {
            processorType = "未知处理器";
        }
        
        // 生成错误跟踪ID
        String errorId = generateErrorId();
        
        // 记录异常到监控服务
        if (exception != null) {
            exceptionMonitoringService.recordException(
                exception.getClass().getSimpleName(), 
                "处理token时发生错误", 
                java.util.Arrays.toString(exception.getStackTrace())
            );
        }
        
        log.error("[{}] {}处理token时发生错误", errorId, processorType, exception);
        if (!isCompleted.getAndSet(true)) {
            try {
                // 检查emitter是否已经完成，避免向已完成的连接发送错误信息
                if (userSseService != null && !userSseService.isEmitterCompleted(emitter)) {
                    String errorMessage = "处理响应时发生错误";
                    String fullErrorMessage = buildFullErrorMessage(errorMessage, exception, errorId, processorType);
                    eventService.sendErrorEvent(emitter, fullErrorMessage);
                } else {
                    log.debug("[{}] SSE emitter已完成，跳过发送错误信息", errorId);
                }
            } catch (Exception ignored) {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] 无法发送错误信息", errorId);
                }
            }
        }
    }
    
    /**
     * 处理完成回调过程中的异常
     * 
     * @param emitter SSE发射器
     * @param exception 异常对象
     */
    public void handleCompletionError(SseEmitter emitter, Exception exception) {
        // 生成错误跟踪ID
        String errorId = generateErrorId();
        
        // 记录异常到监控服务
        if (exception != null) {
            exceptionMonitoringService.recordException(
                exception.getClass().getSimpleName(), 
                "发送完成事件失败", 
                java.util.Arrays.toString(exception.getStackTrace())
            );
        }
        
        log.error("[{}] 发送完成事件失败", errorId, exception);
        
        try {
            // 检查emitter是否已经完成，避免向已完成的连接发送错误信息
            if (userSseService != null && !userSseService.isEmitterCompleted(emitter)) {
                String errorMessage = "发送完成事件失败，请联系技术支持";
                String fullErrorMessage = buildFullErrorMessage(errorMessage, exception, errorId, "完成回调");
                eventService.sendErrorEvent(emitter, fullErrorMessage);
            } else {
                log.debug("[{}] SSE emitter已完成，跳过发送错误信息", errorId);
            }
        } catch (Exception sendErrorEx) {
            log.error("[{}] 发送错误信息失败", errorId, sendErrorEx);
        }
    }
    
    /**
     * 处理流式处理中的错误
     *
     * @param e 异常对象
     * @param tokenConsumer token处理回调函数
     * @param errorMessagePrefix 错误消息前缀
     */
    public void handleStreamError(Throwable e, Consumer<String> tokenConsumer, String errorMessagePrefix) {
        String errorMessage = errorMessagePrefix + ": " + e.getMessage();
        
        // 记录异常到监控服务
        exceptionMonitoringService.recordException(
            e.getClass().getSimpleName(), 
            errorMessage, 
            java.util.Arrays.toString(e.getStackTrace())
        );
        
        log.error("流式处理错误: {}", errorMessage, e);
        if (tokenConsumer != null) {
            tokenConsumer.accept("[ERROR] " + errorMessage);
        }
    }
    
    /**
     * 发送错误信息给客户端
     *
     * @param tokenConsumer token处理回调函数
     * @param errorMessage 错误消息
     */
    public void sendErrorMessage(Consumer<String> tokenConsumer, String errorMessage) {
        if (tokenConsumer != null) {
            tokenConsumer.accept("[ERROR] " + errorMessage);
        }
    }
    
    /**
     * 处理React流程错误
     *
     * @param e 异常对象
     * @param tokenConsumer token处理回调函数
     */
    public void handleReactFlowError(Exception e, Consumer<String> tokenConsumer) {
        String errorMessage = "处理ReAct流程时发生错误: " + e.getMessage();
        
        // 记录异常到监控服务
        exceptionMonitoringService.recordException(
            e.getClass().getSimpleName(), 
            errorMessage, 
            java.util.Arrays.toString(e.getStackTrace())
        );
        
        log.error("ReAct流程错误: {}", errorMessage, e);
        if (tokenConsumer != null) {
            tokenConsumer.accept("[ERROR] " + errorMessage);
        }
    }
    
    /**
     * 处理同步处理中的错误
     *
     * @param e 异常对象
     * @param errorMessagePrefix 错误消息前缀
     * @return 错误消息
     */
    public String handleSyncError(Exception e, String errorMessagePrefix) {
        // 检查是否是401 Unauthorized错误
        if (isUnauthorizedError(e)) {
            log.error("LLM返回401未授权错误: {}", e.getMessage());
            return "请配置API密钥";
        } else {
            String errorMessage = e.getMessage();
            if (errorMessage == null || errorMessage.isEmpty()) {
                errorMessage = "未知错误";
            }
            return errorMessagePrefix + ": " + errorMessage;
        }
    }
    
    /**
     * 处理保存对话记录过程中的异常
     * 
     * @param emitter SSE发射器
     * @param exception 异常对象
     * @param isCompleted 完成状态标记
     */
    public void handleSaveDialogueError(SseEmitter emitter, Exception exception, AtomicBoolean isCompleted) {
        // 生成错误跟踪ID
        String errorId = generateErrorId();
        
        // 记录异常到监控服务
        if (exception != null) {
            exceptionMonitoringService.recordException(
                exception.getClass().getSimpleName(), 
                "保存对话记录失败", 
                java.util.Arrays.toString(exception.getStackTrace())
            );
        }
        
        log.error("[{}] 保存对话记录失败", errorId, exception);
        
        if (!isCompleted.getAndSet(true)) {
            try {
                // 检查emitter是否已经完成，避免向已完成的连接发送错误信息
                if (userSseService != null && !userSseService.isEmitterCompleted(emitter)) {
                    String errorMessage = "保存对话记录失败，请联系技术支持";
                    String fullErrorMessage = buildFullErrorMessage(errorMessage, exception, errorId, "对话记录");
                    eventService.sendErrorEvent(emitter, fullErrorMessage);
                } else {
                    log.debug("[{}] SSE emitter已完成，跳过发送错误信息", errorId);
                }
            } catch (Exception sendErrorEx) {
                log.error("[{}] 发送错误信息失败", errorId, sendErrorEx);
            }
        }
    }
}