package pangea.hiagent.agent.service;

import lombok.extern.slf4j.Slf4j;

import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import pangea.hiagent.agent.processor.AgentProcessor;
import pangea.hiagent.agent.processor.AgentProcessorFactory;
import pangea.hiagent.agent.sse.UserSseService;
import pangea.hiagent.common.utils.UserUtils;
import pangea.hiagent.web.dto.ChatRequest;
import pangea.hiagent.model.Agent;
import pangea.hiagent.tool.AgentToolManager;
import pangea.hiagent.web.dto.AgentRequest;
import pangea.hiagent.workpanel.event.EventService;
import pangea.hiagent.common.utils.AsyncUserContextDecorator;
import jakarta.servlet.http.HttpServletResponse;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * Agent 对话服务
 * 职责：协调整个AI对话流程，作为流式处理的统一入口和协调者
 */
@Slf4j
@Service
public class AgentChatService {

    private final ChatErrorHandler chatErrorHandler;
    private final AgentValidationService agentValidationService;
    private final AgentProcessorFactory agentProcessorFactory;
    private final StreamRequestService streamRequestService;
    private final AgentToolManager agentToolManager;
    private final UserSseService workPanelSseService;
    private final pangea.hiagent.web.service.AgentService agentService;
    
    public AgentChatService(
            EventService eventService,
            ChatErrorHandler chatErrorHandler,
            AgentValidationService agentValidationService,
            AgentProcessorFactory agentProcessorFactory,
            StreamRequestService streamRequestService,
            AgentToolManager agentToolManager,
            UserSseService workPanelSseService,
            pangea.hiagent.web.service.AgentService agentService) {
        this.chatErrorHandler = chatErrorHandler;
        this.agentValidationService = agentValidationService;
        this.agentProcessorFactory = agentProcessorFactory;
        this.streamRequestService = streamRequestService;
        this.agentToolManager = agentToolManager;
        this.workPanelSseService = workPanelSseService;
        this.agentService = agentService;
    }
    
    // 专用线程池配置 - 使用静态变量确保线程池在整个应用中是单例的
    private static final ExecutorService executorService = new ThreadPoolExecutor(
            20,
            200,
            60L,
            TimeUnit.SECONDS,
            new LinkedBlockingQueue<>(1000),
            new ThreadPoolExecutor.CallerRunsPolicy()
    );

    // /**
    //  * 处理同步对话请求的统一入口
    //  * @param agent Agent对象
    //  * @param request 请求对象
    //  * @param userId 用户ID
    //  * @return 处理结果
    //  */
    // public String handleChatSync(Agent agent, AgentRequest request, String userId) {
    //     log.info("开始处理同步对话请求，AgentId: {}, 用户消息: {}", agent.getId(), request.getUserMessage());
    //     
    //     try {
    //         // 获取处理器
    //         AgentProcessor processor = agentProcessorFactory.getProcessor(agent);
    //         if (processor == null) {
    //             log.error("无法获取Agent处理器");
    //             return "[错误] 无法获取Agent处理器";
    //         }
    //        
    //         // 处理请求
    //         return processor.processRequest(agent, request, userId);
    //     } catch (Exception e) {
    //         log.error("处理普通Agent请求时发生错误", e);
    //         return "[错误] 处理请求时发生错误: " + e.getMessage();
    //     }
    // }

    /**
     * 处理流式对话请求的统一入口
     * 
     * @param agentId Agent ID
     * @param chatRequest 对话请求
     * @param response HTTP响应
     * @return SSE emitter
     */
    public SseEmitter handleChatStream(String agentId, ChatRequest chatRequest, HttpServletResponse response) {
        log.info("开始处理流式对话请求，AgentId: {}, 用户消息: {}", agentId, chatRequest.getMessage());

        // 尝试获取当前用户ID，优先从SecurityContext获取，其次从请求中解析JWT
        String userId = UserUtils.getCurrentUserId();
        
        // 如果在主线程中未能获取到用户ID，尝试在异步环境中获取
        if (userId == null) {
            userId = UserUtils.getCurrentUserIdInAsync();
        }
        
        if (userId == null) {
            log.error("用户未认证");
            SseEmitter emitter = workPanelSseService.createEmitter();
            // 检查响应是否已经提交
            if (!response.isCommitted()) {
                chatErrorHandler.handleChatError(emitter, "用户未认证，请重新登录");
            } else {
                log.warn("响应已提交，无法发送用户未认证错误信息");
                // 检查emitter是否已经完成，避免重复关闭
                if (!workPanelSseService.isEmitterCompleted(emitter)) {
                    emitter.complete();
                }
            }
            return emitter;
        }

        // 验证Agent是否存在
        Agent agent = agentService.getAgent(agentId);
        if (agent == null) {
            log.warn("Agent不存在: {}", agentId);
            SseEmitter emitter = workPanelSseService.createEmitter();
            // 检查响应是否已经提交
            if (!response.isCommitted()) {
                chatErrorHandler.handleChatError(emitter, "Agent不存在");
            } else {
                log.warn("响应已提交，无法发送Agent不存在错误信息");
                // 检查emitter是否已经完成，避免重复关闭
                if (!workPanelSseService.isEmitterCompleted(emitter)) {
                    emitter.complete();
                }
            }
            return emitter;
        }

        // 创建 SSE emitter
        SseEmitter emitter = workPanelSseService.createEmitter();
        
        // 将userId和agent设为final以在Lambda表达式中使用
        final String finalUserId = userId;
        final Agent finalAgent = agent;

        // 异步处理对话，避免阻塞HTTP连接
        // 使用用户上下文装饰器来确保在异步线程中也能获取到用户信息
        executorService.execute(AsyncUserContextDecorator.wrapWithContext(() -> {
            try {
                processChatRequest(emitter, finalAgent, chatRequest, finalUserId);
            } catch (Exception e) {
                log.error("处理聊天请求时发生异常", e);
                try {
                    // 检查响应是否已经提交
                    if (emitter != null && !workPanelSseService.isEmitterCompleted(emitter)) {
                        chatErrorHandler.handleChatError(emitter, "处理请求时发生错误", e, null);
                    } else {
                        log.warn("响应已提交或emitter已完成，无法发送处理请求错误信息");
                    }
                } catch (Exception handlerException) {
                    log.error("处理错误信息时发生异常", handlerException);
                }
            }
        }));

        return emitter;
    }

    /**
     * 处理聊天请求的核心逻辑
     * 注意：权限验证已在主线程中完成，此正仅执行业务逻辑不进行权限检查
     *
     * @param emitter SSE发射器
     * @param agent Agent对象
     * @param chatRequest 聊天请求
     * @param userId 用户ID
     */
    private void processChatRequest(SseEmitter emitter, Agent agent, ChatRequest chatRequest, String userId) {
        try {
            // 获取处理器
            AgentProcessor processor = agentProcessorFactory.getProcessor(agent);
            if (processor == null) {
                log.error("无法获取Agent处理器，Agent: {}", agent.getId());
                chatErrorHandler.handleChatError(emitter, "无法获取Agent处理器");
                return;
            }
            
            // 转换请求对象
            AgentRequest request = chatRequest.toAgentRequest(agent.getId(), agent, agentToolManager);

            // 处理流式请求
            streamRequestService.handleStreamRequest(emitter, processor, request, agent, userId);
        } catch (Exception e) {
            log.error("处理聊天请求时发生异常", e);
            chatErrorHandler.handleChatError(emitter, "处理请求时发生错误", e, null);
        }
    }
}