package pangea.hiagent.agent.react;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.client.ChatClient;
import org.springframework.ai.chat.messages.*;
import org.springframework.ai.chat.model.ChatResponse;
import org.springframework.ai.chat.prompt.Prompt;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import pangea.hiagent.agent.service.ErrorHandlerService;
import pangea.hiagent.agent.service.TokenConsumerWithCompletion;
import pangea.hiagent.memory.MemoryService;
import pangea.hiagent.model.Agent;
import pangea.hiagent.tool.AgentToolManager;
import pangea.hiagent.tool.impl.DateTimeTools;
import pangea.hiagent.common.utils.UserUtils;
import java.util.List;
import java.util.ArrayList;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

/**
 * 简化的默认ReAct执行器实现
 */
@Slf4j
@Service
public class DefaultReactExecutor implements ReactExecutor {
    
    private static final String DEFAULT_SYSTEM_PROMPT = 
        "你是一个基于ReAct（推理+行动）框架构建的高性能专业AI助手，原生集成Spring AI生态体系。你的核心任务是通过严谨的逻辑推理解决复杂、精准的用户查询，严格遵循「工具串行调用」和「迭代循环思考」两大核心准则。\n\n" +
        "你必须严格遵守的核心执行原则：\n" +
        "1. 将复杂问题拆解为具备逻辑依赖关系的、清晰分层的子问题\n" +
        "2. 智能选择专业工具并进行**严格串行组合调用** —— 仅当上一个工具完全执行并返回有效结果后，才能启动下一个工具的调用流程，禁止并行/无序调用\n" +
        "3. 深度处理工具执行结果，循环运行「思考→行动→观察」的ReAct闭环，持续迭代修正答案，直至得出精准、满意的结果\n" +
        "4. 严格循环次数限制：ReAct迭代循环**绝对不能超过10轮**，避免无限执行；达到次数上限时，立即终止循环并整合最终答案\n\n" +
        "=== 工具协同策略 & 严格串行调用规则 ===\n" +
        "你可调用全套Spring AI专业工具，优先级绝对遵循「串行工具链」，所有工具组合必须符合核心规则：**上一个工具的输出 = 下一个工具的输入**。你需要通过有序、协同的工具调用创造价值，具体规则如下：\n" +
        "- 针对有逻辑依赖的任务，按串行顺序串联工具：数据准备 → 数据处理 → 计算分析 → 结果存储/输出\n" +
        "- 先使用文本处理工具清洗、格式化原始数据，再将处理后的数据输入分析/计算类工具\n" +
        "- 在串行工作流中嵌入计算工具，为子问题执行针对性的精准运算\n" +
        "- 串行组合多个工具，逐层丰富数据维度，挖掘深度业务洞察\n" +
        "- 串行搭配「数据提取工具+分析工具」，得到完整且经过验证的结果\n" +
        "- 若某工具返回空值/无效结果，暂停当前工具链，为该子问题重新选择备选工具，执行成功后再继续串行调用\n\n" +
        "经典串行工具协同场景（优先级参考）：\n" +
        "1. 读取文件（Spring AI 文件工具）→ 文本内容解析 → 关键信息提取 → 保存处理结果（全串行、分步执行）\n" +
        "2. 获取当前日期/时间 → 日期格式化工具 → 时区转换 → 格式化展示输出（串行依赖关系）\n" +
        "3. 网页内容提取 → 文本清洗与切分 → 统计计算 → 可视化图表生成（串行数据流）\n" +
        "4. 数值运算 → 单位转换 → 数据格式标准化 → 结果持久化存储（串行完整流程）\n" +
        "5. 问题拆解 → 关键词提取 → 知识库检索 → 结果验证 → 最终答案整合（串行推理链）\n\n" +
        "=== 可迭代的ReAct循环思考流程（核心规则，≤ 10轮） ===\n" +
        "此流程为**循环迭代模式**，非一次性线性步骤。重复执行「思考→行动→观察」闭环，直至得出满意答案、精准结果，或达到10轮上限，每一次循环都需清晰标注。\n" +
        "每一轮ReAct循环，均严格遵循以下3个核心步骤：\n\n" +
        "步骤1 - 思考 (迭代轮次 X/10)：全面的问题与结果分析\n" +
        "  - 分析用户的核心诉求，拆解为具备逻辑顺序的分层子问题\n" +
        "  - 针对本轮迭代：评估最新的工具返回结果（如有），校验结果的完整性与准确性\n" +
        "  - 识别未完成的子任务、或需要修正的错误推理点\n" +
        "  - 为当前子任务选择**唯一最合适**的工具（串行规则：每一次行动步骤仅调用一个工具）\n" +
        "  - 确认所选工具的输入来源：要么是用户原始查询，要么是上一次调用工具的输出结果\n" +
        "  - 仅规划下一步动作，不做多步预规划，适配迭代修正的需求\n\n" +
        "步骤2 - 行动 (迭代轮次 X/10)：单工具串行调用\n" +
        "  - 本步骤**仅调用一个专业工具**（严格串行原则，禁止并行调用多个工具）\n" +
        "  - 所有工具均由Spring AI框架自动执行，你只需专注于选择正确的工具、定义合法的入参即可\n" +
        "  - 若上一个工具返回无效/空结果：为当前子任务切换备选工具并重新调用\n" +
        "  - 若工具调用失败：记录失败信息，基于已有有效数据继续后续迭代\n\n" +
        "步骤3 - 观察 (迭代轮次 X/10)：深度结果解读与有效性验证\n" +
        "  - 解析Spring AI工具执行后返回的完整原始结果\n" +
        "  - 对照当前子任务的目标，验证结果的准确性、有效性与完整性\n" +
        "  - 从工具结果中提取核心数据、规律特征与可落地的洞察\n" +
        "  - 判定：当前结果是否足够解决该子任务？是 → 在下一轮循环中推进下一个子任务；否 → 在下一轮思考步骤中重新选择工具\n" +
        "  - 判定：整体问题是否已解决？是 → 退出ReAct循环并整合最终答案；否 → 启动下一轮ReAct迭代（X+1）\n\n" +
        "循环终止核心规则（满足任一即终止）：\n" +
        "  ✔ 当用户的查询被完整、精准解答时，终止循环并整合答案\n" +
        "  ✔ 当ReAct迭代轮次达到10轮（上限），立即终止循环\n" +
        "  ✔ 当剩余子任务无可用有效工具时，终止循环\n\n" +
        "=== 标准化回复格式（强制要求，不可偏离） ===\n" +
        "你必须遵循以下固定结构输出所有回复，标注清晰、逻辑连贯；思考、行动和观察可以迭代出现多次，但最终答案只在最后一轮迭代中最后出现：\n\n" +
        "1. 思考：阐述你的问题拆解思路、迭代推理逻辑，以及串行工具选择策略\n" +
        "   - 拆解后的具备逻辑依赖的分层子问题\n" +
        "   - 为各子任务选定的工具，以及严格的串行调用顺序\n" +
        "   - 当前的ReAct迭代轮次（X/10）与本轮核心目标\n" +
        "   - 说明上一个工具的输出如何作为下一个工具的输入\n\n" +
        "2. 行动：描述本轮迭代的**单次串行工具调用**（仅限一个工具）\n" +
        "   - 工具调用: [当前轮次 X] → [工具全称] → 核心目的: [调用该工具的明确原因、输入来源、预期输出]\n\n" +
        "3. 观察：解读来自Spring AI的真实工具执行结果（严禁编造数据）\n" +
        "   - 被调用工具返回的真实有效结果\n" +
        "   - 针对当前子任务，验证结果的准确性与完整性\n" +
        "   - 从结果中提炼的核心洞察与规律\n" +
        "   - 决策结果：继续迭代 / 终止循环（需说明明确原因）\n\n" +
        "4. 最终答案：将所有迭代结果整合为连贯、精准、具备落地性的答案\n" +
        "   - 用通顺的自然语言直接回答用户的核心问题\n" +
        "   - 重点提炼通过「串行工具协同」和「ReAct迭代推理」得到的核心洞察\n" +
        "   - 明确标注本次解答所用的ReAct迭代轮次（X/10）\n" +
        "   - 提供可落地的后续建议或补充洞察（如有必要）\n" +
        "   - 保持专业的对话语气，面向终端用户时规避技术化的工具术语\n\n" +
        "=== 不可违背的核心硬性规则（全部遵守） ===\n" +
        "- 串行优先：所有工具均严格串行调用，绝不并行；下一个工具的启动，必须等待上一个工具执行完成\n" +
        "- 迭代思考：ReAct是循环流程而非一次性步骤，通过迭代修正答案、纠正错误\n" +
        "- 次数限制：ReAct迭代最多10轮，无例外\n" +
        "- 拒绝编造：始终使用Spring AI返回的真实工具执行结果，绝不虚构数据，也绝不编造结果，也绝不在工具执行结果基础上编造\n" +
        "- Spring AI原生适配：工具由框架自动执行，你只需专注工具选择与入参定义\n" +
        "- 策略透明：在「思考」环节清晰说明你的串行工具选择与迭代推理逻辑\n" +
        "- 容错处理：工具不可用/调用失败时，说明所需的备选能力，并基于已有有效数据继续推进\n" +
        "- 结果校验：对无效/空的工具结果进行驳回，并为当前子任务重新选择工具\n" +
        "- 可追溯：所有ReAct迭代均清晰标注轮次，实现完整的推理链路可追溯";
    
    private final List<ReactCallback> reactCallbacks = new ArrayList<>();
    private final AtomicInteger stepCounter = new AtomicInteger(0);
    
    @Autowired
    private DateTimeTools dateTimeTools;
    
    @Autowired
    private MemoryService memoryService;
    
    @Autowired
    private ErrorHandlerService errorHandlerService;

    @Autowired
    private TokenTextSegmenter tokenTextSegmenter;
    
    private final AgentToolManager agentToolManager;
    
    public DefaultReactExecutor(AgentToolManager agentToolManager) {
        this.agentToolManager = agentToolManager;
    }
    
    @Override
    public void addReactCallback(ReactCallback callback) {
        if (callback != null) {
            reactCallbacks.add(callback);
        }
    }
    
    @Override
    public String execute(ChatClient chatClient, String userInput, List<Object> tools, Agent agent) {
        // 调用带用户ID的方法，首先尝试获取当前用户ID
        String userId = UserUtils.getCurrentUserId();
        return execute(chatClient, userInput, tools, agent, userId);
    }
    
    @Override
    public String execute(ChatClient chatClient, String userInput, List<Object> tools, Agent agent, String userId) {
        log.info("开始执行ReAct流程，用户输入: {}", userInput);
        
        stepCounter.set(0);
        
        List<Object> agentTools = getAgentTools(agent);
        
        try {
            // triggerThinkStep("开始处理用户请求: " + userInput);
            
            Prompt prompt = buildPromptWithHistory(DEFAULT_SYSTEM_PROMPT, userInput, agent, userId);
            
            ChatResponse response = chatClient.prompt(prompt)
                    .tools(agentTools.toArray())
                    .call()
                    .chatResponse();
            
            String responseText = response.getResult().getOutput().getText();
            
            // triggerObservationStep(responseText);
            
            log.info("最终答案: {}", responseText);
            
            // triggerFinalAnswerStep(responseText);
            
            // 保存助手回复到内存，使用提供的用户ID
            saveAssistantResponseToMemory(agent, responseText, userId);
            
            return responseText;
        } catch (Exception e) {
            log.error("执行ReAct流程时发生错误", e);
            return handleReActError(e);
        }
    }
    
    /**
     * 处理ReAct执行过程中发生的错误
     * 
     * @param e 发生的异常
     * @return 错误处理结果
     */
    private String handleReActError(Exception e) {
        return errorHandlerService.handleSyncError(e, "处理ReAct请求时发生错误");
    }
    
    /**
     * 构建带有历史记录的提示词
     * 
     * @param systemPrompt 系统提示词
     * @param userInput 用户输入
     * @param agent 智能体对象
     * @return 构建好的提示词对象
     */
    private Prompt buildPromptWithHistory(String systemPrompt, String userInput, Agent agent) {
        return buildPromptWithHistory(systemPrompt, userInput, agent, null);
    }
    
    /**
     * 构建带有历史记录的提示词
     * 
     * @param systemPrompt 系统提示词
     * @param userInput 用户输入
     * @param agent 智能体对象
     * @param userId 用户ID（可选，如果为null则自动获取）
     * @return 构建好的提示词对象
     */
    private Prompt buildPromptWithHistory(String systemPrompt, String userInput, Agent agent, String userId) {
        List<org.springframework.ai.chat.messages.Message> messages = new ArrayList<>();
        
        messages.add(new SystemMessage(systemPrompt));
        
        if (agent != null) {
            try {
                // 如果没有提供用户ID，则尝试获取当前用户ID
                if (userId == null) {
                    userId = UserUtils.getCurrentUserId();
                }
                String sessionId = memoryService.generateSessionId(agent, userId);
                
                int historyLength = agent.getHistoryLength() != null ? agent.getHistoryLength() : 10;
                
                List<org.springframework.ai.chat.messages.Message> historyMessages = 
                    memoryService.getHistoryMessages(sessionId, historyLength);
                
                messages.addAll(historyMessages);
                
                memoryService.addUserMessageToMemory(sessionId, userInput);
            } catch (Exception e) {
                log.warn("获取历史对话记录时发生错误: {}", e.getMessage());
            }
        }
        
        messages.add(new UserMessage(userInput));
        
        return new Prompt(messages);
    }
    
    @Override
    public void executeStream(ChatClient chatClient, String userInput, List<Object> tools, Consumer<String> tokenConsumer, Agent agent) {
        // 调用带用户ID的方法，但首先尝试获取当前用户ID
        String userId = UserUtils.getCurrentUserId();
        executeStream(chatClient, userInput, tools, tokenConsumer, agent, userId);
    }
    
    @Override
    public void executeStream(ChatClient chatClient, String userInput, List<Object> tools, Consumer<String> tokenConsumer, Agent agent, String userId) {
        log.info("使用stream()方法处理ReAct流程，支持真正的流式输出");
        
        stepCounter.set(0);
        
        List<Object> agentTools = getAgentTools(agent);
        
        StringBuilder fullResponse = new StringBuilder();
        
        try {
            // triggerThinkStep("开始处理用户请求: " + userInput);
            
            Prompt prompt = buildPromptWithHistory(DEFAULT_SYSTEM_PROMPT, userInput, agent, userId);
            
            chatClient.prompt(prompt)
                    .tools(agentTools.toArray())
                    .stream()
                    .chatResponse()
                    .subscribe(
                        chatResponse -> handleTokenResponse(chatResponse, tokenConsumer, fullResponse),
                        throwable -> handleStreamError(throwable, tokenConsumer),
                        () -> handleStreamCompletion(tokenConsumer, fullResponse, agent, userId)
                    );
            
        } catch (Exception e) { 
            log.error("流式执行ReAct流程时发生错误", e);
            errorHandlerService.handleReactFlowError(e, tokenConsumer);
        }
    }
    
    /**
     * 处理流式响应中的单个token
     * 
     * @param chatResponse 聊天响应对象
     * @param tokenConsumer token消费者
     * @param fullResponse 完整响应构建器
     */
    private void handleTokenResponse(org.springframework.ai.chat.model.ChatResponse chatResponse, Consumer<String> tokenConsumer, StringBuilder fullResponse) {
        try {
            String token = chatResponse.getResult().getOutput().getText();
            
            if (isValidToken(token)) {
                fullResponse.append(token);
                
                // analyzeAndRecordToolEvents(token, fullResponse.toString());
                
                if (tokenConsumer != null) {
                    tokenConsumer.accept(token);
                }

                // tokenTextSegmenter.inputChar(token);
                // tokenTextSegmenter.finishInput();
                
                // 改进：在流式处理过程中实时解析关键词
                // processTokenForStepsWithFullResponse(token, fullResponse.toString());
            }
        } catch (Exception e) {
            log.error("处理token时发生错误", e);
        }
    }
    
    /**
     * 处理流式响应完成事件
     * 
     * @param tokenConsumer token消费者
     * @param fullResponse 完整响应内容
     * @param agent 智能体对象
     */
    private void handleStreamCompletion(Consumer<String> tokenConsumer, StringBuilder fullResponse, Agent agent) {
        // 调用带用户ID的版本，使用当前线程的用户ID
        String userId = UserUtils.getCurrentUserId();
        handleStreamCompletion(tokenConsumer, fullResponse, agent, userId);
    }
    
    /**
     * 处理流式响应完成事件
     * 
     * @param tokenConsumer token消费者
     * @param fullResponse 完整响应内容
     * @param agent 智能体对象
     * @param userId 用户ID
     */
    private void handleStreamCompletion(Consumer<String> tokenConsumer, StringBuilder fullResponse, Agent agent, String userId) {
        try {
            log.info("流式处理完成");
            
            // 检查是否已经处理了Final Answer，如果没有，则将整个响应作为最终答案
            String responseStr = fullResponse.toString();
            if (!hasFinalAnswerBeenTriggered(responseStr)) {
                // triggerFinalAnswerStep(responseStr);
            }
            
            saveAssistantResponseToMemory(agent, responseStr, userId);
            
            sendCompletionEvent(tokenConsumer, responseStr);
        } catch (Exception e) {
            log.error("处理流式完成回调时发生错误", e);
            handleCompletionError(tokenConsumer, e);
        }
    }
    
    /**
     * 检查是否已经触发了Final Answer步骤
     * 
     * @param fullResponse 完整响应内容
     * @return 如果已经触发了Final Answer则返回true，否则返回false
     */
    private boolean hasFinalAnswerBeenTriggered(String fullResponse) {
        String[] finalAnswerPatterns = {"Final Answer:", "final answer:", "FINAL ANSWER:", "Final_Answer:", "final_answer:", "FINAL_ANSWER:", "最终答案:"};
        for (String pattern : finalAnswerPatterns) {
            if (fullResponse.toLowerCase().contains(pattern.toLowerCase())) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * 将助手的回复保存到内存中
     * 
     * @param agent 智能体对象
     * @param response 助手的回复内容
     * @param userId 用户ID
     */
    private void saveAssistantResponseToMemory(Agent agent, String response, String userId) {
        if (agent != null) {
            try {
                String sessionId = memoryService.generateSessionId(agent, userId);
                memoryService.addAssistantMessageToMemory(sessionId, response);
            } catch (Exception e) {
                log.warn("保存助理回复到内存时发生错误: {}", e.getMessage());
            }
        }
    }
    
    /**
     * 处理完成事件时发生的错误
     * 
     * @param tokenConsumer token消费者
     * @param e 发生的异常
     */
    private void handleCompletionError(Consumer<String> tokenConsumer, Exception e) {
        if (tokenConsumer instanceof TokenConsumerWithCompletion) {
            try {
                String errorId = errorHandlerService.generateErrorId();
                String fullErrorMessage = errorHandlerService.buildFullErrorMessage("处理完成时发生错误", e, errorId, "ReAct");
                try {
                    ((TokenConsumerWithCompletion) tokenConsumer).onComplete("[" + errorId + "] " + fullErrorMessage);
                } catch (NoClassDefFoundError ex) {
                    log.error("TokenConsumerWithCompletion依赖类未找到，跳过完成回调: {}", ex.getMessage());
                }
            } catch (Exception ex) {
                log.error("调用onComplete时发生错误", ex);
            }
        }
    }
    
    /**
     * 验证token是否有效
     * 
     * @param token 待验证的token
     * @return 如果token有效则返回true，否则返回false
     */
    private boolean isValidToken(String token) {
        return token != null && !token.isEmpty();
    }
    
    /**
     * 基于完整响应进行关键词解析
     * 
     * @param token 当前token
     * @param fullResponse 完整响应内容
     */
    private void processTokenForStepsWithFullResponse(String token, String fullResponse) {
        if (fullResponse == null || fullResponse.isEmpty()) {
            return;
        }
        
        // 检查并处理Thought部分
        processThoughtSection(fullResponse);
        
        // 检查并处理Action部分
        processActionSection(fullResponse);
        
        // 检查并处理Observation部分
        processObservationSection(fullResponse);
        
        // 检查并处理Final Answer部分
        processFinalAnswerSection(fullResponse);
    }
    
    /**
     * 处理Thought部分
     * 
     * @param fullResponse 完整响应内容
     */
    private void processThoughtSection(String fullResponse) {
        // 检查是否包含Thought标记
        String[] thoughtPatterns = {"Thought:", "thought:", "THOUGHT:"};
        for (String pattern : thoughtPatterns) {
            if (fullResponse.toLowerCase().contains(pattern.toLowerCase())) {
                // 检查是否已经有对应的步骤被触发，避免重复触发
                if (!isStepAlreadyTriggered(ReactStepType.THOUGHT, fullResponse, pattern)) {
                    String content = extractContentAfterKeyword(fullResponse, pattern);
                    // 如果内容中包含下一个步骤的关键词，则只提取到下一个关键词之前的内容
                    String[] nextKeywords = {"Action:", "Observation:", "Final Answer:"};
                    for (String nextKeyword : nextKeywords) {
                        int nextKeywordIndex = content.toLowerCase().indexOf(nextKeyword.toLowerCase());
                        if (nextKeywordIndex != -1) {
                            content = content.substring(0, nextKeywordIndex).trim();
                            break;
                        }
                    }
                    if (!content.isEmpty()) {
                        triggerThinkStep(content);
                    }
                }
                break; // 找到一个就跳出
            }
        }
    }
    
    /**
     * 处理Action部分
     * 
     * @param fullResponse 完整响应内容
     */
    private void processActionSection(String fullResponse) {
        // 检查是否包含Action标记
        String[] actionPatterns = {"Action:", "action:", "ACTION:"};
        for (String pattern : actionPatterns) {
            if (fullResponse.toLowerCase().contains(pattern.toLowerCase())) {
                if (!isStepAlreadyTriggered(ReactStepType.ACTION, fullResponse, pattern)) {
                    String content = extractContentAfterKeyword(fullResponse, pattern);
                    String[] nextKeywords = {"Observation:", "Thought:", "Final Answer:"};
                    for (String nextKeyword : nextKeywords) {
                        int nextKeywordIndex = content.toLowerCase().indexOf(nextKeyword.toLowerCase());
                        if (nextKeywordIndex != -1) {
                            content = content.substring(0, nextKeywordIndex).trim();
                            break;
                        }
                    }
                    // 尝试提取工具名称和参数
                    String toolName = extractToolName(fullResponse); // 从整个响应中提取，因为可能在Action后有详细信息
                    Object toolArgs = extractToolArgs(fullResponse);
                    
                    if (toolName != null && !toolName.isEmpty()) {
                        triggerActionStep(toolName, "tool_call", toolArgs);
                    } else {
                        triggerActionStep("unknown", "tool_call", toolArgs);
                    }
                }
                break;
            }
        }
    }
    
    /**
     * 处理Observation部分
     * 
     * @param fullResponse 完整响应内容
     */
    private void processObservationSection(String fullResponse) {
        // 检查是否包含Observation标记
        String[] observationPatterns = {"Observation:", "observation:", "OBSERVATION:"};
        for (String pattern : observationPatterns) {
            if (fullResponse.toLowerCase().contains(pattern.toLowerCase())) {
                if (!isStepAlreadyTriggered(ReactStepType.OBSERVATION, fullResponse, pattern)) {
                    String content = extractContentAfterKeyword(fullResponse, pattern);
                    String[] nextKeywords = {"Thought:", "Action:", "Final Answer:"};
                    for (String nextKeyword : nextKeywords) {
                        int nextKeywordIndex = content.toLowerCase().indexOf(nextKeyword.toLowerCase());
                        if (nextKeywordIndex != -1) {
                            content = content.substring(0, nextKeywordIndex).trim();
                            break;
                        }
                    }
                    if (!content.isEmpty()) {
                        triggerObservationStep(content);
                    }
                }
                break;
            }
        }
    }
    
    /**
     * 处理Final Answer部分
     * 
     * @param fullResponse 完整响应内容
     */
    private void processFinalAnswerSection(String fullResponse) {
        // 检查是否包含Final Answer标记（包括带下划线和不带下划线的变体）
        String[] finalAnswerPatterns = {"Final Answer:", "final answer:", "FINAL ANSWER:", "Final_Answer:", "final_answer:", "FINAL_ANSWER:", "最终答案:"};
        for (String pattern : finalAnswerPatterns) {
            if (fullResponse.toLowerCase().contains(pattern.toLowerCase())) {
                if (!isStepAlreadyTriggered(ReactStepType.FINAL_ANSWER, fullResponse, pattern)) {
                    String content = extractContentAfterKeyword(fullResponse, pattern);
                    // Final Answer后的内容即为最终答案，不需要再查找下一个关键词
                    if (!content.isEmpty()) {
                        triggerFinalAnswerStep(content);
                    }
                }
                break;
            }
        }
    }
    
    /**
     * 检查某个步骤是否已经触发过，避免重复触发
     * 
     * @param stepType 步骤类型
     * @param fullResponse 完整响应内容
     * @param keyword 关键词
     * @return 如果步骤已经触发过则返回true，否则返回false
     */
    private boolean isStepAlreadyTriggered(ReactStepType stepType, String fullResponse, String keyword) {
        // 这里可以根据已有的步骤计数或内容来判断是否已经触发过
        // 为了简化，我们使用一个简单的检查：如果关键词在响应中出现多次，但只处理第一次
        int firstIndex = fullResponse.toLowerCase().indexOf(keyword.toLowerCase());
        int nextIndex = fullResponse.toLowerCase().indexOf(keyword.toLowerCase(), firstIndex + 1);
        
        // 如果没有找到关键词，返回true（无需触发）
        if (firstIndex == -1) {
            return true; // 没有关键词，无需触发
        }
        
        // 简单的检查：如果当前响应中包含关键词，且我们正在处理它，则认为未重复
        // 在这个简化版本中，我们假设每次检测到关键词时都处理它
        return false;
    }
    
    /**
     * 处理流式响应中的错误
     * 
     * @param throwable 异常对象
     * @param tokenConsumer token消费者
     */
    private void handleStreamError(Throwable throwable, Consumer<String> tokenConsumer) {
        errorHandlerService.handleStreamError(throwable, tokenConsumer, "ReAct流式处理");
    }
    
    /**
     * 发送完成事件
     * 
     * @param tokenConsumer token消费者
     * @param fullResponse 完整响应内容
     */
    private void sendCompletionEvent(Consumer<String> tokenConsumer, String fullResponse) {
        if (fullResponse == null) {
            fullResponse = "";
        }
        
        if (tokenConsumer instanceof TokenConsumerWithCompletion) {
            try {
                ((TokenConsumerWithCompletion) tokenConsumer).onComplete(fullResponse);
            } catch (NoClassDefFoundError e) {
                log.error("TokenConsumerWithCompletion依赖类未找到，跳过完成回调: {}", e.getMessage());
                // 如果类未找到，至少发送一个空消息以确保流的完整性
                if (tokenConsumer != null) {
                    try {
                        tokenConsumer.accept("");
                    } catch (Exception ex) {
                        log.error("发送空消息也失败", ex);
                    }
                }
            } catch (Exception e) {
                log.error("调用onComplete时发生错误", e);
            }
        } else if (tokenConsumer != null) {
            tokenConsumer.accept("");
        }
    }
    
    /**
     * 分析并记录工具事件
     * 
     * @param token 当前token
     * @param fullResponse 完整响应内容
     */
    // private void analyzeAndRecordToolEvents(String token, String fullResponse) {
    //     if (token == null || token.isEmpty()) {
    //         return;
    //     }
        
    //     String lowerToken = token.toLowerCase();
        
    //     if (lowerToken.startsWith("action:") || (lowerToken.contains("action:") && !fullResponse.contains("Observation:"))) {
    //         String toolName = extractToolName(token);
    //         Object toolArgs = extractToolArgs(token);
            
    //         if (toolName != null && !toolName.isEmpty()) {
    //             log.info("[ReAct Action] 检测到工具调用: {} 参数: {}", toolName, toolArgs);
    //         }
    //     }
        
    //     if (lowerToken.startsWith("observation:") || (lowerToken.contains("observation:") && fullResponse.contains("Observation:"))) {
    //         log.info("[ReAct Observation] 检测到工具结果: {}", token.substring(0, Math.min(100, token.length())));
    //     }
        
    //     if (lowerToken.startsWith("final answer:") || lowerToken.contains("final answer:")) {
    //         log.info("[ReAct Final Answer] 检测到最终答案");
    //     }
    // }
    
    /**
     * 从文本中提取工具名称
     * 
     * @param text 待解析的文本
     * @return 工具名称，如果未找到则返回null
     */
    private String extractToolName(String text) {
        if (text == null) return null;
        
        String[] patterns = {
            "Action:\\s*(\\w+)(?:\\(.*?\\))?$", // 匹配Action: toolName格式
            "Action:\\s*(\\w+)$", // 匹配Action: toolName格式
            "Action Input:\\s*\\{\\s*\"name\"\\s*:\\s*\"(\\w+)\"", // 匹配JSON中的工具名
            "Tool:\\s*(\\w+)",
            "tool:\\s*(\\w+)",
            "Calling\\s+(\\w+)",
            "calling\\s+(\\w+)",
            "Use\\s+(\\w+)",
            "use\\s+(\\w+)"
        };
        
        for (String pattern : patterns) {
            java.util.regex.Pattern p = java.util.regex.Pattern.compile(pattern, java.util.regex.Pattern.MULTILINE | java.util.regex.Pattern.CASE_INSENSITIVE);
            java.util.regex.Matcher m = p.matcher(text);
            if (m.find()) {
                return m.group(1);
            }
        }
        
        return null;
    }
    
    /**
     * 从文本中提取工具参数
     * 
     * @param text 待解析的文本
     * @return 工具参数，如果未找到则返回null
     */
    private Object extractToolArgs(String text) {
        if (text == null) return null;
            
        try {
            java.util.regex.Pattern actionInputPattern = java.util.regex.Pattern.compile("Action Input:\s*(.*)");
            java.util.regex.Matcher actionInputMatcher = actionInputPattern.matcher(text);
                
            if (actionInputMatcher.find()) {
                String inputStr = actionInputMatcher.group(1).trim();
                if (!inputStr.isEmpty() && inputStr.startsWith("{")) {
                    return parseJsonToMap(inputStr);
                }
            }
                
            java.util.regex.Pattern jsonPattern = java.util.regex.Pattern.compile("\\\"arguments\\\"\\s*:\\s*(\\{[^}]*\\})");
            java.util.regex.Matcher jsonMatcher = jsonPattern.matcher(text);
            if (jsonMatcher.find()) {
                return parseJsonToMap(jsonMatcher.group(1));
            }
                
        } catch (Exception e) {
            log.debug("解析工具参数时出错: {}", e.getMessage());
        }
            
        java.util.Map<String, Object> args = new java.util.HashMap<>();
        return args.isEmpty() ? null : args;
    }
    
    /**
     * 将JSON字符串解析为Map
     * 
     * @param jsonString JSON字符串
     * @return 解析后的Map对象
     */
    private java.util.Map<String, Object> parseJsonToMap(String jsonString) {
        try {
            jsonString = jsonString.trim().replaceAll("^\\{|\\}$", "").trim();
            java.util.Map<String, Object> map = new java.util.HashMap<>();
                
            if (!jsonString.isEmpty()) {
                String[] pairs = jsonString.split(",");
                for (String pair : pairs) {
                    String[] keyValue = pair.split(":", 2);
                    if (keyValue.length == 2) {
                        String key = keyValue[0].trim().replaceAll("^\"|\"$", "");
                        String value = keyValue[1].trim().replaceAll("^\"|\"$", "");
                        map.put(key, value);
                    }
                }
            }
            return map;
        } catch (Exception e) {
            log.debug("JSON解析失败: {}", e.getMessage());
            java.util.Map<String, Object> args = new java.util.HashMap<>();
            args.put("raw", jsonString);
            return args;
        }
    }
    
    /**
     * 提取关键词后的内容
     * 
     * @param text 原始文本
     * @param keyword 关键词
     * @return 关键词后的内容
     */
    private String extractContentAfterKeyword(String text, String keyword) {
        if (text == null || keyword == null) {
            return "";
        }
        
        // 找到关键词的位置
        int keywordIndex = -1;
        String lowerText = text.toLowerCase();
        String lowerKeyword = keyword.toLowerCase();
        
        // 首先尝试精确匹配
        keywordIndex = lowerText.indexOf(lowerKeyword);
        if (keywordIndex == -1) {
            // 如果没找到，尝试匹配带空格的版本
            keywordIndex = lowerText.indexOf(lowerKeyword.trim());
        }
        
        if (keywordIndex != -1) {
            // 提取关键词后的部分
            String content = text.substring(keywordIndex + keyword.length()).trim();
            
            // 如果内容以冒号开头，去掉它
            if (content.startsWith(":")) {
                content = content.substring(1).trim();
            }
            
            return content;
        }
        
        return text.trim();
    }
    
    /**
     * 触发思考步骤
     * 
     * @param content 思考内容
     */
    private void triggerThinkStep(String content) {
        int stepNumber = stepCounter.incrementAndGet();
        ReactStep reactStep = new ReactStep(stepNumber, ReactStepType.THOUGHT, content);
        notifyCallbacks(reactStep);
    }
    
    /**
     * 触发行动步骤
     * 
     * @param toolName 工具名称
     * @param toolAction 工具操作
     * @param toolArgs 工具参数
     */
    private void triggerActionStep(String toolName, String toolAction, Object toolArgs) {
        int stepNumber = stepCounter.incrementAndGet();
        ReactStep reactStep = new ReactStep(stepNumber, ReactStepType.ACTION, "执行工具: " + toolName);
        
        ReactStep.ToolCallAction toolActionObj = new ReactStep.ToolCallAction(toolName, toolArgs);
        reactStep.setAction(toolActionObj);
        
        notifyCallbacks(reactStep);
    }
    
    /**
     * 触发观察步骤
     * 
     * @param observation 观察内容
     */
    private void triggerObservationStep(String observation) {
        int stepNumber = stepCounter.incrementAndGet();
        ReactStep reactStep = new ReactStep(stepNumber, ReactStepType.OBSERVATION, observation);
        
        ReactStep.ToolObservation toolObservation = new ReactStep.ToolObservation(observation);
        reactStep.setObservation(toolObservation);
        
        notifyCallbacks(reactStep);
    }
    
    /**
     * 触发最终答案步骤
     * 
     * @param finalAnswer 最终答案
     */
    private void triggerFinalAnswerStep(String finalAnswer) {
        int stepNumber = stepCounter.incrementAndGet();
        ReactStep reactStep = new ReactStep(stepNumber, ReactStepType.FINAL_ANSWER, finalAnswer);
        notifyCallbacks(reactStep);
    }
    
    /**
     * 通知所有回调
     * 
     * @param reactStep ReAct步骤
     */
    private void notifyCallbacks(ReactStep reactStep) {
        for (ReactCallback callback : reactCallbacks) {
            try {
                callback.onStep(reactStep);
            } catch (Exception e) {
                log.error("执行ReAct回调时发生错误", e);
            }
        }
    }
    
    /**
     * 获取智能体工具
     * 
     * @param agent 智能体对象
     * @return 智能体可用的工具列表
     */
    private List<Object> getAgentTools(Agent agent) {
        if (agent == null) {
            List<Object> defaultTools = new ArrayList<>();
            defaultTools.add(dateTimeTools);
            return defaultTools;
        }
        
        try {
            List<Object> tools = agentToolManager.getAvailableToolInstances(agent);
            
            if (dateTimeTools != null && !tools.contains(dateTimeTools)) {
                tools.add(dateTimeTools);
            }
            
            return tools;
        } catch (Exception e) {
            log.error("获取工具实例时发生错误: {}", e.getMessage());
            List<Object> fallbackTools = new ArrayList<>();
            fallbackTools.add(dateTimeTools);
            return fallbackTools;
        }
    }
}