package pangea.hiagent.tool.aspect;

import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import pangea.hiagent.workpanel.IWorkPanelDataCollector;

import java.util.HashMap;
import java.util.Map;

/**
 * 工具执行日志记录切面类
 * 自动记录带有@Tool注解的方法执行信息，包括工具名称、方法名、输入参数、输出结果、运行时长等
 */
@Slf4j
@Aspect
@Component
public class ToolExecutionLoggerAspect {

    @Autowired
    private IWorkPanelDataCollector workPanelDataCollector;

    /**
     * 环绕通知，拦截所有带有@Tool注解的方法
     * @param joinPoint 连接点
     * @return 方法执行结果
     * @throws Throwable 异常
     */
    @Around("@annotation(tool)")
    public Object logToolExecution(ProceedingJoinPoint joinPoint, Tool tool) throws Throwable {
        // 获取方法签名
        MethodSignature signature = (MethodSignature) joinPoint.getSignature();
        String methodName = signature.getName();
        String className = signature.getDeclaringType().getSimpleName();
        String fullMethodName = className + "." + methodName;
        
        // 获取工具描述
        String toolDescription = tool.description();
        
        // 获取方法参数
        String[] paramNames = signature.getParameterNames();
        Object[] args = joinPoint.getArgs();
        
        // 构建输入参数映射
        Map<String, Object> inputParams = new HashMap<>();
        if (paramNames != null && args != null) {
            for (int i = 0; i < paramNames.length; i++) {
                if (i < args.length) {
                    inputParams.put(paramNames[i], args[i]);
                }
            }
        }
        
        // 记录开始时间
        long startTime = System.currentTimeMillis();
        
        // 记录工具调用开始
        if (workPanelDataCollector != null) {
            try {
                workPanelDataCollector.recordToolCallAction(className, inputParams, null, "pending", null);
            } catch (Exception e) {
                log.warn("记录工具调用开始时发生错误: {}", e.getMessage(), e);
            }
        }
        
        log.info("开始执行工具方法: {}，描述: {}", fullMethodName, toolDescription);
        log.debug("工具方法参数: {}", inputParams);
        
        try {
            // 执行原方法
            Object result = joinPoint.proceed();
            
            // 记录结束时间
            long endTime = System.currentTimeMillis();
            long executionTime = endTime - startTime;
            
            // 记录工具调用完成
            if (workPanelDataCollector != null) {
                try {
                    workPanelDataCollector.recordToolCallAction(className, inputParams, result, "success", executionTime);
                } catch (Exception e) {
                    log.warn("记录工具调用完成时发生错误: {}", e.getMessage(), e);
                }
            }
            
            log.info("工具方法执行成功: {}，描述: {}，耗时: {}ms", fullMethodName, toolDescription, executionTime);
            // 精简日志记录，避免过多的debug级别日志
            if (log.isTraceEnabled()) {
                log.trace("工具方法执行结果类型: {}，结果: {}",
                    result != null ? result.getClass().getSimpleName() : "null", result);
            }
            
            return result;
        } catch (Exception e) {
            // 记录结束时间
            long endTime = System.currentTimeMillis();
            long executionTime = endTime - startTime;
            
            // 记录工具调用错误
            if (workPanelDataCollector != null) {
                try {
                    workPanelDataCollector.recordToolCallAction(className, inputParams, e, "error", executionTime);
                } catch (Exception ex) {
                    log.warn("记录工具调用错误时发生错误: {}", ex.getMessage(), ex);
                }
            }
            
            log.error("工具方法执行失败: {}，描述: {}，耗时: {}ms，错误类型: {}",
                fullMethodName, toolDescription, executionTime, e.getClass().getSimpleName(), e);
            
            throw e;
        }
    }
}