package pangea.hiagent.agent.service;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;
import pangea.hiagent.model.Agent;
import pangea.hiagent.agent.processor.AgentProcessor;
import pangea.hiagent.agent.processor.AgentProcessorFactory;
import pangea.hiagent.agent.sse.UserSseService;
import pangea.hiagent.common.utils.LogUtils;
import pangea.hiagent.common.utils.ValidationUtils;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Agent处理器服务
 * 负责处理Agent处理器的获取和心跳机制
 */
@Slf4j
@Service
public class AgentProcessorService {
    
    @Autowired
    private AgentProcessorFactory agentProcessorFactory;
    
    @Autowired
    private UserSseService workPanelSseService;
    
    @Autowired
    private ChatErrorHandler chatErrorHandler;
    
    /**
     * 获取处理器并启动心跳保活机制
     *
     * @param agent Agent对象
     * @param emitter SSE发射器
     * @return Agent处理器，如果获取失败则返回null
     */
    public AgentProcessor getProcessorAndStartHeartbeat(Agent agent, SseEmitter emitter) {
        LogUtils.enterMethod("getProcessorAndStartHeartbeat", agent);
        // 参数验证
        if (ValidationUtils.isNull(agent, "agent")) {
            chatErrorHandler.handleChatError(emitter, "Agent对象不能为空");
            LogUtils.exitMethod("getProcessorAndStartHeartbeat", "Agent对象不能为空");
            return null;
        }
        
        if (ValidationUtils.isNull(emitter, "emitter")) {
            chatErrorHandler.handleChatError(emitter, "SSE发射器不能为空");
            LogUtils.exitMethod("getProcessorAndStartHeartbeat", "SSE发射器不能为空");
            return null;
        }
        
        try {
            // 根据Agent类型选择处理器并处理请求
            AgentProcessor processor = agentProcessorFactory.getProcessor(agent);
            if (processor == null) {
                chatErrorHandler.handleChatError(emitter, "无法获取Agent处理器");
                LogUtils.exitMethod("getProcessorAndStartHeartbeat", "无法获取Agent处理器");
                return null;
            }

            log.info("使用{} Agent处理器处理对话", processor.getProcessorType());

            // 启动心跳保活机制
            workPanelSseService.startHeartbeat(emitter, new AtomicBoolean(false));

            LogUtils.exitMethod("getProcessorAndStartHeartbeat", processor);
            return processor;
        } catch (Exception e) {
            chatErrorHandler.handleChatError(emitter, "获取处理器或启动心跳时发生错误", e, null);
            LogUtils.exitMethod("getProcessorAndStartHeartbeat", e);
            return null;
        }
    }
}