package pangea.hiagent.tool.impl;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.microsoft.playwright.*;
import com.microsoft.playwright.options.AriaRole;
import com.microsoft.playwright.options.Cookie;
import com.microsoft.playwright.options.WaitForSelectorState;
import com.microsoft.playwright.options.WaitUntilState;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.ai.chat.client.ChatClient;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.ai.tool.annotation.ToolParam;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;
import pangea.hiagent.agent.sse.UserSseService;
import pangea.hiagent.common.utils.Contants;
import pangea.hiagent.common.utils.HybridUniqueLongGenerator;
import pangea.hiagent.common.utils.InputCodeGenerator;
import pangea.hiagent.model.Agent;
import pangea.hiagent.model.UserToken;
import pangea.hiagent.web.service.AgentService;
import pangea.hiagent.web.service.InfoCollectorService;

import java.io.IOException;
import java.net.MalformedURLException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;

@Slf4j
public class VisitorAppointmentTool {
    public static final String pageId = "visitorAppointment";

    private static final String destUrl = "https://vrms-proxy.hisense.com/ipark/hichat/#/Liteapp";
    private static final DateTimeFormatter  customFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");

    private String ssoToken;

    // SSO密码（从配置文件读取）
    private String ldapToken;

    private String iparkToken;

    // Playwright实例
    private Playwright playwright;

    // 浏览器实例
    private Browser browser;

    // 共享的浏览器上下文，用于保持登录状态
    private BrowserContext sharedContext;

    // 上次登录时间
    private long lastLoginTime = 0;

    private AgentService agentService;
    private InfoCollectorService infoCollectorService;
    private UserSseService userSseService;


    // 登录状态有效期（毫秒），设置为30分钟
    private static final long LOGIN_VALIDITY_PERIOD = 30 * 60 * 1000;

    public VisitorAppointmentTool(UserToken userToken,AgentService agentService, InfoCollectorService infoCollectorService, UserSseService userSseService) {
        this.agentService = agentService;
        this.infoCollectorService = infoCollectorService;
        this.ssoToken = "d10bc61aa4e00dcc6f08de64ca42012814fdbcee9b88aa977f7fb07d3a4018f4";
        this.ldapToken = "AAECAzY5NDRBNTQ1Njk0NTRFMDV5b3V4aWFvamlaLv+jUGNEEORN24GLIC3OlqcCdw==";
        this.iparkToken = userToken.getTokenValue();
        this.userSseService = userSseService;
    }

    @PostConstruct
    public void initialize() {
        try {
            log.info("正在初始化海信SSO认证工具的Playwright...");
            this.playwright = Playwright.create();
            // 使用chromium浏览器，无头模式（headless=true），适合服务器运行
            // 可根据需要修改为有头模式（headless=false）用于调试
            this.browser = playwright.chromium().launch(new BrowserType.LaunchOptions().setHeadless(true));
            // 初始化共享上下文
            this.sharedContext = browser.newContext();
            // 检查是否已有有效的登录会话
            Cookie ssoTokenCookie = new Cookie("ssoLoginToken", ssoToken);
            ssoTokenCookie.setDomain(".hisense.com");
            ssoTokenCookie.setPath("/");
            Cookie ldapTokenCookie = new Cookie("LtpaToken", ldapToken);
            ldapTokenCookie.setDomain(".hisense.com");
            ldapTokenCookie.setPath("/");
            Cookie tripCookie = new Cookie("jwtToken", iparkToken);
            tripCookie.setDomain("vrms-proxy.hisense.com");
            tripCookie.setPath("/");

            //  String userName= SecurityContextHolder.getContext().getAuthentication().getName();


            List<Cookie> cookies = new ArrayList<>();
            cookies.add(ssoTokenCookie);
            cookies.add(ldapTokenCookie);
            cookies.add(tripCookie);
            sharedContext.addCookies(cookies);


            log.info("海信SSO认证工具的Playwright初始化成功");
        } catch (Exception e) {
            log.error("海信SSO认证工具的Playwright初始化失败: ", e);
        }
    }

    /**
     * 销毁Playwright资源
     */
    @PreDestroy
    public void destroy() {
        try {
            if (sharedContext != null) {
                sharedContext.close();
                log.info("海信SSO认证工具的共享浏览器上下文已关闭");
            }
            if (browser != null) {
                browser.close();
                log.info("海信SSO认证工具的浏览器实例已关闭");
            }
            if (playwright != null) {
                playwright.close();
                log.info("海信SSO认证工具的Playwright实例已关闭");
            }
        } catch (Exception e) {
            log.error("海信SSO认证工具的Playwright资源释放失败: ", e);
        }
    }
    @Tool(description = "提交访客预约申请")
    public String submitAppointmentApply(JSONObject jsonObject){
        sharedContext.tracing().start(new Tracing.StartOptions()
                .setScreenshots(true)
                .setSnapshots(true)
                .setSources(true));
        log.info("jsonObject = {}",jsonObject);
        JSONArray jsonArray = infoCollectorService.getInfo(pageId);
        long startTime = System.currentTimeMillis();

        Page page = null;

        try {

            page = sharedContext.newPage();
            page.setDefaultTimeout(2 * 60 * 1000);

            // 访问业务系统页面
            log.info("正在访问业务系统页面: {}", destUrl);
            String faviconUrl = "https://vrms-proxy.hisense.com/favicon.ico";
            page.navigate(faviconUrl);
            page.evaluate("() => sessionStorage.setItem('Access-Token', '" + iparkToken + "')");
            page.navigate(destUrl, new Page.NavigateOptions().setWaitUntil(WaitUntilState.NETWORKIDLE));
            // 检查是否重定向到了SSO登录页面
            String currentUrl = page.url();
            log.info("当前页面URL: {}", currentUrl);
            String parkValue = "";
            List<JSONObject> dateJson = new ArrayList<>();
            for(int i=0;i<jsonArray.size();i++){
                log.info("index {} ",i);
                JSONObject obj = jsonArray.getJSONObject(i);
                log.info("json {}",obj);
                String fieldName = obj.getString("field_name");

                String fieldValue = jsonObject.getString(obj.getString("code"));
                if(fieldName.compareToIgnoreCase("访问园区") == 0){
                    parkValue = fieldValue;
                    continue;
                }
                if(fieldName.compareToIgnoreCase("接访员工手机号")== 0 || fieldName.compareToIgnoreCase("接访员工姓名")==0){
                    continue;
                }

                if(fieldName.contains("日期")) {
                    dateJson.add(obj);
                    //fieldLocator.evaluate("el => el.value = '"+fieldValue+"'");;
                }else {
                    Locator fieldLocator = page.locator(".van-cell")
                            .filter(new Locator.FilterOptions().setHasText(Pattern.compile("^" + fieldName + "$")))
                            .locator("input");
                    Locator.FillOptions fillOptions = new Locator.FillOptions();
                    fillOptions.setForce(true);
                    fieldLocator .fill(fieldValue, fillOptions);
                }
                //saveScreenShot(page.screenshot(),"ipark");
            }
            page.locator(".van-cell")
                    .filter(new Locator.FilterOptions().setHasText(Pattern.compile("^访问园区$")))
                    .locator("input").click();
            Locator.WaitForOptions waitUntilOptions = new Locator.WaitForOptions();
            waitUntilOptions.setState(WaitForSelectorState.ATTACHED);
            page.locator(".van-overlay").waitFor(waitUntilOptions);
            page.getByText(parkValue, new Page.GetByTextOptions().setExact(false)).click();
            page.getByText("确认", new Page.GetByTextOptions().setExact(true)).click();
            page.locator(".van-overlay").waitFor(new Locator.WaitForOptions().setState(WaitForSelectorState.HIDDEN));
            for(JSONObject tmp:dateJson){
                String fieldName = tmp.getString("field_name");

                String fieldValue = jsonObject.getString(tmp.getString("code"));
                String[] values = fieldValue.split("-");

                page.locator(".van-cell")
                        .filter(new Locator.FilterOptions().setHasText(Pattern.compile("^"+fieldName+"$")))
                        .locator("input").click();
                page.locator(".van-overlay").first().waitFor(waitUntilOptions);

                page.getByRole(AriaRole.BUTTON, new Page.GetByRoleOptions().setName(values[0]+"年")).click();
                page.getByRole(AriaRole.BUTTON, new Page.GetByRoleOptions().setName(values[1]+"月")).click();
                page.getByRole(AriaRole.BUTTON, new Page.GetByRoleOptions().setName(values[2]+"日")).click();

                // 点击确认
                page.getByRole(AriaRole.BUTTON, new Page.GetByRoleOptions().setName("确认")).click();
                page.locator(".van-overlay").first().waitFor(new Locator.WaitForOptions().setState(WaitForSelectorState.HIDDEN));
            }
            saveScreenShot(page.screenshot(),"ipark");
            page.getByRole(AriaRole.BUTTON, new Page.GetByRoleOptions().setName("提交"))
                    .click();
            page.waitForTimeout(3000);
            if(page.url().compareToIgnoreCase(destUrl) == 0){
                log.info("有必填项未提供");
            }
            return "提交成功";
        }catch (Exception e){
            long endTime = System.currentTimeMillis();
            String errorMsg = "获取海信出差申请页面内容失败: " + e.getMessage();
            log.error("获取海信海信出差申请内容失败，耗时: {} ms", endTime - startTime, e);
            return errorMsg;
        }finally {
            // 释放页面资源
            if (page != null) {
                try {
                    page.close();
                } catch (Exception e) {
                    log.warn("关闭页面时发生异常: {}", e.getMessage());
                }
            }
            sharedContext.tracing().stop(new Tracing.StopOptions()
                    .setPath(Paths.get("trace1.zip")));
        }
    }
    private void saveScreenShot(byte[] bytes,String suffix){
        // 生成一个唯一的文件名，防止覆盖
        String timestamp = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMdd_HHmmss"));
        String fileName = "screenshot_" + timestamp +"_"+suffix+ ".png";

        try {
            // Paths.get() 指定存储路径，默认在项目根目录
            Files.write(Paths.get(fileName), bytes);
            log.info("截图已保存至: {}" , fileName);
        } catch (IOException e) {
            log.info("保存截图失败: " + e.getMessage());
            e.printStackTrace();
        }
    }

    @Tool(description = "如果在用户信息中有任何与访客预约相关的信息，调用这个工具来保存访客预约信息")
    public String applyInfoSave(@ToolParam(required = true) JSONObject infos) throws IOException {
        log.info("applyInfoSave(infos={})", infos);
        infos.keySet().forEach(key -> {
            infoCollectorService.saveValue(key, infos.get(key));
        });
        Set<String> keys = infoCollectorService.findLackInfo(pageId);
        JSONArray jsonArray = infoCollectorService.getInfo(pageId);
        JSONArray lackJson = new JSONArray();
        for(int i=0;i<jsonArray.size();i++){
            JSONObject tmp = jsonArray.getJSONObject(i);
            if(keys.contains(tmp.getString("field_name"))){
                lackJson.add(tmp.getJSONObject("pangea_json"));
            }
        }
        JSONObject formMessage = new JSONObject();
        formMessage.put("coms", lackJson);
        sendFormMessage(formMessage);
        StringBuilder sb = new StringBuilder();
        if (keys.isEmpty()) {
            sb.append("用户已提交全部数据，提示用户提交申请");
        } else {
            sb.append("用户还有以下信息未提交:");
            sb.append("\n");
            for (String key : keys) {
                sb.append(key);
                sb.append("\n");
            }
            sb.append("提示用户继续以json格式提交信息");

        }
        return sb.toString();
    }
    private void sendFormMessage(JSONObject formMessage) throws IOException {
        SseEmitter sseEmitter = userSseService.getEmitter("worker1");
        log.info("Send Form Message {}", formMessage);
        sseEmitter.send(SseEmitter.event().name("form").data(formMessage));
    }

    /**
     * 工具方法：获取海信访客预约申请的网页内容
     *
     * @return 页面内容（HTML文本）
     */
    @Tool(description = "获取访客预约申请必要信息")
    public String getAppointmentApplyNecessaryInfo() {
        if(infoCollectorService.exists(pageId)){
            return "已获取必要信息，保存用户提交的信息";
        }
        long startTime = System.currentTimeMillis();

        Page page = null;

        try {

            page = sharedContext.newPage();
            page.setDefaultTimeout(2 * 60 * 1000);

            // 访问业务系统页面
            log.info("正在访问业务系统页面: {}", destUrl);
            String faviconUrl = "https://vrms-proxy.hisense.com/favicon.ico";
            page.navigate(faviconUrl);
            page.evaluate("() => sessionStorage.setItem('Access-Token', '"+iparkToken+"')");
            page.navigate(destUrl, new Page.NavigateOptions().setWaitUntil(WaitUntilState.NETWORKIDLE));
            // 检查是否重定向到了SSO登录页面
            String currentUrl = page.url();
            log.info("当前页面URL: {}", currentUrl);

            // 如果页面尚未导航到业务系统URL，则导航到该URL
            if (!page.url().equals(destUrl) && !page.url().startsWith(destUrl)) {
                log.info("正在访问业务系统页面: {}", destUrl);
                page.navigate(destUrl, new Page.NavigateOptions().setWaitUntil(WaitUntilState.NETWORKIDLE));
            }
            JSONArray jsonArray = null;
            if (!infoCollectorService.exists(pageId)) {
                JSONArray tmp = getLocators(page.locator("body").innerHTML());
                jsonArray = tmp;
                infoCollectorService.register(pageId, tmp);
            } else {
                jsonArray = infoCollectorService.getInfo(pageId);
            }
            StringBuilder stringBuilder = new StringBuilder();
            // 提取页面内容
            String content = stringBuilder.toString();
            JSONObject formMessage = generateJson(jsonArray);
            sendFormMessage(formMessage);
            long endTime = System.currentTimeMillis();
            log.info("成功获取海信出差申请页面内容，耗时: {} ms", endTime - startTime);
            log.info("用户需要提交的信息包括:{}", formMessage);
            stringBuilder.append(formMessage.toJSONString());
            stringBuilder.append("提示用户以json格式提交信息;如果用户已提供部分信息,需要将这些信息与`props.name`属性的值进行匹配，并将匹配之后的信息以json格式提交到`applyInfoSave`以保存信息");

            return stringBuilder.toString();
        } catch (Exception e) {
            long endTime = System.currentTimeMillis();
            String errorMsg = "获取海信出差申请页面内容失败: " + e.getMessage();
            log.error("获取海信海信出差申请内容失败，耗时: {} ms", endTime - startTime, e);
            return errorMsg;
        } finally {
            // 释放页面资源
            if (page != null) {
                try {
                    page.close();
                } catch (Exception e) {
                    log.warn("关闭页面时发生异常: {}", e.getMessage());
                }
            }
        }
    }

    public JSONObject generateJson(JSONArray jsonArray) {
        JSONArray components = new JSONArray();
        for (int i = 0; i < jsonArray.size(); i++) {
            JSONObject tmp = jsonArray.getJSONObject(i);
            JSONObject obj = buildComponents(tmp);
            tmp.put("code", obj.getJSONObject("props").getString("name"));
            tmp.put("pangea_json",obj);
            components.add(obj);
        }
        JSONObject obj = new JSONObject();
        obj.put("coms", components);
        return obj;
    }

    private JSONObject buildComponents(JSONObject source) {
        JSONObject obj = new JSONObject();
        obj.put("key", HybridUniqueLongGenerator.generateUnique13DigitNumber());
        obj.put("name", "输入框");
        String name = InputCodeGenerator.generateUniqueInputCode("INPUT_");
        JSONObject props = new JSONObject();
        props.put("title", source.getString("field_name"));
        props.put("status", "default");
        if (source.getString("field_name").contains("日期")) {
            obj.put("code", "HiDatePicker");
            name = InputCodeGenerator.generateUniqueInputCode("DATE_");
            props.put("format", "YYYY-MM-DD");
        } else {
            obj.put("code", "HiInput");
        }
        props.put("name", name);
        obj.put("props", props);
        return obj;
    }


    private JSONArray getLocators(String html) throws MalformedURLException {
//        System.out.println("----------------------------------------");
//        System.out.println(html);
//
//        System.out.println("----------------------------------------");

        Agent agent = agentService.getAgent("agent-7");
        ChatClient chatClient = ChatClient.builder(agentService.getChatModelForAgent(agent)).build();
        String systemPrompt = "你是一个网页解析助手，你可以将html {htmlData} 中所有的必填项的名称标题和对应的html元素的唯一的定位表达式，html标签类型，attributes完整的解析出来;无论元素是否动态生成，都需要解析;以{jsonSchema}格式告诉我;定位表达式要求可以直接被playwright使用，准确并且可以定位唯一元素";
        JSONArray response = chatClient.prompt().user(u -> u.text(systemPrompt).param("htmlData", html).param("jsonSchema", Contants.LOCATOR_SCHEMA))
                .call()
                .entity(JSONArray.class);

        // 获取响应文本
        // String responseText = response.getResult().getOutput().getText();

        log.info(response.toJSONString());
        return response;
    }
}
