// 异常监控工具类
import { ElMessage, ElNotification } from 'element-plus'

// 异常统计信息接口
interface ExceptionStatistics {
  [key: string]: number
}

// 异常详细信息接口
interface ExceptionDetails {
  [key: string]: string
}

/**
 * 异常监控类
 * 负责监控、统计和报告前端异常情况
 */
class ExceptionMonitoring {
  // 异常统计信息
  private exceptionCounters: Map<string, number> = new Map()
  
  // 异常详细信息缓存
  private exceptionDetails: Map<string, string> = new Map()
  
  // 最大缓存条目数
  private readonly MAX_CACHE_SIZE: number = 1000
  
  /**
   * 记录异常信息
   * 
   * @param exceptionType 异常类型
   * @param errorMessage 错误消息
   * @param stackTrace 堆栈跟踪
   */
  recordException(exceptionType: string, errorMessage: string, stackTrace?: string): void {
    try {
      // 更新异常计数器
      const currentCount = this.exceptionCounters.get(exceptionType) || 0
      this.exceptionCounters.set(exceptionType, currentCount + 1)
      
      // 记录异常详细信息（保留最新的）
      const detailKey = `${exceptionType}_${Date.now()}`
      const detailInfo = this.formatExceptionDetail(exceptionType, errorMessage, stackTrace)
      this.exceptionDetails.set(detailKey, detailInfo)
      
      // 控制缓存大小
      if (this.exceptionDetails.size > this.MAX_CACHE_SIZE) {
        // 移除最老的条目
        const firstKey = this.exceptionDetails.keys().next().value
        if (firstKey) {
          this.exceptionDetails.delete(firstKey)
        }
      }
      
      // 显示通知
      ElNotification({
        title: '异常监控',
        message: `类型: ${exceptionType}, 消息: ${errorMessage}`,
        type: 'error',
        duration: 5000
      })
    } catch (e) {
      console.warn('记录异常信息时发生错误:', e)
    }
  }
  
  /**
   * 格式化异常详细信息
   * 
   * @param exceptionType 异常类型
   * @param errorMessage 错误消息
   * @param stackTrace 堆栈跟踪
   * @returns 格式化后的异常详细信息
   */
  private formatExceptionDetail(exceptionType: string, errorMessage: string, stackTrace?: string): string {
    let detail = `异常类型: ${exceptionType}\n`
    detail += `错误消息: ${errorMessage}\n`
    detail += `发生时间: ${new Date().toISOString()}\n`
    
    if (stackTrace) {
      detail += `堆栈跟踪:\n${stackTrace}\n`
    }
    
    detail += '---\n'
    return detail
  }
  
  /**
   * 获取异常统计信息
   * 
   * @returns 异常统计信息
   */
  getExceptionStatistics(): ExceptionStatistics {
    const statistics: ExceptionStatistics = {}
    this.exceptionCounters.forEach((value, key) => {
      statistics[key] = value
    })
    return statistics
  }
  
  /**
   * 获取指定类型的异常计数
   * 
   * @param exceptionType 异常类型
   * @returns 异常计数
   */
  getExceptionCount(exceptionType: string): number {
    return this.exceptionCounters.get(exceptionType) || 0
  }
  
  /**
   * 获取所有异常详细信息
   * 
   * @returns 异常详细信息
   */
  getExceptionDetails(): ExceptionDetails {
    const details: ExceptionDetails = {}
    this.exceptionDetails.forEach((value, key) => {
      details[key] = value
    })
    return details
  }
  
  /**
   * 清空异常统计信息
   */
  clearStatistics(): void {
    this.exceptionCounters.clear()
  }
  
  /**
   * 清空异常详细信息
   */
  clearDetails(): void {
    this.exceptionDetails.clear()
  }
  
  /**
   * 清空所有异常信息
   */
  clearAll(): void {
    this.clearStatistics()
    this.clearDetails()
  }
  
  /**
   * 从后端获取异常统计信息
   * 
   * @returns 异常统计信息
   */
  async fetchBackendExceptionStatistics(): Promise<ExceptionStatistics> {
    try {
      const response = await fetch('/api/monitoring/exceptions/statistics')
      if (response.ok) {
        return await response.json()
      } else {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`)
      }
    } catch (error) {
      console.error('获取后端异常统计信息失败:', error)
      ElMessage.error('获取后端异常统计信息失败')
      return {}
    }
  }
  
  /**
   * 从后端获取指定类型的异常计数
   * 
   * @param exceptionType 异常类型
   * @returns 异常计数
   */
  async fetchBackendExceptionCount(exceptionType: string): Promise<number> {
    try {
      const response = await fetch(`/api/monitoring/exceptions/count/${encodeURIComponent(exceptionType)}`)
      if (response.ok) {
        const text = await response.text()
        return parseInt(text, 10) || 0
      } else {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`)
      }
    } catch (error) {
      console.error('获取后端异常计数失败:', error)
      ElMessage.error('获取后端异常计数失败')
      return 0
    }
  }
  
  /**
   * 从后端获取所有异常详细信息
   * 
   * @returns 异常详细信息
   */
  async fetchBackendExceptionDetails(): Promise<ExceptionDetails> {
    try {
      const response = await fetch('/api/monitoring/exceptions/details')
      if (response.ok) {
        return await response.json()
      } else {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`)
      }
    } catch (error) {
      console.error('获取后端异常详细信息失败:', error)
      ElMessage.error('获取后端异常详细信息失败')
      return {}
    }
  }
  
  /**
   * 清空后端异常统计信息
   */
  async clearBackendStatistics(): Promise<void> {
    try {
      const response = await fetch('/api/monitoring/exceptions/statistics', {
        method: 'DELETE'
      })
      
      if (response.ok) {
        ElMessage.success('后端异常统计信息已清空')
      } else {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`)
      }
    } catch (error) {
      console.error('清空后端异常统计信息失败:', error)
      ElMessage.error('清空后端异常统计信息失败')
    }
  }
  
  /**
   * 清空后端异常详细信息
   */
  async clearBackendDetails(): Promise<void> {
    try {
      const response = await fetch('/api/monitoring/exceptions/details', {
        method: 'DELETE'
      })
      
      if (response.ok) {
        ElMessage.success('后端异常详细信息已清空')
      } else {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`)
      }
    } catch (error) {
      console.error('清空后端异常详细信息失败:', error)
      ElMessage.error('清空后端异常详细信息失败')
    }
  }
  
  /**
   * 清空所有后端异常信息
   */
  async clearAllBackend(): Promise<void> {
    try {
      const response = await fetch('/api/monitoring/exceptions/all', {
        method: 'DELETE'
      })
      
      if (response.ok) {
        ElMessage.success('所有后端异常信息已清空')
      } else {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`)
      }
    } catch (error) {
      console.error('清空所有后端异常信息失败:', error)
      ElMessage.error('清空所有后端异常信息失败')
    }
  }
}

// 导出单例实例
export const exceptionMonitoring = new ExceptionMonitoring()

// 导出类型定义
export type { ExceptionStatistics, ExceptionDetails }