package pangea.hiagent.sse;

import lombok.extern.slf4j.Slf4j;
import pangea.hiagent.web.dto.WorkPanelEvent;
import pangea.hiagent.workpanel.event.EventService;

import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.io.IOException;
import java.util.concurrent.ScheduledFuture;

/**
 * 统一SSE服务类
 * 整合SSE连接管理、生命周期管理、回调处理等功能
 */
@Slf4j
@Service
public class WorkPanelSseService {
    
    // 存储所有活动的 emitter
    private final List<SseEmitter> emitters = new CopyOnWriteArrayList<>();
    
    // 存储用户ID到SSE Emitter的映射关系
    private final ConcurrentMap<String, SseEmitter> userEmitters = new ConcurrentHashMap<>();
    
    // 存储SSE Emitter到用户ID的反向映射关系（用于快速查找）
    private final ConcurrentMap<SseEmitter, String> emitterUsers = new ConcurrentHashMap<>();
    
    // 心跳任务执行器 - 使用共享线程池以提高资源利用率
    private final ScheduledExecutorService heartbeatExecutor;
    
    // SSE超时时间（毫秒）
    private static final long SSE_TIMEOUT = 120000L; // 2分钟超时，提高连接稳定性
    
    private final EventService eventService;
    
    public WorkPanelSseService(EventService eventService) {
        this.eventService = eventService;
        this.heartbeatExecutor = Executors.newScheduledThreadPool(2);
    }
    
    /**
     * 创建并注册SSE连接
     * 
     * @param userId 用户ID
     * @return SSE Emitter
     */
    public SseEmitter createAndRegisterConnection(String userId) {
        log.debug("开始为用户 {} 创建SSE连接", userId);
        
        // 创建 SSE emitter
        SseEmitter emitter = createEmitter();
        log.debug("SSE Emitter创建成功");
        
        // 注册用户的SSE连接
        registerSession(userId, emitter);
        log.debug("用户 {} 的SSE连接注册成功", userId);
        
        // 注册 emitter 回调
        registerCallbacks(emitter, userId);
        log.debug("SSE Emitter回调注册成功");
        
        // 启动心跳机制
        startHeartbeat(emitter, new AtomicBoolean(false));
        log.debug("心跳机制启动成功");
        
        log.info("用户 {} 的SSE连接创建和注册完成", userId);
        return emitter;
    }
    
    /**
     * 创建SSE发射器
     * 
     * @return SSE发射器
     */
    public SseEmitter createEmitter() {
        SseEmitter emitter = new SseEmitter(SSE_TIMEOUT);
        registerCallbacks(emitter);
        emitters.add(emitter);
        return emitter;
    }
    
    /**
     * 注册用户的SSE连接
     * 如果该用户已有连接，则先关闭旧连接再注册新连接
     * 
     * @param userId 用户ID
     * @param emitter SSE Emitter
     * @return true表示注册成功，false表示注册失败
     */
    public boolean registerSession(String userId, SseEmitter emitter) {
        if (userId == null || userId.isEmpty() || emitter == null) {
            log.warn("注册SSE会话失败：用户ID或Emitter为空");
            return false;
        }
        
        try {
            // 检查该用户是否已有连接
            SseEmitter existingEmitter = userEmitters.get(userId);
            if (existingEmitter != null) {
                log.info("用户 {} 已有SSE连接，关闭旧连接", userId);
                try {
                    existingEmitter.complete();
                } catch (Exception e) {
                    log.warn("关闭用户 {} 的旧SSE连接时发生异常: {}", userId, e.getMessage());
                }
                // 从映射表中移除旧连接
                userEmitters.remove(userId);
                emitterUsers.remove(existingEmitter);
            }
            
            // 注册新连接
            userEmitters.put(userId, emitter);
            emitterUsers.put(emitter, userId);
            
            log.info("用户 {} 的SSE连接注册成功", userId);
            return true;
        } catch (Exception e) {
            log.error("注册用户 {} 的SSE连接时发生异常", userId, e);
            return false;
        }
    }
    
    /**
     * 获取用户的SSE连接
     * 
     * @param userId 用户ID
     * @return SSE Emitter，如果不存在则返回null
     */
    public SseEmitter getSession(String userId) {
        return userEmitters.get(userId);
    }
    
    /**
     * 处理连接完成事件
     * 
     * @param emitter SSE Emitter
     */
    public void handleConnectionCompletion(SseEmitter emitter) {
        if (emitter == null) {
            return;
        }
        
        try {
            // 从映射表中移除连接
            String userId = emitterUsers.remove(emitter);
            if (userId != null) {
                userEmitters.remove(userId);
            }
            emitters.remove(emitter);
            
            log.debug("SSE连接完成，用户: {}", userId);
        } catch (Exception e) {
            log.error("处理SSE连接完成事件时发生异常", e);
        }
    }
    
    /**
     * 处理连接超时事件
     * 
     * @param emitter SSE Emitter
     */
    public void handleConnectionTimeout(SseEmitter emitter) {
        if (emitter == null) {
            return;
        }
        
        try {
            // 从映射表中移除连接
            String userId = emitterUsers.remove(emitter);
            if (userId != null) {
                userEmitters.remove(userId);
            }
            emitters.remove(emitter);
            
            log.debug("SSE连接超时，用户: {}", userId);
        } catch (Exception e) {
            log.error("处理SSE连接超时事件时发生异常", e);
        }
    }
    
    /**
     * 处理连接错误事件
     * 
     * @param emitter SSE Emitter
     */
    public void handleConnectionError(SseEmitter emitter) {
        if (emitter == null) {
            return;
        }
        
        try {
            // 从映射表中移除连接
            String userId = emitterUsers.remove(emitter);
            if (userId != null) {
                userEmitters.remove(userId);
            }
            emitters.remove(emitter);
            
            log.debug("SSE连接错误，用户: {}", userId);
        } catch (Exception e) {
            log.error("处理SSE连接错误事件时发生异常", e);
        }
    }
    
    /**
     * 移除SSE发射器
     * 
     * @param emitter SSE发射器
     */
    public void removeEmitter(SseEmitter emitter) {
        if (emitter != null && emitters.remove(emitter)) {
            log.debug("已移除SSE Emitter，剩余连接数: {}", emitters.size());
        }
    }
    
    /**
     * 启动心跳机制
     * 
     * @param emitter SSE发射器
     * @param isCompleted 是否已完成
     */
    public void startHeartbeat(SseEmitter emitter, AtomicBoolean isCompleted) {
        if (emitter == null) {
            log.warn("SSE发射器为空，无法启动心跳机制");
            return;
        }
        
        // 创建心跳任务并保存ScheduledFuture引用
        ScheduledFuture<?> heartbeatTask = heartbeatExecutor.scheduleAtFixedRate(() -> {
            // 检查emitter是否已经完成
            if (isCompleted.get() || !isEmitterValid(emitter)) {
                log.debug("SSE Emitter已完成或无效，取消心跳任务");
                // 返回前确保任务被取消
                return;
            }
            
            try {
                // 发送心跳事件
                sendHeartbeat(emitter);
            } catch (Exception e) {
                log.warn("发送心跳事件失败: {}", e.getMessage());
                // 如果是emitter已完成的异常，标记为已完成
                if (e instanceof IllegalStateException && 
                    e.getMessage() != null && 
                    e.getMessage().contains("Emitter is already completed")) {
                    isCompleted.set(true);
                }
            }
        }, 30, 30, TimeUnit.SECONDS); // 每30秒发送一次心跳
        
        // 注册回调，在连接完成时取消心跳任务
        emitter.onCompletion(() -> {
            if (heartbeatTask != null && !heartbeatTask.isCancelled()) {
                heartbeatTask.cancel(true);
                log.debug("SSE连接完成，心跳任务已取消");
            }
        });
        
        // 注册回调，在连接超时时取消心跳任务
        emitter.onTimeout(() -> {
            if (heartbeatTask != null && !heartbeatTask.isCancelled()) {
                heartbeatTask.cancel(true);
                log.debug("SSE连接超时，心跳任务已取消");
            }
        });
        
        // 注册回调，在连接错误时取消心跳任务
        emitter.onError(throwable -> {
            if (heartbeatTask != null && !heartbeatTask.isCancelled()) {
                heartbeatTask.cancel(true);
                log.debug("SSE连接错误，心跳任务已取消");
            }
        });
    }
    
    /**
     * 注册回调函数
     * 
     * @param emitter SSE发射器
     */
    public void registerCallbacks(SseEmitter emitter) {
        emitter.onCompletion(() -> {
            log.debug("SSE连接完成");
            removeEmitter(emitter);
        });
        emitter.onError((Throwable t) -> {
            log.error("SSE连接发生错误: {}", t.getMessage(), t);
            removeEmitter(emitter);
        });
        emitter.onTimeout(() -> {
            log.warn("SSE连接超时");
            try {
                emitter.complete();
            } catch (Exception e) {
                log.error("关闭SSE连接时发生异常: {}", e.getMessage(), e);
            }
            removeEmitter(emitter);
        });
    }
    
    /**
     * 注册Emitter回调函数
     * 职责：注册所有必要的回调处理函数
     * 
     * @param emitter SSE Emitter
     * @param userId 用户ID
     */
    public void registerCallbacks(SseEmitter emitter, String userId) {
        emitter.onCompletion(() -> {
            log.debug("SSE连接完成");
            // 通知用户连接管理器连接已完成
            handleConnectionCompletion(emitter);
        });
        
        emitter.onTimeout(() -> {
            log.warn("SSE连接超时");
            try {
                emitter.complete();
            } catch (Exception e) {
                log.error("关闭SSE连接失败", e);
            }
            // 通知用户连接管理器连接已超时
            handleConnectionTimeout(emitter);
        });
        
        emitter.onError(throwable -> {
            // 记录详细的错误信息，包括异常类型和消息
            String errorMessage = throwable.getMessage();
            String errorType = throwable.getClass().getSimpleName();
            log.error("SSE连接错误 - 类型: {}, 消息: {}", errorType, errorMessage, throwable);
            // 通知用户连接管理器连接发生错误
            handleConnectionError(emitter);
        });
        
        // 注册 emitter 到管理器
        registerCallbacks(emitter);
    }
    
    /**
     * 完成SSE发射器
     * 
     * @param emitter SSE发射器
     * @param isCompleted 是否已完成
     */
    public void completeEmitter(SseEmitter emitter, AtomicBoolean isCompleted) {
        if (isCompleted != null) {
            isCompleted.set(true);
        }
        if (emitter != null) {
            try {
                emitter.complete();
            } catch (Exception e) {
                log.warn("完成Emitter时发生异常: {}", e.getMessage());
            }
        }
    }
    
    /**
     * 检查SSE Emitter是否仍然有效
     * 职责：提供轻量级的连接有效性检查
     * 
     * @param emitter 要检查的emitter
     * @return 如果有效返回true，否则返回false
     */
    public boolean isEmitterValid(SseEmitter emitter) {
        if (emitter == null) {
            return false;
        }
        
        // 检查逻辑，仅通过尝试发送ping事件来验证连接状态
        try {
            // 尝试发送一个空事件来检查连接状态
            emitter.send(SseEmitter.event().name("ping").data("").build());
            return true;
        } catch (Exception ex) {
            // 如果出现任何异常，认为连接已失效
            return false;
        }
    }
    
    /**
     * 发送SSE事件
     * 职责：统一发送SSE事件的基础方法
     * 
     * @param emitter SSE发射器
     * @param eventName 事件名称
     * @param data 事件数据
     * @throws IOException IO异常
     */
    public void sendEvent(SseEmitter emitter, String eventName, Object data) throws IOException {
        // 参数验证
        if (emitter == null || eventName == null || eventName.isEmpty() || data == null) {
            log.warn("参数验证失败，无法发送事件");
            return;
        }
        
        try {
            emitter.send(SseEmitter.event().name(eventName).data(data));
        } catch (IllegalStateException e) {
            // 处理 emitter 已关闭的情况
            log.debug("无法发送事件，emitter已关闭: {}", e.getMessage());
            // 不重新抛出异常，避免影响主流程
        }
    }
    
    /**
     * 发送心跳事件
     * 
     * @param emitter SSE发射器
     * @throws IOException IO异常
     */
    public void sendHeartbeat(SseEmitter emitter) throws IOException {
        if (emitter == null) {
            log.warn("SSE发射器为空，无法发送心跳事件");
            return;
        }
        
        try {
            // 发送心跳事件
            emitter.send(SseEmitter.event().name("heartbeat").data(System.currentTimeMillis()));
        } catch (IllegalStateException e) {
            // 处理 emitter 已关闭的情况
            log.debug("无法发送心跳事件，emitter已关闭: {}", e.getMessage());
            // 不重新抛出异常，避免影响主流程
        } catch (Exception e) {
            log.warn("发送心跳事件失败: {}", e.getMessage());
            throw e;
        }
    }
    
    /**
     * 发送工作面板事件给指定的SSE连接
     * 
     * @param emitter SSE发射器
     * @param event 工作面板事件
     * @throws IOException IO异常
     */
    public void sendWorkPanelEvent(SseEmitter emitter, WorkPanelEvent event) throws IOException {
        if (event == null) {
            log.warn("工作面板事件为空，无法发送事件");
            return;
        }

        try {
            // 构建事件数据
            Map<String, Object> data = eventService.buildWorkPanelEventData(event);
            
            if (data != null) {
                log.debug("准备发送工作面板事件: 类型={}, 事件内容={}", event.getType(), event);
                log.debug("事件数据: {}", data);
                
                // 发送事件
                emitter.send(SseEmitter.event().name("message").data(data));
                
                log.debug("工作面板事件发送成功: 类型={}", event.getType());
            } else {
                log.warn("构建事件数据失败，无法发送事件: 类型={}", event.getType());
            }
        } catch (Exception e) {
            // 记录详细错误信息，但不中断主流程
            log.error("发送工作面板事件失败: 类型={}, 错误={}", event.getType(), e.getMessage(), e);
            
            // 如果是连接已关闭的异常，重新抛出以便上层处理
            if (e instanceof IllegalStateException && e.getMessage() != null && 
                e.getMessage().contains("Emitter is already completed")) {
                throw e;
            }
        }
    }
    
    /**
     * 发送工作面板事件给指定用户
     * 
     * @param userId 用户ID
     * @param event 工作面板事件
     */
    public void sendWorkPanelEventToUser(String userId, WorkPanelEvent event) {
        log.debug("开始向用户 {} 发送工作面板事件: {}", userId, event.getType());
        
        // 检查连接是否仍然有效
        SseEmitter emitter = getSession(userId);
        if (emitter != null) {
            try {
                // 直接向当前 emitter 发送事件
                sendWorkPanelEvent(emitter, event);
                log.debug("已发送工作面板事件到客户端: {}", event.getType());
            } catch (IOException e) {
                log.error("发送工作面板事件失败: {}", e.getMessage(), e);
            }
        } else {
            log.debug("连接已失效，跳过发送事件: {}", event.getType());
        }
    }
    
    /**
     * 发送连接成功事件
     * 
     * @param emitter SSE发射器
     * @throws IOException IO异常
     */
    public void sendConnectedEvent(SseEmitter emitter) throws IOException {
        if (emitter == null) {
            log.warn("SSE发射器为空，无法发送连接成功事件");
            return;
        }
        
        try {
            WorkPanelEvent connectedEvent = WorkPanelEvent.builder()
                .type("observation")
                .title("连接成功")
                .timestamp(System.currentTimeMillis())
                .build();
                
            Map<String, Object> data = eventService.buildWorkPanelEventData(connectedEvent);
            emitter.send(SseEmitter.event().name("message").data(data));
            
            log.debug("已发送连接成功事件");
        } catch (IOException e) {
            log.error("发送连接成功事件失败", e);
            throw e;
        }
    }
    
    /**
     * 获取所有活动的emitters
     * 
     * @return 所有活动的emitters列表
     */
    public List<SseEmitter> getEmitters() {
        return new ArrayList<>(emitters);
    }
    
    /**
     * 销毁资源
     */
    public void destroy() {
        // 关闭线程池
        heartbeatExecutor.shutdown();
        try {
            if (!heartbeatExecutor.awaitTermination(5, TimeUnit.SECONDS)) {
                heartbeatExecutor.shutdownNow();
            }
        } catch (InterruptedException e) {
            heartbeatExecutor.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }
}