package pangea.hiagent.agent.service;

import lombok.extern.slf4j.Slf4j;

import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import pangea.hiagent.agent.processor.AgentProcessor;
import pangea.hiagent.agent.processor.AgentProcessorFactory;
import pangea.hiagent.common.utils.UserUtils;
import pangea.hiagent.web.dto.ChatRequest;
import pangea.hiagent.model.Agent;
import pangea.hiagent.sse.WorkPanelSseService;
import pangea.hiagent.tool.AgentToolManager;
import pangea.hiagent.web.dto.AgentRequest;
import pangea.hiagent.workpanel.event.EventService;
import jakarta.servlet.http.HttpServletResponse;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * Agent 对话服务
 * 职责：协调整个AI对话流程，作为流式处理的统一入口和协调者
 */
@Slf4j
@Service
public class AgentChatService {

    private final ChatErrorHandler chatErrorHandler;
    private final AgentValidationService agentValidationService;
    private final AgentProcessorFactory agentProcessorFactory;
    private final StreamRequestService streamRequestService;
    private final AgentToolManager agentToolManager;
    private final WorkPanelSseService workPanelSseService;
    
    public AgentChatService(
            EventService eventService,
            ChatErrorHandler chatErrorHandler,
            AgentValidationService agentValidationService,
            AgentProcessorFactory agentProcessorFactory,
            StreamRequestService streamRequestService,
            AgentToolManager agentToolManager,
            WorkPanelSseService workPanelSseService) {
        this.chatErrorHandler = chatErrorHandler;
        this.agentValidationService = agentValidationService;
        this.agentProcessorFactory = agentProcessorFactory;
        this.streamRequestService = streamRequestService;
        this.agentToolManager = agentToolManager;
        this.workPanelSseService = workPanelSseService;
    }
    
    // 专用线程池配置 - 使用静态变量确保线程池在整个应用中是单例的
    private static final ExecutorService executorService = new ThreadPoolExecutor(
            20,
            200,
            60L,
            TimeUnit.SECONDS,
            new LinkedBlockingQueue<>(1000),
            new ThreadPoolExecutor.CallerRunsPolicy()
    );

    // /**
    //  * 处理同步对话请求的统一入口
    //  * @param agent Agent对象
    //  * @param request 请求对象
    //  * @param userId 用户ID
    //  * @return 处理结果
    //  */
    // public String handleChatSync(Agent agent, AgentRequest request, String userId) {
    //     log.info("开始处理同步对话请求，AgentId: {}, 用户消息: {}", agent.getId(), request.getUserMessage());
    //     
    //     try {
    //         // 获取处理器
    //         AgentProcessor processor = agentProcessorFactory.getProcessor(agent);
    //         if (processor == null) {
    //             log.error("无法获取Agent处理器");
    //             return "[错误] 无法获取Agent处理器";
    //         }
    //        
    //         // 处理请求
    //         return processor.processRequest(agent, request, userId);
    //     } catch (Exception e) {
    //         log.error("处理普通Agent请求时发生错误", e);
    //         return "[错误] 处理请求时发生错误: " + e.getMessage();
    //     }
    // }

    /**
     * 处理流式对话请求的统一入口
     * 
     * @param agentId Agent ID
     * @param chatRequest 对话请求
     * @param response HTTP响应
     * @return SSE emitter
     */
    public SseEmitter handleChatStream(String agentId, ChatRequest chatRequest, HttpServletResponse response) {
        log.info("开始处理流式对话请求，AgentId: {}, 用户消息: {}", agentId, chatRequest.getMessage());

        // 尝试获取当前用户ID，优先从SecurityContext获取，其次从请求中解析JWT
        String userId = UserUtils.getCurrentUserId();
        
        // 如果在主线程中未能获取到用户ID，尝试在异步环境中获取
        if (userId == null) {
            userId = UserUtils.getCurrentUserIdInAsync();
        }
        
        if (userId == null) {
            log.error("用户未认证");
            SseEmitter emitter = workPanelSseService.createEmitter();
            // 检查响应是否已经提交
            if (!response.isCommitted()) {
                chatErrorHandler.handleChatError(emitter, "用户未认证，请重新登录");
            } else {
                log.warn("响应已提交，无法发送用户未认证错误信息");
                emitter.complete();
            }
            return emitter;
        }

        // 创建 SSE emitter
        SseEmitter emitter = workPanelSseService.createEmitter();
        
        // 将userId设为final以在Lambda表达式中使用
        final String finalUserId = userId;

        // 异步处理对话，避免阻塞HTTP连接
        executorService.execute(() -> {
            try {
                processChatRequest(emitter, agentId, chatRequest, finalUserId);
            } catch (Exception e) {
                log.error("处理聊天请求时发生异常", e);
                // 检查响应是否已经提交
                if (emitter != null) {
                    chatErrorHandler.handleChatError(emitter, "处理请求时发生错误", e, null);
                } else {
                    log.warn("响应已提交，无法发送处理请求错误信息");
                }
            }
        });

        return emitter;
    }

    /**
     * 处理聊天请求的核心逻辑
     *
     * @param emitter SSE发射器
     * @param agentId Agent ID
     * @param chatRequest 聊天请求
     * @param userId 用户ID
     */
    private void processChatRequest(SseEmitter emitter, String agentId, ChatRequest chatRequest, String userId) {
        try {
            // 获取Agent信息并进行权限检查
            Agent agent = agentValidationService.validateAgentAndPermission(agentId, userId, emitter);
            if (agent == null) {
                return; // 权限验证失败，直接返回
            }

            // 获取处理器并启动心跳保活机制
            AgentProcessor processor = agentProcessorFactory.getProcessor(agent);
            if (processor == null) {
                return; // 获取处理器失败，直接返回
            }
            
            // 启动心跳机制
            workPanelSseService.startHeartbeat(emitter, new java.util.concurrent.atomic.AtomicBoolean(false));

            // 转换请求对象
            AgentRequest request = chatRequest.toAgentRequest(agentId, agent, agentToolManager);

            // 处理流式请求
            streamRequestService.handleStreamRequest(emitter, processor, request, agent, userId);
        } catch (Exception e) {
            chatErrorHandler.handleChatError(emitter, "处理请求时发生错误", e, null);
        }
    }
}