package pangea.hiagent.workpanel;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;
import pangea.hiagent.workpanel.event.EventDeduplicationService;
import pangea.hiagent.workpanel.event.EventService;
import pangea.hiagent.sse.WorkPanelSseService;
import pangea.hiagent.web.dto.LogEvent;
import pangea.hiagent.web.dto.ResultEvent;
import pangea.hiagent.web.dto.ThoughtEvent;
import pangea.hiagent.web.dto.ToolEvent;
import pangea.hiagent.web.dto.WorkPanelEvent;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Consumer;
/**
 * 工作面板数据收集器实现
 * 专门负责采集Agent执行过程中的各类数据，遵循单一职责原则
 */
@Slf4j
@Component
public class WorkPanelDataCollector implements IWorkPanelDataCollector {
    
    /**
     * 事件列表（线程安全）
     */
    private final List<WorkPanelEvent> events = new CopyOnWriteArrayList<>();
    
    /**
     * 事件订阅者列表（线程安全）
     */
    private final List<Consumer<WorkPanelEvent>> subscribers = new CopyOnWriteArrayList<>();
    
    /**
     * 用户ID到订阅者的映射，用于更好地管理订阅者
     */
    private final Map<String, Consumer<WorkPanelEvent>> userSubscribers = new ConcurrentHashMap<>();
    
    /**
     * 事件去重服务
     */
    @Autowired
    private EventDeduplicationService eventDeduplicationService;
    
    /**
     * 统一事件服务
     */
    @Autowired
    private EventService eventService;
    
    /**
     * SSE服务
     */
    @Autowired
    private WorkPanelSseService unifiedSseService;
    
    /**
     * 最大事件数量，防止内存溢出
     */
    private static final int MAX_EVENTS = 1000;
    
    public WorkPanelDataCollector() {
        // 默认构造函数
    }    
    @Override
    public void recordThinking(String content, String thinkingType) {
        try {
            WorkPanelEvent event = createThinkingEvent(content, thinkingType);
            
            if (event != null && !eventDeduplicationService.isDuplicateEvent(event)) {
                addEvent(event);
            }
        } catch (Exception e) {
            logError("记录思考过程时发生错误", "content={}, type={}", content, thinkingType, e);
        }
    }
    
    @Override
    public void recordToolCallAction(String toolName, Object input, Object output, String status, Long executionTime) {
        try {
            WorkPanelEvent event = createToolCallActionEvent(toolName, input, output, status, executionTime);
            
            if (event != null) {
                handleToolCallAction(toolName, event, status, null);
            }
        } catch (Exception e) {
            logError("记录工具调用Action时发生错误", "toolName={}", toolName, null, e);
        }
    }
    

    

    

    
    @Override
    public void recordLog(String message, String level) {
        try {
            WorkPanelEvent event = createLogEvent(message, level);
            
            if (event != null) {
                addEvent(event);
            }
        } catch (Exception e) {
            logError("记录日志时发生错误", "message={}, level={}", message, level, e);
        }
    }
    
    @Override
    public void recordEmbed(String url, String type, String title, String htmlContent) {
        try {
            WorkPanelEvent event = createEmbedEvent(url, type, title, htmlContent);
            
            if (event != null) {
                addEvent(event);
            }
        } catch (Exception e) {
            logError("recordEmbed时发生错误", "title={}, type={}", title, type, e);
        }
    }
    
    @Override
    public void recordFinalAnswer(String content) {
        try {
            WorkPanelEvent event = createFinalAnswerEvent(content);
            
            if (event != null && !eventDeduplicationService.isDuplicateEvent(event)) {
                addEvent(event);
            }
        } catch (Exception e) {
            logError("记录最终答案时发生错误", "content={}", content, null, e);
        }
    }
    
    @Override
    public List<WorkPanelEvent> getEvents() {
        return new ArrayList<>(events);
    }
    
    @Override
    public void subscribe(Consumer<WorkPanelEvent> consumer) {
        if (consumer != null) {
            subscribers.add(consumer);
            if (log.isTraceEnabled()) {
                log.trace("已添加事件订阅者，当前订阅者数量: {}", subscribers.size());
            }
        }
    }
    
    /**
     * 为指定用户订阅事件
     * 
     * @param userId 用户ID
     * @param consumer 事件消费者
     */
    public void subscribe(String userId, Consumer<WorkPanelEvent> consumer) {
        if (!isValidSubscription(userId, consumer)) {
            return;
        }
        
        // 移除该用户之前的订阅者（如果存在）
        removeExistingSubscriber(userId);
        
        // 添加新的订阅者
        addNewSubscriber(userId, consumer);
        
        logSubscriptionDetails(userId);
    }
    
    /**
     * 取消指定用户的事件订阅
     * 
     * @param userId 用户ID
     */
    public void unsubscribe(String userId) {
        if (userId != null) {
            Consumer<WorkPanelEvent> subscriber = userSubscribers.remove(userId);
            if (subscriber != null) {
                subscribers.remove(subscriber);
                log.debug("已移除用户 {} 的事件订阅者", userId);
            }
        }
    }
    
    @Override
    public void clear() {
        try {
            events.clear();
            // 精简日志记录，避免过多的debug级别日志
            if (log.isTraceEnabled()) {
                log.trace("已清空工作面板事件");
            }
        } catch (Exception e) {
            log.error("清空事件时发生错误", e);
        }
    }
    
    @Override
    public WorkPanelEvent getLastToolCall() {
        // 从后往前查找最后一个工具调用事件
        for (int i = events.size() - 1; i >= 0; i--) {
            WorkPanelEvent event = events.get(i);
            if (event instanceof ToolEvent) {
                String type = event.getType();
                if ("tool_call".equals(type) || "tool_result".equals(type)) {
                    return event;
                }
            }
        }
        return null;
    }
    
    @Override
    public void addEvent(WorkPanelEvent event) {
        // 直接复用已有的私有方法
        if (event != null) {
            // 调用内部的addEvent方法
            addEventInternal(event);
        }
    }
    
    /**
     * 内部的添加事件方法，避免接口方法与私有方法的命名冲突
     */
    private void addEventInternal(WorkPanelEvent event) {
        try {
            // 控制事件数量，防止内存溢出
            if (events.size() >= MAX_EVENTS) {
                events.remove(0); // 移除最老的事件
            }
            
            events.add(event);
            
            String content = "";
            if (event instanceof ThoughtEvent) {
                content = ((ThoughtEvent) event).getContent();
            } else if (event instanceof LogEvent) {
                content = ((LogEvent) event).getContent();
            } else if (event instanceof ResultEvent) {
                content = ((ResultEvent) event).getContent();
            }
            log.debug("添加事件到列表: 类型={}, 内容={}", event.getType(), content);
            
            // 更新最近事件缓存
            eventDeduplicationService.updateRecentEventsCache(event);
            
            // 通知所有订阅者
            notifySubscribers(event);
        } catch (Exception e) {
            // 即使在addEvent方法中也增加异常保护，防止影响主流程
            logDebug("添加事件失败: {}", e.getMessage());
        }
    }
    
    // ==================== 私有方法 ====================
    
    /**
     * 创建思考事件
     */
    private WorkPanelEvent createThinkingEvent(String content, String thinkingType) {
        return eventService.recordThinking(content, thinkingType);
    }
    
    /**
     * 创建工具调用Action事件
     */
    private WorkPanelEvent createToolCallActionEvent(String toolName, Object input, Object output, String status, Long executionTime) {
        return eventService.recordToolCallComplete(toolName, output, status, executionTime);
    }
    
    /**
     * 处理工具调用Action
     */
    private void handleToolCallAction(String toolName, WorkPanelEvent event, String status, String errorMessage) {
        // 对于pending状态，直接添加事件
        if ("pending".equals(status)) {
            addEvent(event);
            return;
        }
        
        // 查找最近的该工具的pending事件并更新它
        WorkPanelEvent lastToolCall = getLastPendingToolCall(toolName);
        if (lastToolCall != null) {
            // 更新现有事件
            if ("error".equals(status) && lastToolCall instanceof ToolEvent) {
                updateEventAsError((ToolEvent) lastToolCall, errorMessage);
            } else if (lastToolCall instanceof ToolEvent) {
                ToolEvent toolEvent = (ToolEvent) lastToolCall;
                toolEvent.setType(WorkPanelUtils.getEventTypeFromStatus(status));
                toolEvent.setToolStatus(status);
                // 注意：这里不设置toolOutput，因为event中可能包含完整的输出信息
                toolEvent.setTimestamp(System.currentTimeMillis());
            }
            // 重新发布更新后的事件
            notifySubscribers(lastToolCall);
        } else {
            addEvent(event);
        }
    }
    
    /**
     * 更新事件为错误状态
     */
    private void updateEventAsError(WorkPanelEvent event, String errorMessage) {
        event.setType("tool_error");
        if (event instanceof ToolEvent) {
            ToolEvent toolEvent = (ToolEvent) event;
            // 使用反射设置content字段，因为ToolEvent没有setContent方法
            try {
                java.lang.reflect.Field contentField = WorkPanelEvent.class.getDeclaredField("content");
                contentField.setAccessible(true);
                contentField.set(toolEvent, errorMessage);
            } catch (Exception e) {
                // 如果反射失败，忽略错误
            }
        }
        if (event instanceof ToolEvent) {
            ((ToolEvent) event).setToolStatus("failure");
        }
        event.setTimestamp(System.currentTimeMillis());
    }
    
    /**
     * 创建日志事件
     */
    private WorkPanelEvent createLogEvent(String message, String level) {
        return eventService.recordLog(message, level);
    }
    
    /**
     * 创建嵌入事件
     */
    private WorkPanelEvent createEmbedEvent(String url, String type, String title, String htmlContent) {
        return eventService.recordEmbed(url, type, title, htmlContent);
    }
    
    /**
     * 创建最终答案事件
     */
    private WorkPanelEvent createFinalAnswerEvent(String content) {
        return eventService.recordFinalAnswer(content);
    }
    

    
    /**
     * 通知所有订阅者
     */
    private void notifySubscribers(WorkPanelEvent event) {
        if (event == null) {
            return;
        }
        
        try {
            // 遍历所有订阅者并发送事件
            for (Consumer<WorkPanelEvent> subscriber : subscribers) {
                try {
                    if (subscriber != null) {
                        subscriber.accept(event);
                    }
                } catch (Exception e) {
                    // 异常降级为debug日志，避免过度日志记录
                    // 异常通常由于SSE连接已断开导致，这是正常情况
                    if (e instanceof org.springframework.web.context.request.async.AsyncRequestNotUsableException) {
                        log.debug("订阅者处理事件失败：异步请求不可用（客户端已断开连接）");
                    } else if (e instanceof java.io.IOException) {
                        log.debug("订阅者处理事件失败：客户端连接已断开");
                    } else if (e.getMessage() != null && e.getMessage().contains("response has already been committed")) {
                        log.debug("订阅者处理事件失败：响应已提交");
                    } else {
                        // 其他异常也降级为debug，避免日志污染
                        log.debug("订阅者处理事件失败: {}", e.getMessage());
                    }
                }
            }
            
            // 通过EventService发送事件到所有SSE连接
            for (SseEmitter emitter : unifiedSseService.getEmitters()) {
                try {
                    eventService.sendWorkPanelEvent(emitter, event);
                } catch (Exception e) {
                    log.debug("通过EventService发送事件失败: {}", e.getMessage());
                }
            }
        } catch (Exception e) {
            log.debug("通知订阅者时发生错误: {}", e.getMessage());
        }
    }
    
    /**
     * 统一日志错误处理
     */
    private void logError(String message, String format, Object arg1, Object arg2, Exception e) {
        if (arg2 != null) {
            log.error(message + ": " + format, arg1, arg2, e);
        } else {
            log.error(message + ": " + format, arg1, e);
        }
    }
    
    /**
     * 统一日志调试处理
     */
    private void logDebug(String format, Object... arguments) {
        // 即使在addEvent方法中也增加异常保护，防止影响主流程
        log.debug(format, arguments);
    }
    
    /**
     * 验证订阅参数是否有效
     */
    private boolean isValidSubscription(String userId, Consumer<WorkPanelEvent> consumer) {
        return userId != null && consumer != null;
    }
    
    /**
     * 移除已存在的订阅者
     */
    private void removeExistingSubscriber(String userId) {
        Consumer<WorkPanelEvent> existingSubscriber = userSubscribers.get(userId);
        if (existingSubscriber != null) {
            subscribers.remove(existingSubscriber);
            log.debug("已移除用户 {} 的旧订阅者", userId);
        }
    }
    
    /**
     * 添加新的订阅者
     */
    private void addNewSubscriber(String userId, Consumer<WorkPanelEvent> consumer) {
        subscribers.add(consumer);
        userSubscribers.put(userId, consumer);
    }
    
    /**
     * 记录订阅详情
     */
    private void logSubscriptionDetails(String userId) {
        // 精简日志记录，避免过多的debug级别日志
        if (log.isTraceEnabled()) {
            log.trace("已为用户 {} 添加事件订阅者，当前订阅者数量: {}", userId, subscribers.size());
        }
    }
    
    /**
     * 检查是否为指定工具的待处理调用事件
     */
    private boolean isPendingToolCallEvent(WorkPanelEvent event, String toolName) {
        // 检查事件是否为ToolEvent类型
        if (!(event instanceof ToolEvent)) {
            return false;
        }
        
        ToolEvent toolEvent = (ToolEvent) event;
        return "tool_call".equals(event.getType()) && 
               toolName != null && toolName.equals(toolEvent.getToolName()) && 
               "pending".equals(toolEvent.getToolStatus());
    }
        
    /**
     * 查找最近的指定工具的pending事件
     * 
     * @param toolName 工具名称
     * @return 最近的pending事件，如果不存在则返回null
     */
    public WorkPanelEvent getLastPendingToolCall(String toolName) {
        // 从后往前查找最近的该工具的pending事件
        for (int i = events.size() - 1; i >= 0; i--) {
            WorkPanelEvent event = events.get(i);
            if (isPendingToolCallEvent(event, toolName)) {
                return event;
            }
        }
        return null;
    }
}