package pangea.hiagent.web.controller;

import lombok.extern.slf4j.Slf4j;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import pangea.hiagent.web.dto.ApiResponse;
import pangea.hiagent.web.dto.PageData;
import pangea.hiagent.web.service.AgentService;
import pangea.hiagent.web.service.AgentToolRelationService;
import pangea.hiagent.web.dto.AgentWithToolsDTO;

import com.baomidou.mybatisplus.core.metadata.IPage;
import pangea.hiagent.common.utils.UserUtils;
import pangea.hiagent.model.Agent;
import pangea.hiagent.model.Tool;
import pangea.hiagent.common.exception.BusinessException;

import java.util.ArrayList;
import java.util.List;

/**
 * Agent API控制器
 * 提供Agent的増删改查功能
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/agent")
public class AgentController {
    
    // 服务依赖
    private final AgentService agentService;
    private final AgentToolRelationService agentToolRelationService;
    
    public AgentController(AgentService agentService, AgentToolRelationService agentToolRelationService) {
        this.agentService = agentService;
        this.agentToolRelationService = agentToolRelationService;
    }
    
    // ==================== Agent管理API ====================
    
    /**
     * 创建Agent（基本API，不含工具）
     */
    @PostMapping
    public ApiResponse<Agent> createAgent(@RequestBody Agent agent) {
        try {
            String userId = UserUtils.getCurrentUserId();
            if (userId == null) {
                return ApiResponse.error(4001, "用户未认证");
            }
            
            log.info("用户 {} 开始创建Agent: {}", userId, agent.getName());
            agent.setOwner(userId);
            Agent created = agentService.createAgentWithTools(agent);
            log.info("用户 {} 成功创建Agent: {} (ID: {})", userId, created.getName(), created.getId());
            return ApiResponse.success(created, "创建Agent成功");
        } catch (Exception e) {
            log.error("创建Agent失败", e);
            return ApiResponse.error(4001, "创建Agent失败: " + e.getMessage());
        }
    }
    
    /**
     * 创建Agent（包含工具信息，优先使用此方法）
     * 一次性保存Agent与其关联的工具
     */
    @PostMapping("/with-tools")
    public ApiResponse<Agent> createAgentWithTools(@RequestBody AgentWithToolsDTO agentWithToolsDTO) {
        try {
            String userId = UserUtils.getCurrentUserId();
            if (userId == null) {
                return ApiResponse.error(4001, "用户未认证");
            }
            
            Agent agent = agentWithToolsDTO.getAgent();
            if (agent == null) {
                return ApiResponse.error(1001, "Agent信息不能为空");
            }
            
            log.info("用户 {} 开始创建Agent（包含工具）: {}", userId, agent.getName());
            agent.setOwner(userId);
            
            // 设置工具信息到Agent对象
            if (agentWithToolsDTO.getToolIds() != null && !agentWithToolsDTO.getToolIds().isEmpty()) {
                // 是一个虚拟的tools列表，包含ID，戗平层正会被使用
                List<Tool> tools = agentWithToolsDTO.getToolIds().stream()
                    .map(toolId -> {
                        Tool tool = new Tool();
                        tool.setId(toolId);
                        return tool;
                    })
                    .toList();
                agent.setTools(tools);
            }
            
            Agent created = agentService.createAgentWithTools(agent);
            log.info("用户 {} 成功创建Agent（包含工具）: {} (ID: {})", userId, created.getName(), created.getId());
            return ApiResponse.success(created, "创建Agent成功");
        } catch (Exception e) {
            log.error("创建Agent（包含工具）失败", e);
            return ApiResponse.error(4001, "创建Agent失败: " + e.getMessage());
        }
    }
    
    /**
     * 更新Agent（优先使用此方法）
     * 一次性更新Agent与其关联的工具
     */
    @PreAuthorize("@permissionEvaluator.hasPermission(authentication, #id, 'Agent', 'write')")
    @PutMapping("/{id}")
    public ApiResponse<Agent> updateAgent(@PathVariable(name = "id") String id, @RequestBody Agent agent) {
        String userId = UserUtils.getCurrentUserId();
        if (userId == null) {
            log.warn("用户未认证，无法更新Agent: {}", id);
            return ApiResponse.error(4001, "用户未认证");
        }
        
        try {
            log.info("用户 {} 开始更新Agent: {}", userId, id);
            // 设置Agent ID，忽略前端可能传递的ID
            agent.setId(id);
            
            // 验证必要字段
            if (agent.getName() == null || agent.getName().trim().isEmpty()) {
                log.warn("Agent名称为空，用户: {}, Agent ID: {}", userId, id);
                return ApiResponse.error(1001, "Agent名称不能为空");
            }
            
            if (agent.getDefaultModel() == null || agent.getDefaultModel().trim().isEmpty()) {
                log.warn("默认模型为空，用户: {}, Agent ID: {}", userId, id);
                return ApiResponse.error(1001, "默认模型不能为空");
            }
            
            if (agent.getStatus() == null || agent.getStatus().trim().isEmpty()) {
                log.warn("状态为空，用户: {}, Agent ID: {}", userId, id);
                return ApiResponse.error(1001, "状态不能为空");
            }
            
            Agent updated = agentService.updateAgentWithTools(agent);
            log.info("用户 {} 成功更新Agent: {}", userId, updated.getId());
            return ApiResponse.success(updated, "更新Agent成功");
        } catch (IllegalArgumentException e) {
            log.warn("更新Agent参数验证失败，用户: {}, Agent ID: {}, 错误: {}", userId, id, e.getMessage());
            return ApiResponse.error(1001, "参数验证失败: " + e.getMessage());
        } catch (BusinessException e) {
            log.warn("更新Agent业务异常，用户: {}, Agent ID: {}, 错误代码: {}, 错误信息: {}", userId, id, e.getCode(), e.getMessage());
            return ApiResponse.error(e.getCode(), e.getMessage());
        } catch (Exception e) {
            log.error("更新Agent失败，用户: {}, Agent ID: {}", userId, id, e);
            String errorMessage = "更新Agent失败";
            if (e.getMessage() != null) {
                errorMessage += ": " + e.getMessage();
            } else {
                errorMessage += "，请检查请求数据格式是否正确";
            }
            return ApiResponse.error(4001, errorMessage);
        }
    }
    
    /**
     * 更新Agent（包含工具信息，作为懂口方法）
     */
    @PreAuthorize("@permissionEvaluator.hasPermission(authentication, #id, 'Agent', 'write')")
    @PutMapping("/{id}/with-tools")
    public ApiResponse<Agent> updateAgentWithTools(@PathVariable(name = "id") String id, @RequestBody AgentWithToolsDTO agentWithToolsDTO) {
        String userId = UserUtils.getCurrentUserId();
        if (userId == null) {
            log.warn("用户未认证，无法更新Agent: {}", id);
            return ApiResponse.error(4001, "用户未认证");
        }
        
        try {
            log.info("用户 {} 开始更新Agent（包含工具信息）: {}", userId, id);
            
            Agent agent = agentWithToolsDTO.getAgent();
            if (agent == null) {
                return ApiResponse.error(1001, "Agent信息不能为空");
            }
            
            // 设置Agent ID，忽略前端可能传递的ID
            agent.setId(id);
            
            // 验证必要字段
            if (agent.getName() == null || agent.getName().trim().isEmpty()) {
                log.warn("Agent名称为空，用户: {}, Agent ID: {}", userId, id);
                return ApiResponse.error(1001, "Agent名称不能为空");
            }
            
            if (agent.getDefaultModel() == null || agent.getDefaultModel().trim().isEmpty()) {
                log.warn("默认模型为空，用户: {}, Agent ID: {}", userId, id);
                return ApiResponse.error(1001, "默认模型不能为空");
            }
            
            if (agent.getStatus() == null || agent.getStatus().trim().isEmpty()) {
                log.warn("状态为空，用户: {}, Agent ID: {}", userId, id);
                return ApiResponse.error(1001, "状态不能为空");
            }
            
            // 设置工具信息到Agent对象
            if (agentWithToolsDTO.getToolIds() != null && !agentWithToolsDTO.getToolIds().isEmpty()) {
                List<Tool> tools = agentWithToolsDTO.getToolIds().stream()
                    .map(toolId -> {
                        Tool tool = new Tool();
                        tool.setId(toolId);
                        return tool;
                    })
                    .toList();
                agent.setTools(tools);
            } else {
                agent.setTools(new ArrayList<>());
            }
            
            Agent updated = agentService.updateAgentWithTools(agent);
            log.info("用户 {} 成功更新Agent（包含工具信息）: {}", userId, updated.getId());
            return ApiResponse.success(updated, "更新Agent成功");
        } catch (IllegalArgumentException e) {
            log.warn("更新Agent（包含工具信息）参数验证失败，用户: {}, Agent ID: {}, 错误: {}", userId, id, e.getMessage());
            return ApiResponse.error(1001, "参数验证失败: " + e.getMessage());
        } catch (BusinessException e) {
            log.warn("更新Agent（包含工具信息）业务异常，用户: {}, Agent ID: {}, 错误代码: {}, 错误信息: {}", userId, id, e.getCode(), e.getMessage());
            return ApiResponse.error(e.getCode(), e.getMessage());
        } catch (Exception e) {
            log.error("更新Agent（包含工具信息）失败，用户: {}, Agent ID: {}", userId, id, e);
            String errorMessage = "更新Agent失败";
            if (e.getMessage() != null) {
                errorMessage += ": " + e.getMessage();
            } else {
                errorMessage += "，请检查请求数据格式是否正确";
            }
            return ApiResponse.error(4001, errorMessage);
        }
    }
    
    /**
     * 删除Agent
     */
    @PreAuthorize("@permissionEvaluator.hasPermission(authentication, #id, 'Agent', 'delete')")
    @DeleteMapping("/{id}")
    public ApiResponse<Void> deleteAgent(@PathVariable(name = "id") String id) {
        try {
            String userId = UserUtils.getCurrentUserId();
            log.info("用户 {} 开始删除Agent: {}", userId, id);
            agentService.deleteAgent(id);
            log.info("用户 {} 成功删除Agent: {}", userId, id);
            return ApiResponse.success(null, "删除Agent成功");
        } catch (Exception e) {
            log.error("删除Agent失败", e);
            return ApiResponse.error(4001, "删除Agent失败: " + e.getMessage());
        }
    }
    
    /**
     * 获取Agent详情
     */
    @PreAuthorize("@permissionEvaluator.hasPermission(authentication, #id, 'Agent', 'read')")
    @GetMapping("/{id}")
    public ApiResponse<Agent> getAgent(@PathVariable(name = "id") String id) {
        try {
            Agent agent = agentService.getAgent(id);
            if (agent == null) {
                return ApiResponse.error(4001, "Agent不存在");
            }
            return ApiResponse.success(agent);
        } catch (Exception e) {
            log.error("获取Agent详情失败", e);
            return ApiResponse.error(4001, "获取Agent详情失败: " + e.getMessage());
        }
    }
    
    /**
     * 分页获取Agent列表
     */
    @GetMapping("/list")
    public ApiResponse<PageData<Agent>> listAgents(
            @RequestParam(defaultValue = "1") Long current,
            @RequestParam(defaultValue = "10") Long size,
            @RequestParam(required = false) String name,
            @RequestParam(required = false) String status) {
        try {
            IPage<Agent> page = agentService.pageAgents(current, size, name, status);
            return ApiResponse.success(PageData.from(page));
        } catch (Exception e) {
            log.error("获取Agent列表失败", e);
            return ApiResponse.error(4001, "获取Agent列表失败: " + e.getMessage());
        }
    }
    
    /**
     * 获取用户的Agent列表
     */
    @GetMapping
    public ApiResponse<java.util.List<Agent>> getUserAgents() {
        try {
            String userId = UserUtils.getCurrentUserId();
            if (userId == null) {
                return ApiResponse.error(4001, "用户未认证");
            }
            java.util.List<Agent> agents = agentService.getUserAgents(userId);
            return ApiResponse.success(agents);
        } catch (Exception e) {
            log.error("获取用户Agent列表失败", e);
            return ApiResponse.error(4001, "获取用户Agent列表失败: " + e.getMessage());
        }
    }
    
    /**
     * 为Agent设置工具关联关系
     */
    @PreAuthorize("@permissionEvaluator.hasPermission(authentication, #agentId, 'Agent', 'write')")
    @PostMapping("/{agentId}/tools")
    public ApiResponse<Void> setAgentTools(@PathVariable(name = "agentId") String agentId, @RequestBody List<String> toolIds) {
        try {
            log.info("为Agent {} 设置工具关联关系", agentId);
            agentToolRelationService.setAgentTools(agentId, toolIds);
            return ApiResponse.success(null, "设置工具关联关系成功");
        } catch (Exception e) {
            log.error("设置Agent工具关联关系失败", e);
            return ApiResponse.error(4001, "设置工具关联关系失败: " + e.getMessage());
        }
    }
    
    /**
     * 获取Agent关联的工具ID列表
     */
    @PreAuthorize("@permissionEvaluator.hasPermission(authentication, #agentId, 'Agent', 'read')")
    @GetMapping("/{agentId}/tools")
    public ApiResponse<List<String>> getAgentTools(@PathVariable(name = "agentId") String agentId) {
        try {
            log.info("获取Agent {} 关联的工具ID列表", agentId);
            List<String> toolIds = agentToolRelationService.getToolIdsByAgentId(agentId);
            return ApiResponse.success(toolIds);
        } catch (Exception e) {
            log.error("获取Agent工具关联关系失败", e);
            return ApiResponse.error(4001, "获取工具关联关系失败: " + e.getMessage());
        }
    }
    
    /**
     * 获取带有关联工具信息的Agent详情
     */
    @PreAuthorize("@permissionEvaluator.hasPermission(authentication, #id, 'Agent', 'read')")
    @GetMapping("/{id}/detail")
    public ApiResponse<Agent> getAgentDetail(@PathVariable(name = "id") String id) {
        try {
            Agent agent = agentService.getAgentWithTools(id);
            if (agent == null) {
                return ApiResponse.error(4001, "Agent不存在");
            }
            return ApiResponse.success(agent);
        } catch (Exception e) {
            log.error("获取Agent详情失败", e);
            return ApiResponse.error(4001, "获取Agent详情失败: " + e.getMessage());
        }
    }
}