package pangea.hiagent.sse;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicLong;

/**
 * 用户会话管理器
 * 专门负责管理用户的SSE会话连接
 */
@Slf4j
@Component
public class UserSseManager {
    
    // 存储用户ID到SSE Emitter的映射关系
    private final ConcurrentMap<String, SseEmitter> userEmitters = new ConcurrentHashMap<>();
    
    // 存储SSE Emitter到用户ID的反向映射关系（用于快速查找）
    private final ConcurrentMap<SseEmitter, String> emitterUsers = new ConcurrentHashMap<>();
    
    // 存储连接创建时间，用于超时检查
    private final ConcurrentMap<SseEmitter, AtomicLong> connectionTimes = new ConcurrentHashMap<>();
    
    // 连接超时时间（毫秒），默认30分钟
    private static final long CONNECTION_TIMEOUT = 30 * 60 * 1000L;
    
    /**
     * 注册用户的SSE连接
     * 如果该用户已有连接，则先关闭旧连接再注册新连接
     * 
     * @param userId 用户ID
     * @param emitter SSE Emitter
     * @return true表示注册成功，false表示注册失败
     */
    public boolean registerSession(String userId, SseEmitter emitter) {
        if (userId == null || userId.isEmpty() || emitter == null) {
            log.warn("注册SSE会话失败：用户ID或Emitter为空");
            return false;
        }
        
        try {
            // 检查该用户是否已有连接
            SseEmitter existingEmitter = userEmitters.get(userId);
            if (existingEmitter != null) {
                // 如果已有连接，先关闭旧连接
                log.info("用户 {} 已有SSE连接，关闭旧连接", userId);
                closeEmitter(existingEmitter);
                // 从映射中移除旧连接
                userEmitters.remove(userId, existingEmitter);
                emitterUsers.remove(existingEmitter);
            }
            
            // 注册新连接
            userEmitters.put(userId, emitter);
            emitterUsers.put(emitter, userId);
            // 记录连接创建时间
            connectionTimes.put(emitter, new AtomicLong(System.currentTimeMillis()));
            
            log.info("成功为用户 {} 注册SSE会话", userId);
            return true;
        } catch (Exception e) {
            log.error("注册SSE会话时发生异常：用户ID={}", userId, e);
            return false;
        }
    }
    
    /**
     * 移除用户的SSE会话
     * 
     * @param emitter SSE Emitter
     */
    public void removeSession(SseEmitter emitter) {
        if (emitter == null) {
            return;
        }
        
        try {
            String userId = emitterUsers.get(emitter);
            if (userId != null) {
                userEmitters.remove(userId, emitter);
                emitterUsers.remove(emitter);
                connectionTimes.remove(emitter);
                log.debug("已移除用户 {} 的SSE会话", userId);
            }
        } catch (Exception e) {
            log.warn("移除SSE会话时发生异常", e);
        }
    }
    
    /**
     * 获取用户的当前SSE连接
     * 
     * @param userId 用户ID
     * @return SSE Emitter，如果不存在则返回null
     */
    public SseEmitter getSession(String userId) {
        if (userId == null || userId.isEmpty()) {
            return null;
        }
        return userEmitters.get(userId);
    }
    
    /**
     * 检查用户是否有活跃的SSE连接
     * 
     * @param userId 用户ID
     * @return true表示有活跃连接，false表示没有
     */
    public boolean hasActiveSession(String userId) {
        if (userId == null || userId.isEmpty()) {
            return false;
        }
        SseEmitter emitter = userEmitters.get(userId);
        return emitter != null && isEmitterValid(emitter) && !isSessionExpired(emitter);
    }
    
    /**
     * 检查SSE Emitter是否仍然有效
     * 
     * @param emitter 要检查的emitter
     * @return 如果有效返回true，否则返回false
     */
    public boolean isEmitterValid(SseEmitter emitter) {
        if (emitter == null) {
            return false;
        }
        
        // 检查emitter是否已完成
        try {
            // 尝试发送一个空事件来检查连接状态
            emitter.send(SseEmitter.event().name("ping").data("").build());
            return true;
        } catch (Exception e) {
            // 如果出现任何异常，认为连接已失效
            return false;
        }
    }
    
    /**
     * 检查会话是否已过期
     * 
     * @param emitter 要检查的emitter
     * @return 如果过期返回true，否则返回false
     */
    public boolean isSessionExpired(SseEmitter emitter) {
        if (emitter == null) {
            return true;
        }
        
        AtomicLong connectionTime = connectionTimes.get(emitter);
        if (connectionTime == null) {
            return false; // 如果没有记录时间，假设未过期
        }
        
        long currentTime = System.currentTimeMillis();
        long connectionStartTime = connectionTime.get();
        
        return (currentTime - connectionStartTime) > CONNECTION_TIMEOUT;
    }
    
    /**
     * 关闭指定的SSE Emitter
     * 
     * @param emitter SSE Emitter
     */
    public void closeEmitter(SseEmitter emitter) {
        if (emitter == null) {
            return;
        }
        
        try {
            emitter.complete();
            log.debug("已关闭SSE Emitter");
        } catch (Exception e) {
            log.warn("关闭SSE Emitter时发生异常", e);
        }
    }
    
    /**
     * 获取当前会话的用户数量
     * 
     * @return 用户数量
     */
    public int getSessionCount() {
        return userEmitters.size();
    }
    
    /**
     * 清理所有会话（用于系统关闭时）
     */
    public void clearAllSessions() {
        try {
            for (SseEmitter emitter : emitterUsers.keySet()) {
                try {
                    emitter.complete();
                } catch (Exception e) {
                    log.warn("关闭SSE Emitter时发生异常", e);
                }
            }
            userEmitters.clear();
            emitterUsers.clear();
            connectionTimes.clear();
            log.info("已清理所有SSE会话");
        } catch (Exception e) {
            log.error("清理所有SSE会话时发生异常", e);
        }
    }
    
    /**
     * 处理连接完成事件
     * 职责：协调完成连接的清理工作
     * 
     * @param emitter SSE Emitter
     */
    public void handleConnectionCompletion(SseEmitter emitter) {
        log.debug("处理SSE连接完成事件");
        // 移除连接
        removeSession(emitter);
    }
    
    /**
     * 处理连接超时事件
     * 职责：协调超时连接的清理工作
     * 
     * @param emitter SSE Emitter
     */
    public void handleConnectionTimeout(SseEmitter emitter) {
        log.debug("处理SSE连接超时事件");
        // 移除连接
        removeSession(emitter);
    }
    
    /**
     * 处理连接错误事件
     * 职责：协调错误连接的清理工作
     * 
     * @param emitter SSE Emitter
     */
    public void handleConnectionError(SseEmitter emitter) {
        log.debug("处理SSE连接错误事件");
        // 移除连接
        removeSession(emitter);
    }
}