package pangea.hiagent.agent.service;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import pangea.hiagent.model.Agent;
import pangea.hiagent.web.dto.AgentRequest;

/**
 * SSE Token发射器
 * 专注于将token转换为SSE事件并发送
 */
@Slf4j
@Component
public class SseTokenEmitter implements TokenConsumerWithCompletion {
    
    private final UserSseService userSseService;
    
    // 当前处理的emitter
    private SseEmitter emitter;
    
    // 上下文信息
    private Agent agent;
    private AgentRequest request;
    private String userId;
    private String emitterId;
    
    // 完成回调
    private CompletionCallback completionCallback;
    
    public SseTokenEmitter(UserSseService userSseService) {
        this.userSseService = userSseService;
    }
    
    /**
     * 设置当前使用的SSE发射器
     */
    public void setEmitter(SseEmitter emitter) {
        this.emitter = emitter;
    }
    
    /**
     * 设置上下文信息
     */
    public void setContext(Agent agent, AgentRequest request, String userId) {
        this.agent = agent;
        this.request = request;
        this.userId = userId;
    }

    
    /**
     * 设置完成回调
     */
    public void setCompletionCallback(CompletionCallback completionCallback) {
        this.completionCallback = completionCallback;
    }
    public void setEmitterId(String emitterId) {
        this.emitterId  = emitterId;
    }
    public String getEmitterId() {
        return emitterId;
    }
    @Override
    public void accept(String token) {
        // 使用JSON格式发送token，确保转义序列被正确处理
        try {
            if (emitter != null && userSseService.isEmitterValidSafe(emitter)) {
                // 检查是否是错误消息（以[错误]或[ERROR]开头）
                if (token != null && (token.startsWith("[错误]") || token.startsWith("[ERROR]"))) {
                    // 发送标准错误事件而不是纯文本
                    userSseService.sendErrorEvent(emitter, token);
                } else {
                    // 使用SSE标准事件格式发送token，以JSON格式确保转义序列正确处理
                    userSseService.sendTokenEvent(emitter, token);
                }
            } else {
                log.debug("SSE emitter已无效，跳过发送token");
            }
        } catch (Exception e) {
            log.error("发送token失败", e);
        }
    }
    
    @Override
    public void onComplete(String fullContent) {
        try {
            if (emitter != null && !userSseService.isEmitterCompleted(emitter)) {
                // 发送完成事件
                emitter.send(SseEmitter.event().name("done").data("[DONE]").build());
                log.debug("完成信号已发送");
            }
            
            // 调用完成回调
            if (completionCallback != null) {
                completionCallback.onComplete(emitter, agent, request, userId, fullContent);
            }
        } catch (Exception e) {
            log.error("处理完成事件失败", e);
        } finally {
            // 关闭连接
            closeEmitter();
        }
    }
    
    /**
     * 安全关闭SSE连接
     */
    public void closeEmitter() {
        try {
            if (emitter != null && !userSseService.isEmitterCompleted(emitter)) {
                emitter.complete();
                log.debug("SSE连接已关闭");
            }
        } catch (Exception ex) {
            log.error("完成emitter时发生错误", ex);
        }
    }
    
    /**
     * 完成回调接口
     */
    @FunctionalInterface
    public interface CompletionCallback {
        void onComplete(SseEmitter emitter, Agent agent, AgentRequest request, String userId, String fullContent);
    }

    public String getUserId() {
        return userId;
    }
}